package com.elitescloud.cloudt.ucenter.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.ucenter.api.dto.ArticleInfoDTO;
import com.elitescloud.cloudt.ucenter.api.vo.param.*;
import com.elitescloud.cloudt.ucenter.api.vo.resp.*;
import com.elitescloud.cloudt.ucenter.api.vo.save.ArticleCategorySaveVO;
import com.elitescloud.cloudt.ucenter.api.vo.save.ArticleInfoSaveVO;
import com.elitescloud.cloudt.ucenter.common.constant.ConstantsUcenter;
import com.elitescloud.cloudt.ucenter.common.constant.UdcEnum;
import com.elitescloud.cloudt.ucenter.convert.ArticleManageConvert;
import com.elitescloud.cloudt.ucenter.dto.ArticleCategoryDTO;
import com.elitescloud.cloudt.ucenter.entity.ArticleCategoryDO;
import com.elitescloud.cloudt.ucenter.entity.ArticleInfoDO;
import com.elitescloud.cloudt.ucenter.repo.ArticleCategoryRepo;
import com.elitescloud.cloudt.ucenter.repo.ArticleCategoryRepoProc;
import com.elitescloud.cloudt.ucenter.repo.ArticleInfoRepo;
import com.elitescloud.cloudt.ucenter.repo.ArticleInfoRepoProc;
import com.elitescloud.cloudt.ucenter.service.ArticleManageService;
import com.elitescloud.cloudt.ucenter.utils.SysGenerator;
import com.elitescloud.cloudt.ucenter.utils.TreeUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

@Service
@RequiredArgsConstructor
@Slf4j
public class ArticleManageServiceImpl implements ArticleManageService {

    private final ArticleCategoryRepo articleCategoryRepo;

    private final ArticleInfoRepo articleInfoRepo;

    private final ArticleCategoryRepoProc articleCategoryRepoProc;

    private final ArticleInfoRepoProc articleInfoRepoProc;

    private final SysGenerator sysGenerator;

    private Long loopMatch(ArticleCategoryDTO articleCategoryDTO, List<ArticleCategoryDTO> treeDtoList) {

        Long id;
        // 根节点的id
        id = articleCategoryDTO.getId();
        //
        for (ArticleCategoryDTO item : treeDtoList) {
            // 根节点的父id 是否是当前节点
            if (articleCategoryDTO.getPid().equals(item.getId())) {
                id = loopMatch(item, treeDtoList);
            }
        }
        return id;
    }

    @Override
    @SysCodeProc
    public PagingVO<ArticleCategoryRespVO> search(ArticleCategoryPagingParam searchParam) {

        // 查询出所有的分类进行树化
        List<ArticleCategoryDTO> categoryDTOS = articleCategoryRepoProc.findAll();
        List<ArticleCategoryDTO> treeDtoList = TreeUtils.buildTree(null, false, categoryDTOS);


        for (ArticleCategoryDTO child : categoryDTOS) {
            if (StrUtil.equals(child.getShowFlag(), UdcEnum.AD_SHOW_FLAG_HIDE.getValueCode())) {
                child.setSortNo(null);
            }
            child.setShowFlagName(sysGenerator.getUdcName(UdcEnum.AD_SHOW_FLAG_DISPLAY.getCode(), child.getShowFlag()));
        }

        // 查询出满足条件的分类
        List<ArticleCategoryDTO> categoryDTOList = articleCategoryRepoProc.findByCategoryNameAndShowFlag(searchParam);
        // 查找一级分类id
        Set<Long> firstIdList = categoryDTOList.stream().filter(item -> ObjectUtil.isNull(item.getRootId())).map(item -> item.getId()).collect(Collectors.toSet());
        Set<Long> firstIdList1 = categoryDTOList.stream().filter(item -> ObjectUtil.isNotNull(item.getRootId())).map(item -> item.getRootId()).collect(Collectors.toSet());
        firstIdList.addAll(firstIdList1);
        // 满足条件的分页查询
        PagingVO<ArticleCategoryDTO> parentDTOList = articleCategoryRepoProc.findParentCategoryPagingResults(firstIdList, searchParam);
        Set<Long> collect = parentDTOList.stream().map(item -> item.getId()).collect(Collectors.toSet());
        // 筛选出符合条件的树节点
        List<ArticleCategoryDTO> treeList = new ArrayList<>();
        for (ArticleCategoryDTO articleCategoryDTO : treeDtoList) {
            for (Long firstId : collect) {
                if (articleCategoryDTO.getId().equals(firstId)) {
                    treeList.add(articleCategoryDTO);
                }
            }
        }
//        treeList = treeList.stream().sorted(Comparator.comparing(ArticleCategoryDTO::getShowFlag)).collect(Collectors.toList());
        List<ArticleCategoryRespVO> respVOList = ArticleManageConvert.INSTANCE.categoryDtosToVos(treeList);
        return PagingVO.<ArticleCategoryRespVO>builder()
                .total(parentDTOList.getTotal())
                .records(respVOList).build();
    }

    private void removeItem(List<ArticleCategoryDTO> list, ArticleCategoryDTO articleCategoryDTO) {
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdateArticleCategory(ArticleCategorySaveVO saveVO) {

        ArticleCategoryDO categoryDO = ArticleManageConvert.INSTANCE.categorySaveVOToDo(saveVO);
        // 新增
        if (ObjectUtil.isNull(saveVO.getId())) {
            // 新增分类 设置一级分类id
            if (CollectionUtil.isNotEmpty(saveVO.getPidList())) {
                saveVO.setRootId(saveVO.getPidList().get(0));
            }
            // 下级分类为显示 需将对应一级分类设置为显示
            if (ObjectUtil.isNotNull(saveVO.getPid()) && StrUtil.equals(saveVO.getShowFlag(), UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode())) {
                articleCategoryRepo.updateShowById(saveVO.getPid(), saveVO.getShowFlag());
                if (CollectionUtil.isNotEmpty(saveVO.getPidList())) {
                    articleCategoryRepo.updateShowById(saveVO.getPidList().get(0), saveVO.getShowFlag());
                }
            }
            // 设置一级分类id
            if (ObjectUtil.isNotNull(saveVO.getPid())) {
                Set<Long> set = new HashSet<>();
                findParentId(saveVO.getPid(), set);
                if (CollectionUtil.isNotEmpty(set)) {
                    categoryDO.setRootId(set.iterator().next());
                } else {
                    categoryDO.setRootId(saveVO.getPid());
                }
            }

        } else {
            // 编辑
            Optional<ArticleCategoryDO> optional = articleCategoryRepo.findById(saveVO.getId());
            if (optional.isEmpty()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未找到文章分类信息");
            }
            if (StrUtil.equals(optional.get().getShowFlag(), UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能编辑隐藏的分类");
            }
            // 编辑时一级分类有二级分类时 不允许选择一级分类
            if (ObjectUtil.isNull(optional.get().getPid())) {
                long count = articleCategoryRepo.countByPid(optional.get().getId());
                if (ObjectUtil.isNotNull(saveVO.getPid()) && count > 0) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "一级文章分类下还有二级分类，不允许选择父类");
                }
            }

        }
        ArticleCategoryDO save = articleCategoryRepo.save(categoryDO);
        return save.getId();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long deleteArticleCategory(Long id) {
        Optional<ArticleCategoryDO> byId = articleCategoryRepo.findById(id);
        if (byId.isPresent()) {
            if (StrUtil.equals(byId.get().getShowFlag(), UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能删除隐藏的文章分类");
            }
            articleCategoryRepo.updateDeleteFlagById(id);
            articleInfoRepo.updateDeleteFlagByCategoryId(id);
            // 找到当前分类下的子类
            Set<Long> deleteIdSet = new HashSet<>();
            findChildId(Set.of(id), deleteIdSet);
            if (CollectionUtil.isNotEmpty(deleteIdSet)) {
                articleCategoryRepo.updateDeleteFlagByIdIn(deleteIdSet);
                articleInfoRepo.updateDeleteFlagByCategoryIdIn(deleteIdSet);
            }
            return id;
        } else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到文章分类信息");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long deleteArticleInfo(Long id) {
        Optional<ArticleInfoDO> byId = articleInfoRepo.findById(id);
        if (byId.isPresent()) {
            if (StrUtil.equals(byId.get().getShowFlag(), UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能删除隐藏的文章信息");
            }
            articleInfoRepo.updateDeleteFlagById(id);
            return id;
        } else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到文章信息");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdateArticleInfo(ArticleInfoSaveVO saveVO) {

        // 新增
        if (ObjectUtil.isNull(saveVO.getId())) {
            Optional<ArticleCategoryDO> byId = articleCategoryRepo.findById(saveVO.getCategoryId());
            if (byId.isPresent()) {
                saveVO.setCategoryName(byId.get().getCategoryName());
            }
            String articleCode = sysGenerator.generateCode(ConstantsUcenter.ARTICLE_NO);
            if (articleInfoRepo.existsByArticleCode(articleCode)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION,
                        "发号器生成的文章编号：" + articleCode + " 已存在，请检查发号器配置是否正确");
            }
            // 使用发号器生成文章编号
            saveVO.setArticleCode(articleCode);
        } else {
            // 编辑
            Optional<ArticleInfoDO> optional = articleInfoRepo.findById(saveVO.getId());

            if (optional.isEmpty()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未找到文章信息");
            }
            ArticleInfoDO articleInfoDO = optional.get();
            if (StrUtil.equals(articleInfoDO.getShowFlag(), UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能编辑隐藏的文章信息");
            }
            saveVO.setArticleCode(articleInfoDO.getArticleCode());
            Optional<ArticleCategoryDO> byId = articleCategoryRepo.findById(saveVO.getCategoryId());
            if (byId.isEmpty()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "没有查询到该文章分类信息");
            }
            saveVO.setCategoryName(byId.get().getCategoryName());
        }
        ArticleInfoDO articleInfoDO = ArticleManageConvert.INSTANCE.copyArticleInfoSaveToDo(saveVO);
        ArticleInfoDO saveDo = articleInfoRepo.save(articleInfoDO);
        return saveDo.getId();
    }

//    @Override
//    public List<ArticleCategoryRespVO> queryUpCategory() {
//
//        List<ArticleCategoryDTO> categoryDTOS = articleCategoryRepoProc.findAll();
//        List<ArticleCategoryDTO> treeDtoList = TreeUtils.buildTree(null, false, categoryDTOS);
//        List<ArticleCategoryRespVO> voList = ArticleManageConvert.INSTANCE.categoryDtosToVos(articleCategoryRepoProc.queryUpCategory());
//        List<ArticleCategoryRespVO> respVOS = ArticleManageConvert.INSTANCE.categoryDtosToVos(treeDtoList);
//        return respVOS;
//    }

    @Override
    public List<UpCategoryRespVO> queryUpCategory(Long id) {
        Optional<ArticleCategoryDO> byId = articleCategoryRepo.findById(id);
        List<UpCategoryRespVO> respVOS = new ArrayList<>();
        if (byId.isPresent()) {
            if (ObjectUtil.isNotNull(byId.get().getPid())) {
                Optional<ArticleCategoryDO> parentDo = articleCategoryRepo.findById(byId.get().getPid());
                if (parentDo.isPresent()) {
                    UpCategoryRespVO resp = new UpCategoryRespVO();
                    resp.setId(parentDo.get().getId());
                    resp.setCategoryName(parentDo.get().getCategoryName());
                    respVOS.add(resp);
                }
            }
        }
        return respVOS;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long categoryShowUpdate(CategoryShowParam param) {
//        Integer num = articleCategoryRepo.updateShowById(param.getId(), param.getShowFlag());
        // 将分类关联的文章同步显示隐藏
//        articleInfoRepo.updateShowByCategoryId(param.getId(), param.getShowFlag());
//        articleCategoryRepo.updateShowByPid(param.getId(), param.getShowFlag());

        // 将分类的子集/父级也同步进行隐藏显示
//        Set<Long> set = new HashSet<>();
//        set.add(param.getId());
//        findChildId(Set.of(param.getId()), set);
//        findParentId(param.getId(), set);
        // 子分类更新显示隐藏
//        articleCategoryRepo.updateShowByIdIn(set, param.getShowFlag());
        // 将分类关联的文章同步显示隐藏
//        articleInfoRepo.updateShowByCategoryIdIn(set, param.getShowFlag());
        // 子集分类设置显示 同步将上级分类也设置为显示
        if (StrUtil.equals(param.getShowFlag(), UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode())) {
            Set<Long> set = new HashSet<>();
            set.add(param.getId());
            findParentId(param.getId(), set);
            articleCategoryRepo.updateShowByIdIn(set, param.getShowFlag());
//            Optional<ArticleCategoryDO> byId = articleCategoryRepo.findById(param.getId());
//            if (byId.isPresent()) {
//                ArticleCategoryDO categoryDO = byId.get();
//                if (ObjectUtil.isNotNull(categoryDO.getPid())) {
//                    // 一级分类设置为显示之后 将关联的文章也设置为显示
//                    articleInfoRepo.updateShowByCategoryId(categoryDO.getPid(), param.getShowFlag());
//                    articleCategoryRepo.updateShowById(categoryDO.getPid(), param.getShowFlag());
//                }
//            }
        } else {
            // 隐藏只隐藏当前分类
            articleCategoryRepo.updateShowById(param.getId(), param.getShowFlag());
        }

        return param.getId();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer categorySortUpdate(ArticleCategorySortParam param) {
        return articleCategoryRepo.updateSortById(param.getId(), param.getSortNo());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer infoShowUpdate(ArticleInfoShowParam param) {

        /**
         * 除此文章外，是否还存在同文章分类的其他显示文章? 隐藏
         *      有，隐藏本条文章
         *      无，隐藏本条文章后，将文章分类隐藏掉
         *
         *  文章分类是否为显示状态
         *      是，显示本条,
         *      不是，显示本条，同时将文章分类更新为显示
         */

//        Optional<ArticleInfoDO> byId = articleInfoRepo.findById(param.getId());
//        if (byId.isEmpty()) {
//            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到文章信息");
//        }
//        Optional<ArticleCategoryDO> optional = articleCategoryRepo.findById(byId.get().getCategoryId());
//        if (optional.isEmpty()) {
//            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到文章分类信息");
//        }
//        // 隐藏
//        if (StrUtil.equals(param.getShowFlag(), UdcEnum.AD_SHOW_FLAG_HIDE.getValueCode())) {
//            List<ArticleInfoDO> doList = articleInfoRepo.findByCategoryIdAndShowFlag(byId.get().getCategoryId(), UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode());
//            if (CollectionUtil.isNotEmpty(doList) && doList.size() < 2) {
//                articleCategoryRepo.updateShowById(byId.get().getCategoryId(), param.getShowFlag());
//                // 一级分类隐藏之后 需要将二级分类也隐藏
//                articleCategoryRepo.updateShowByPid(byId.get().getCategoryId(), param.getShowFlag());
//            }
//        } else {
//            if (StrUtil.equals(optional.get().getShowFlag(), UdcEnum.AD_SHOW_FLAG_HIDE.getValueCode())) {
//                articleCategoryRepo.updateShowById(byId.get().getCategoryId(), param.getShowFlag());
//            }
//        }
        // 隐藏文章分类信息
        return articleInfoRepo.updateShowById(param.getId(), param.getShowFlag());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer infoSortUpdate(ArticleInfoSortParam param) {
        return articleInfoRepo.updateSortById(param.getId(), param.getSortNo());
    }

    @Override
    @SysCodeProc
    public PagingVO<ArticleInfoRespVO> infoSearch(ArticleInfoPagingParam searchParam) {
        PagingVO<ArticleInfoDTO> pagingResults = articleInfoRepoProc.findArticleInfoPagingResults(searchParam);
        List<ArticleInfoRespVO> respVoList = ArticleManageConvert.INSTANCE.copyInfoDTOsToVos(pagingResults.getRecords());

        if (CollectionUtil.isNotEmpty(respVoList)) {
            // 查询出上级分类名称
            for (ArticleInfoRespVO vo : respVoList) {
                Set<Long> set = new HashSet<>();
                set.add(vo.getCategoryId());
                findParentId(vo.getCategoryId(), set);
                List<ArticleCategoryDO> categoryDOList = articleCategoryRepo.findAllById(set);
                List<String> nameList = categoryDOList.stream().map(item -> item.getCategoryName()).collect(Collectors.toList());
                String name = CollectionUtil.join(nameList, "/");
                vo.setCategoryName(name);
            }
//            List<Long> pidList = respVoList.stream().filter(item -> ObjectUtil.isNotNull(item.getPid())).map(item -> item.getPid()).distinct().collect(Collectors.toList());
//            List<ArticleCategoryDO> parentList = articleCategoryRepo.findAllById(pidList);
//            Map<Long, String> parentMap = parentList.stream().collect(Collectors.toMap(ArticleCategoryDO::getId, ArticleCategoryDO::getCategoryName));
//            respVoList.stream().forEach(item -> {
//                if (StrUtil.equals(item.getShowFlag(), UdcEnum.AD_SHOW_FLAG_HIDE.getValueCode())) {
//                    item.setSortNo(null);
//                }
//                if (ObjectUtil.isNotNull(item.getPid())) {
//                    String name = item.getCategoryName();
//                    item.setCategoryName(parentMap.get(item.getPid()) + "/" + name);
//                }
//            });
        }
        return PagingVO.<ArticleInfoRespVO>builder()
                .total(pagingResults.getTotal())
                .records(respVoList).build();
    }

    @Override
    @SysCodeProc
    public ArticleInfoDetailRespVO queryInfoDetail(Long id) {
        ArticleInfoDTO dto = articleInfoRepoProc.findById(id);
        if (ObjectUtil.isNull(dto)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到文章详情");
        }
//        Long categoryId = dto.getCategoryId();
        ArticleInfoDetailRespVO respVO = ArticleManageConvert.INSTANCE.infoDTO2DetailVo(dto);
//        Optional<ArticleCategoryDO> byId = articleCategoryRepo.findById(categoryId);

//        if (byId.isPresent()) {
//            if (ObjectUtil.isNotNull(byId.get().getPid())) {
//                idList.add(byId.get().getPid());
//            }
//        }
        List<Long> idList = new ArrayList<>();
        if (ObjectUtil.isNotNull(dto.getPid())) {
            idList.add(dto.getPid());
        }
        idList.add(dto.getCategoryId());
        respVO.setCategoryIdList(idList);
        return respVO;
    }

    @Override
    @SysCodeProc
    public List<ArticleInfoCategoryRespVO> queryAllCategory() {
//        List<ArticleCategoryDO> categoryDOS = articleCategoryRepo.findAll();
//        List<ArticleInfoCategoryRespVO> voList = ArticleManageConvert.INSTANCE.categoryDOsToRespVos(categoryDOS);
//
//        // 一级分类
//        List<ArticleInfoCategoryRespVO> parentList = voList.stream()
//                .filter(item -> ObjectUtil.isNull(item.getPid()))
//                .sorted(Comparator.comparing(ArticleInfoCategoryRespVO::getCategoryName))
//                .collect(Collectors.toList());
//
//        // 二级分类
//        List<ArticleInfoCategoryRespVO> childList = voList.stream()
//                .filter(item -> ObjectUtil.isNotNull(item.getPid()))
//                .sorted(Comparator.comparing(ArticleInfoCategoryRespVO::getCategoryName))
//                .collect(Collectors.toList());
//
//        // 一级分类对应的二级分类
//        Map<Long, List<ArticleInfoCategoryRespVO>> map = childList.stream().collect(Collectors.groupingBy(ArticleInfoCategoryRespVO::getPid));
//
//        // 设置二级分类
//        for (ArticleInfoCategoryRespVO respVO : parentList) {
//            respVO.setTreeNodes(map.get(respVO.getId()));
//        }

        // 查询出所有的分类进行树化
        List<ArticleCategoryDTO> categoryDTOS = articleCategoryRepoProc.findAll();
        List<ArticleCategoryDTO> treeDtoList = TreeUtils.buildTree(null, false, categoryDTOS);
        List<ArticleInfoCategoryRespVO> respVOList = ArticleManageConvert.INSTANCE.categoryDTOS2InfoCategoryRespVOs(treeDtoList);

        return respVOList;
    }

    @Override
    public String queryArticleDetail(Long id) {
        Optional<ArticleInfoDO> byId = articleInfoRepo.findById(id);
        if (byId.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到文章详情");
        }
        return byId.get().getArticleDetail();
    }

    private void findChildId(Set<Long> pidSet, Set<Long> childIdSet) {
        // 查找当前父类对应的子类
        List<ArticleCategoryDO> categoryDOList = articleCategoryRepo.findByPidIn(pidSet);
        if (CollectionUtil.isNotEmpty(categoryDOList)) {
            Set<Long> collect = categoryDOList.stream().map(item -> item.getId()).collect(Collectors.toSet());
            childIdSet.addAll(collect);
            findChildId(collect, childIdSet);
        }
    }

    private void findParentId(Long childId, Set<Long> parentIdSet) {
        Optional<ArticleCategoryDO> byId = articleCategoryRepo.findById(childId);
        if (byId.isPresent()) {
            if (ObjectUtil.isNotNull(byId.get().getPid())) {
                parentIdSet.add(byId.get().getPid());
                findParentId(byId.get().getPid(), parentIdSet);
            }
        }
    }


    @Override
    public List<CategoryArticleRespVO> queryCategoryArticle() {
        // 查询出所有可用的文章分类信息
        List<ArticleCategoryDTO> categoryDTOS = articleCategoryRepoProc.findAllByShowFlag(UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode());
        if (CollectionUtil.isEmpty(categoryDTOS)) {
            return Collections.emptyList();
        }
        // 给文章分类匹配对应的文章信息
        Set<Long> categoryIdSet = categoryDTOS.stream().map(item -> item.getId()).collect(Collectors.toSet());
        List<ArticleInfoRespVO> articleInfoRespVOS = articleInfoRepoProc.findByCategoryIdInAndShowFlag(categoryIdSet, UdcEnum.AD_SHOW_FLAG_DISPLAY.getValueCode());
        if (CollectionUtil.isNotEmpty(articleInfoRespVOS)) {
            categoryDTOS.forEach(category -> {
                List<ArticleInfoRespVO> articleInfoList = new ArrayList<>();
                articleInfoRespVOS.stream().forEach(articleInfoRespVO -> {
                    if (ObjectUtil.equals(articleInfoRespVO, category.getId())) {
                        articleInfoList.add(articleInfoRespVO);
                    }
                });
                category.setArticleInfoRespVOS(articleInfoList);
            });
        }
        List<ArticleCategoryDTO> treeDtoList = TreeUtils.buildTree(null, true, categoryDTOS);
        return ArticleManageConvert.INSTANCE.categoryDTOS2CategoryRespVOs(treeDtoList);
    }
}
