package com.elitescloud.cloudt.ucenter.service.impl;


import cn.hutool.core.lang.Assert;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.udc.UdcProvider;
import com.elitescloud.cloudt.ucenter.api.dto.CommonSearchDTO;
import com.elitescloud.cloudt.ucenter.api.vo.param.CommonSearchFieldParam;
import com.elitescloud.cloudt.ucenter.api.vo.param.CommonSearchParam;
import com.elitescloud.cloudt.ucenter.common.constant.Constant;
import com.elitescloud.cloudt.ucenter.common.constant.UdcEnum;
import com.elitescloud.cloudt.ucenter.convert.CommonSearchConvert;
import com.elitescloud.cloudt.ucenter.entity.CommonSearchDO;
import com.elitescloud.cloudt.ucenter.repo.CommonSearchRepo;
import com.elitescloud.cloudt.ucenter.repo.CommonSearchRepoProc;
import com.elitescloud.cloudt.ucenter.rmi.RmiSysUDCService;
import com.elitescloud.cloudt.ucenter.service.CommonSearchService;
import com.google.common.collect.ImmutableMap;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;

import java.sql.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @Auther gyj
 * @Date 2023/8/2
 * @Version 1.0
 **/
@Service
@RequiredArgsConstructor
@Slf4j
public class CommonSearchServiceImpl implements CommonSearchService {

    private final CommonSearchRepo commonSearchRepo;

    private final CommonSearchRepoProc commonSearchRepoProc;

    private final UdcProvider udcProvider;

    private final RedisUtils redisUtils;

    private final RmiSysUDCService rmiSysUDCService;

    public static final String SELECT = "select";

    public static final String STRING = "STRING";

    public static final String NUMBER = "NUMBER";

    private static final Map<String, String> DATABASE_DRIVEN_MAP = ImmutableMap
            .<String, String>builder()
            .put(UdcEnum.DATABASE_TYPE_MYSQL.getValueCode(), "com.mysql.cj.jdbc.Driver")
            .put(UdcEnum.DATABASE_TYPE_ORACLE.getValueCode(), "")
            .build();


    @Override
    public void insertOrUpdate(CommonSearchParam commonSearchParam) {

        checkInsertOrUpdateParam(commonSearchParam);

        checkExist(commonSearchParam);

        if (commonSearchParam.getId() == null) {
            commonSearchRepo.save(CommonSearchConvert.INSTANCE.commonSearchParam2CommonSearchDO(commonSearchParam));
        } else {
            List<CommonSearchDO> commonSearchDOList = commonSearchRepo.findAllById(commonSearchParam.getId());
            Assert.isTrue(commonSearchDOList.size() == 1, "根据id查询不到/查询到多条数据");

            commonSearchRepo.save(CommonSearchConvert.INSTANCE.commonSearchParam2CommonSearchDO(commonSearchParam, commonSearchDOList.get(0)));
        }
    }


    private void checkInsertOrUpdateParam(CommonSearchParam param) {

        Assert.notEmpty(param.getCommonSearchId(), "通查ID必填");
        Assert.notEmpty(param.getCommonSearchName(), "通查名称必填");
        Assert.notEmpty(param.getCommonSearchContent(), "通查内容必填");
        Assert.notEmpty(param.getCommonSearchParam(), "通查参数必填");
        Assert.notNull(param.getConnectionName(), "连接名字必填");
        Assert.notEmpty(param.getDatabaseType(), "数据库类型必填");
        Assert.notEmpty(param.getUserName(), "数据库用户名必填");
        Assert.notEmpty(param.getPassword(), "数据库密码必填");
        Assert.notNull(param.getCacheFlag(), "是否缓存必填");

        if (param.getCacheFlag() == true) {
            Assert.notNull(param.getCacheLiveTime(), "缓存存活时间必填");
        }

        String commonSearchContent = param.getCommonSearchContent();
        //通查内容
        Assert.isTrue(commonSearchContent.startsWith(SELECT), "通查内容只能是select语句");

        List<CommonSearchFieldParam> commonSearchFieldParams;
        //通查参数
        try {
            String commonSearchParam = param.getCommonSearchParam();
            commonSearchFieldParams = JSONObject.parseArray(commonSearchParam, CommonSearchFieldParam.class);
        } catch (Exception e) {
            throw new BusinessException("通查参数解析异常", e);
        }

        if (CollectionUtils.isNotEmpty(commonSearchFieldParams)) {
            commonSearchFieldParams.stream().forEach(item -> Assert.isTrue(commonSearchContent.contains(item.getName()), "通查内容和通查参数名不匹配"));
        }

        int count = 0;
        int index = commonSearchContent.indexOf(Constant.QUESTION);

        while (index != -1) {
            count++;
            index = commonSearchContent.indexOf(Constant.QUESTION, index + 1);
        }

        Assert.equals(count, commonSearchFieldParams.size(), "通查内容和通查参数个数不匹配");
    }

    private void checkExist(CommonSearchParam commonSearchParam) {

        CommonSearchParam queryCommonSearchParam = new CommonSearchParam();
        queryCommonSearchParam.setCommonSearchId(commonSearchParam.getCommonSearchId());

        List<CommonSearchDTO> itmItemUomConvDTOList = commonSearchRepoProc.query(queryCommonSearchParam);
        CommonSearchDTO commonSearchDTO = itmItemUomConvDTOList.stream()
                .filter(item -> commonSearchParam.getCommonSearchId().equals(item.getCommonSearchId()))
                .findFirst()
                .orElse(null);
        if (commonSearchDTO != null) {
            Assert.equals(commonSearchParam.getId(), commonSearchDTO.getId(), "通查ID在系统已经存在");
        }
    }


    @Override
    public void delete(Long id) {
        Assert.notNull(id, "id必填");
        commonSearchRepo.deleteById(id);
    }

    @Override
    @SysCodeProc
    public PagingVO<CommonSearchDTO> page(CommonSearchParam commonSearchParam) {
        return commonSearchRepoProc.page(commonSearchParam);

    }

    @Override
    public void clearCache(String commonSearchId) {
        Assert.notEmpty(commonSearchId, "通查ID必填");
        redisUtils.del(commonSearchId);
    }

    @Override
    public List<Map> testInterface(CommonSearchParam param) {
        Assert.notNull(param.getId(), "id必填");
        Assert.notEmpty(param.getTestContent(), "测试内容必填");

        List<CommonSearchDO> commonSearchDOList = commonSearchRepo.findAllById(param.getId());
        Assert.isTrue(commonSearchDOList.size() == 1, "根据id查询不到/查询到多条数据");

        CommonSearchDO commonSearchDO = commonSearchDOList.get(0);

        String commonSearchParam = commonSearchDO.getCommonSearchParam();
        List<CommonSearchFieldParam> commonSearchFieldParamList = JSONObject.parseArray(commonSearchParam, CommonSearchFieldParam.class);

        String testContent = param.getTestContent();
        JSONObject jsonObject;

        try {
            jsonObject = JSONObject.parseObject(testContent);
        } catch (Exception e) {
            throw new BusinessException("测试内容解析失败", e);
        }

        if (CollectionUtils.isNotEmpty(commonSearchFieldParamList)) {
            commonSearchFieldParamList.stream().forEach(item -> {
                if (item.getMustHave() == true) {
                    Assert.notNull(jsonObject.get(item.getName()), item.getName().concat("必填"));
                }
            });
        }
        //连接数据库进行数据查询
        return query(commonSearchDO, commonSearchFieldParamList, jsonObject);
    }

    private List<Map> query(CommonSearchDO commonSearchDO,
                                  List<CommonSearchFieldParam> commonSearchFieldParamList,
                                  JSONObject jsonObject) {

        ResultSet resultSet = null;
        Connection connection = null;
        PreparedStatement preparedStatement = null;
        List<Map> mapList=new ArrayList<>();

        try {
            String drivenName = DATABASE_DRIVEN_MAP.get(commonSearchDO.getDatabaseType());
            Assert.notEmpty(drivenName, "数据库驱动名字为空");

            Class.forName(drivenName);

            connection = DriverManager.getConnection(commonSearchDO.getConnectionName(), commonSearchDO.getUserName(), commonSearchDO.getPassword());

            String sql = commonSearchDO.getCommonSearchContent();
            preparedStatement = connection.prepareStatement(sql);

            int index = 1;

            for (Map.Entry<String, Object> entry : jsonObject.entrySet()) {
                if (CollectionUtils.isNotEmpty(commonSearchFieldParamList)) {
                    CommonSearchFieldParam commonSearchFieldParamResult = commonSearchFieldParamList.stream()
                            .filter(commonSearchFieldParam -> entry.getKey().equals(commonSearchFieldParam.getName()))
                            .findFirst()
                            .orElse(null);
                    if (commonSearchFieldParamResult != null) {
                        if (STRING.equals(commonSearchFieldParamResult.getType())) {
                            preparedStatement.setString(index, entry.getValue().toString());
                        }
                        if (NUMBER.equals(commonSearchFieldParamResult.getType())) {
                            preparedStatement.setLong(index, Long.valueOf(entry.getValue().toString()));
                        }
                    }
                }
                index++;
            }

            resultSet = preparedStatement.executeQuery();

            // 处理查询结果集
            while (resultSet.next()) {
                Map<String,Object> map=new HashMap();
                int columnCount = resultSet.getMetaData().getColumnCount();
                for (int i = 1; i <= columnCount; i++) {
                    map.put(resultSet.getMetaData().getColumnName(i), resultSet.getObject(resultSet.getMetaData().getColumnName(i)));
                }
                mapList.add(map);
            }
        } catch (Exception e) {
            throw new BusinessException("查询数据异常", e);
        } finally {
            // 释放资源
            if (resultSet != null) {
                try {
                    resultSet.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
            if (connection != null) {
                try {
                    connection.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }

        }
        return mapList;
    }
}
