package com.elitescloud.cloudt.ucenter.service.impl;


import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSON;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.ucenter.api.dto.ReceiptSourceConfigDTO;
import com.elitescloud.cloudt.ucenter.api.dto.ReceiptSourceConfigDetailDTO;
import com.elitescloud.cloudt.ucenter.api.vo.param.GenerateTableParam;
import com.elitescloud.cloudt.ucenter.api.vo.param.ReceiptSourceConfigPagingParam;
import com.elitescloud.cloudt.ucenter.api.vo.resp.ReceiptSourceConfigDetailPagingRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.resp.ReceiptSourceConfigPagingRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.save.ReceiptSourceConfigSaveVO;
import com.elitescloud.cloudt.ucenter.convert.ReceiptSourceConfigConvert;
import com.elitescloud.cloudt.ucenter.convert.ReceiptSourceConfigDetailConvert;
import com.elitescloud.cloudt.ucenter.entity.ReceiptSourceConfigDO;
import com.elitescloud.cloudt.ucenter.entity.ReceiptSourceConfigDetailDO;
import com.elitescloud.cloudt.ucenter.repo.ReceiptSourceConfigDetailRepo;
import com.elitescloud.cloudt.ucenter.repo.ReceiptSourceConfigDetailRepoProc;
import com.elitescloud.cloudt.ucenter.repo.ReceiptSourceConfigRepo;
import com.elitescloud.cloudt.ucenter.repo.ReceiptSourceConfigRepoProc;
import com.elitescloud.cloudt.ucenter.service.ReceiptSourceConfigService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.util.*;
import java.util.stream.Collectors;

@Service
@AllArgsConstructor
@Slf4j
public class ReceiptSourceConfigServiceImpl implements ReceiptSourceConfigService {


    private final ReceiptSourceConfigRepo receiptSourceConfigRepo;
    private final ReceiptSourceConfigRepoProc configRepoProc;

    private final ReceiptSourceConfigDetailRepo configDetailRepo;
    private final ReceiptSourceConfigDetailRepoProc configDetailRepoProc;


    @Override
    @SysCodeProc
    public PagingVO<ReceiptSourceConfigPagingRespVO> searchPaging(ReceiptSourceConfigPagingParam param) {
        PagingVO<ReceiptSourceConfigDTO> pagingDto = configRepoProc.searchPaging(param);
        return ReceiptSourceConfigConvert.INSTANCE.dto2PagingVo(pagingDto);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long save(ReceiptSourceConfigSaveVO saveVO) {

        // 新增
        if (ObjectUtil.isNull(saveVO.getId())) {
            if (receiptSourceConfigRepo.existsBySourceNo(saveVO.getSourceNo())) {
                throw new BusinessException("来源单号已存在");
            }
            if (StrUtil.isNotBlank(saveVO.getParamValue())) {
                throw new BusinessException("请生成表体之后再保存接口参数信息");
            }
        } else {
            // 编辑
            Optional<ReceiptSourceConfigDO> byId = receiptSourceConfigRepo.findById(saveVO.getId());
            if (byId.isEmpty()) {
                throw new BusinessException("没有查询到来源单据配置");
            }
            // 校验接口参数
            checkParamValue(saveVO);
        }
        ReceiptSourceConfigDO receiptSourceConfigDO = ReceiptSourceConfigConvert.INSTANCE.SourceConfigSaveVo2Do(saveVO);
        ReceiptSourceConfigDO save = receiptSourceConfigRepo.save(receiptSourceConfigDO);
        return save.getId();
    }

    private void checkParamValue(ReceiptSourceConfigSaveVO saveVO) {

        if (StrUtil.isNotBlank(saveVO.getParamValue())) {
//            List<ReceiptSourceConfigDetailDO> detailDOS = configDetailRepo.findAllBySourceNo(saveVO.getSourceNo());
            List<ReceiptSourceConfigDetailDO> detailDOS = configDetailRepo.findByMasId(saveVO.getId());
            if (CollectionUtil.isEmpty(detailDOS)) {
                throw new BusinessException("请生成表体之后再保存接口参数信息");
            }
            Map<String, String> map = Map.of();
            try {
                map = JSONUtil.toBean(saveVO.getParamValue(), Map.class);
            } catch (Exception e) {
                log.error("解析接口参数出错：{}", e.getMessage());
                throw new BusinessException("解析接口参数出错");
            }

            if (map.entrySet().size() > detailDOS.size()) {
                throw new BusinessException("接口参数数量大于配置明细数量");
            }
            // 透出字段集合
            List<String> collect = detailDOS.stream().map(detail -> detail.getFieldName()).collect(Collectors.toList());

            // 需要参数中的每一个key都必须是在配置明细里面的透出字段中
            Iterator<String> paramKey = map.keySet().iterator();
            //每条的错误信息
            List<String> errorList = new ArrayList<>();
            while (paramKey.hasNext()) {
                String paramStr = paramKey.next();
                if (!collect.contains(paramStr)) {
                    errorList.add("参数{" + paramStr + "}没有透出字段匹配");
                }
            }
            if (CollectionUtil.isNotEmpty(errorList)) {
                throw new BusinessException(errorList.toString());
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Object delete(List<Long> ids) {

        if (CollectionUtil.isEmpty(ids)) {
            return ids;
        }
        List<ReceiptSourceConfigDO> receiptSourceConfigDOList = receiptSourceConfigRepo.findAllById(ids);
        if (CollectionUtil.isEmpty(receiptSourceConfigDOList)) {
            return ids;
        }
        List<String> sourceNoList = receiptSourceConfigDOList.stream()
                .filter(item -> StrUtil.isNotBlank(item.getSourceNo()))
                .map(item -> item.getSourceNo())
                .collect(Collectors.toList());
        // 删除配置明细
        if (CollectionUtil.isNotEmpty(sourceNoList)) {
            configDetailRepo.deleteAllBySourceNoIn(sourceNoList);
        }
        // 删除配置
        receiptSourceConfigRepo.deleteAllById(ids);
        return ids;
    }

    @Override
    @SysCodeProc
    public PagingVO<ReceiptSourceConfigDetailPagingRespVO> detailSearchPaging(ReceiptSourceConfigPagingParam param) {
        PagingVO<ReceiptSourceConfigDetailDTO> pagingDto = configDetailRepoProc.detailSearchPaging(param);
        return ReceiptSourceConfigDetailConvert.INSTANCE.dto2PagingVo(pagingDto);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Object generateTable(GenerateTableParam param) {

        if (StrUtil.isNotBlank(param.getTableDetail())) {
            Map<String, String> map = null;
            try {
                map = JSONUtil.toBean(param.getTableDetail(), Map.class);
            } catch (Exception e) {
                log.error("解析表体内容出错：{}", e.getMessage());
                throw new BusinessException("解析表体内容异常");
            }
            List<ReceiptSourceConfigDetailDO> detailDOS = new ArrayList<>();
            map.forEach((key, value) -> {
                ReceiptSourceConfigDetailDO detailDO = new ReceiptSourceConfigDetailDO();
                detailDO.setFieldName(key);
                detailDO.setFieldDefinition(value);
                detailDO.setSourceNo(param.getSourceNo());
                detailDO.setMasId(param.getMasId());
                detailDOS.add(detailDO);
            });
            configDetailRepo.saveAll(detailDOS);
        }
        return null;
    }

    @Override
    public Object test(String sourceNo) {
        ReceiptSourceConfigDO configDO = receiptSourceConfigRepo.findBySourceNo(sourceNo);

        if (ObjectUtil.isNull(configDO)) {
            throw new BusinessException("没查询到配置信息");
        }
        RestTemplate restTemplate = new RestTemplate();
        HttpHeaders headers = new HttpHeaders();
        String token = "Bearer " + SecurityContextUtil.currentToken();
        headers.add("Authorization", token);
        headers.add("Content-Type", "application/json");
        String paramValue = configDO.getParamValue();
        JSON jsonStr = JSONUtil.parse(paramValue);
        try {
            // restTemplate.getForEntity(configDO.getConnectUrl(), Object.class, configDO.getParamValue());
            ResponseEntity<Object> responseEntity = restTemplate.postForEntity(configDO.getConnectUrl(), new HttpEntity<>(jsonStr, headers), Object.class);
            return responseEntity.getBody();
        } catch (RestClientException e) {
            log.error("调用出错 {}", e.getMessage());
            throw new BusinessException("通用查询失败" + e.getMessage());
        }
    }
}
