package com.elitescloud.cloudt.system.controller.mng;

import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.model.vo.resp.index.*;
import com.elitescloud.cloudt.system.model.vo.save.index.ModifyPasswordSaveVO;
import com.elitescloud.cloudt.system.service.IndexUserService;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.constraints.NotBlank;
import java.util.List;

/**
 * 用户主页相关接口.
 *
 * @author Kaiser（wang shao）
 * 2022/9/19
 */
@Api(tags = "用户主页相关接口")
@RestController
@RequestMapping(value = "/mng/index/user", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class IndexUserController {

    private final IndexUserService indexUserService;

    public IndexUserController(IndexUserService indexUserService) {
        this.indexUserService = indexUserService;
    }

    /**
     * 获取当前用户信息
     *
     * @return 当前用户
     */
    @ApiOperation(value = "获取当前用户信息")
    @ApiOperationSupport(order = 1)
    @GetMapping(value = "/currentUser")
    public ApiResult<SysUserDTO> currentUser() {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        currentUser.getUser().setPassword(null);
        return ApiResult.ok(currentUser.getUser());
    }

    @ApiOperation(value = "获取当前员工信息")
    @ApiOperationSupport(order = 2)
    @GetMapping(value = "/currentEmp")
    public ApiResult<CurrentEmployeeRespVO> getEmployeeInfo() {
        return indexUserService.getEmployeeInfo();
    }

    /**
     * 获取用户菜单
     *
     * @return 用户菜单
     */
    @ApiOperation(value = "获取当前用户菜单")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "includeAction", value = "是否包含按钮", dataType = "boolean", defaultValue = "false"),
            @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean", defaultValue = "true"),
    })
    @GetMapping(value = "/menus")
    public ApiResult<List<UserMenuRespVO>> getUserMenu(@RequestParam(name = "includeAction", defaultValue = "false") Boolean includeAction,
                                                       @RequestParam(name = "tree", defaultValue = "true") Boolean tree) {
        return indexUserService.getUserMenu(includeAction, tree);
    }

    /**
     * 获取当前用户菜单下的按钮
     *
     * @return 按钮列表
     */
    @ApiOperation(value = "获取当前用户菜单下的按钮")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "menuCode", value = "菜单编码", dataType = "String", required = true),
    })
    @GetMapping(value = "/menu/action")
    public ApiResult<List<UserMenuRespVO>> getUserAction(@RequestParam(name = "menuCode") @NotBlank(message = "菜单编码为空") String menuCode) {
        return indexUserService.getUserAction(menuCode);
    }

    /**
     * 获取当前用户所有的按钮
     *
     * @return 按钮列表
     */
    @ApiOperation(value = "获取当前用户所有的按钮")
    @ApiOperationSupport(order = 4)
    @GetMapping(value = "/allAction")
    public ApiResult<List<UserMenuRespVO>> getUserActions() {
        return indexUserService.getUserAction(null);
    }

    /**
     * 获取用户的表格配置
     *
     * @param tableCode 表格编码
     * @return 配置内容
     */
    @ApiOperation(value = "获取用户的表格配置")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "tableCode", value = "表格编码", required = true, paramType = "query")
    @GetMapping(value = "/frontTable/cfg")
    public ApiResult<String> getFrontTableCfg(@RequestParam("tableCode") @NotBlank(message = "tableCode") String tableCode) {
        return indexUserService.getFrontTableCfg(tableCode);
    }

    /**
     * 获取用户的数据权限配置
     *
     * @return 数据权限
     */
    @ApiOperation(value = "获取用户的当前页面的数据权限配置")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParam(name = "all", value = "是否获取所有", defaultValue = "false", dataType = "boolean")
    @GetMapping(value = "/dataPermission/current")
    public ApiResult<UserDataPermissionRespVO> getUserDataPermission(@RequestParam(value = "all", defaultValue = "false", required = false) Boolean all) {
        return indexUserService.getDataPermission(all);
    }

    @ApiOperation(value = "获取用户的当前页面的字段权限")
    @ApiOperationSupport(order = 7)
    @GetMapping(value = "/fieldPermission/current")
    public ApiResult<List<AuthedUserFieldRespVO>> getUserFieldPermission() {
        return indexUserService.getFieldPermission();
    }

    /**
     * 修改当前用户密码
     *
     * @param saveVO 保存信息
     * @return 修改结果
     */
    @ApiOperation(value = "修改当前用户密码【通过老密码】")
    @ApiOperationSupport(order = 11)
    @PostMapping(value = "/update/currentPwd")
    public ApiResult<Boolean> updatePassword(@RequestBody ModifyPasswordSaveVO saveVO) {
        return indexUserService.updatePassword(saveVO);
    }

    /**
     * 切换用户的当前租户
     *
     * @param tenantId 租户ID
     * @return 切换结果
     */
    @ApiOperation(value = "切换用户的当前租户")
    @ApiOperationSupport(order = 12)
    @ApiImplicitParam(name = "tenantId", value = "租户ID", dataType = "long", required = true)
    @GetMapping(value = "/switchTenant")
    public ApiResult<SysUserDTO> switchTenant(@RequestParam("tenantId") Long tenantId) {
        return indexUserService.switchTenant(tenantId);
    }

    /**
     * 切换用户的当前组织
     *
     * @param orgId 组织ID
     * @return 切换结果
     */
    @ApiOperation(value = "切换用户的当前组织")
    @ApiOperationSupport(order = 13)
    @ApiImplicitParam(name = "orgId", value = "组织ID", dataType = "long", required = true)
    @GetMapping(value = "/switchOrg")
    public ApiResult<SysUserDTO> switchOrg(@RequestParam("orgId") Long orgId) {
        return indexUserService.switchOrg(orgId);
    }

    /**
     * 获取当前用户授权的字段
     *
     * @param menuCode 菜单编码
     * @param apiCode  API接口编码
     * @return 字段列表
     * @deprecated 已废弃，改用 {@link #getUserFieldPermission()}
     */
    @ApiOperation(value = "获取当前用户授权的字段")
    @ApiOperationSupport(order = 4)
    @ApiIgnore
    @GetMapping(value = "/user/field")
    @Deprecated(since = "3.3.0")
    public ApiResult<List<UserFieldRespVO>> getUserAuthorizedField(@RequestParam(name = "menuCode") @NotBlank(message = "菜单编码为空") String menuCode,
                                                                   @RequestParam(name = "apiCode") @NotBlank(message = "接口编码为空") String apiCode) {
        return indexUserService.getUserField(menuCode, apiCode);
    }
}
