package com.elitescloud.cloudt.system.controller.mng.role;

import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameCheckParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.constant.BusinessObjectConstant;
import com.elitescloud.cloudt.system.constant.DataPermissionType;
import com.elitescloud.cloudt.system.model.vo.resp.api.SysMenuApiRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.dpr.DataPermissionRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.dpr.SysDprRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeeUserInfoRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.permission.DataPermissionTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.GroupRoleRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.PermissionDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.PermissionMenuRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RolePermissionRespVO;
import com.elitescloud.cloudt.system.model.vo.save.dpr.DataPermissionSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.dpr.SysDprBatchSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.dpr.SysDprSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.role.MenuApiSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.role.RoleAppPermissionSaveVO;
import com.elitescloud.cloudt.system.service.PermissionMngService;
import com.elitescloud.cloudt.system.service.PermissionQueryService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 权限管理.
 *
 * @author Kaiser（wang shao）
 * 2023/1/10
 */
@Api(tags = "权限管理")
@RestController
@RequestMapping(value = "/mng/permission", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_ROLE_PERMISSION)
public class PermissionMngController {

    private final PermissionQueryService permissionQueryService;
    private final PermissionMngService permissionMngService;

    public PermissionMngController(PermissionQueryService permissionQueryService, PermissionMngService permissionMngService) {
        this.permissionQueryService = permissionQueryService;
        this.permissionMngService = permissionMngService;
    }

    @ApiOperation(value = "获取角色组及角色")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParam(name = "showDisabled", value = "是否显示禁用的", defaultValue = "false")
    @GetMapping(value = "/listGroupRole")
    public ApiResult<List<GroupRoleRespVO>> listGroupRole(Boolean showDisabled) {
        return permissionQueryService.listGroupRole(showDisabled);
    }

    @ApiOperation(value = "向角色中添加账号")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "incremental", value = "是否为增量操作", dataType = "boolean", required = true, defaultValue = "true"),
    })
    @PostMapping(value = "/role/users")
    public ApiResult<Long> addRoleUser(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                       @RequestParam(name = "incremental", required = false, defaultValue = "true") Boolean incremental,
                                       @RequestBody @NotEmpty(message = "请选择账号") List<Long> userIds) {
        return permissionMngService.saveRoleUser(roleId, userIds, incremental);
    }

    @ApiOperation(value = "向角色中添加员工")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "incremental", value = "是否为增量操作", dataType = "boolean", required = true, defaultValue = "true"),
    })
    @PostMapping(value = "/role/employee")
    public ApiResult<Long> addRoleEmployee(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                           @RequestParam(name = "incremental", required = false, defaultValue = "true") Boolean incremental,
                                           @RequestBody @NotEmpty(message = "请选择账号") List<Long> userIds) {
        return permissionMngService.saveRoleEmployee(roleId, userIds, incremental);
    }

    @ApiOperation(value = "从角色中删除账号")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "userId", value = "账号ID", dataType = "long", required = true)
    })
    @DeleteMapping(value = "/role/user")
    public ApiResult<Long> removeRoleUser(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                          @RequestParam(name = "userId") @NotNull(message = "账号ID为空") Long userId) {
        return permissionMngService.removeRoleUser(roleId, userId);
    }

    @ApiOperation(value = "获取角色下的所有用户")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", required = true, dataType = "long", paramType = "query"),
            @ApiImplicitParam(name = "onlyEmployee", value = "是否仅显示员工", dataType = "boolean", paramType = "query", defaultValue = "true"),
    })
    @GetMapping(value = "/role/user")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取角色下的所有用户"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<List<EmployeeUserInfoRespVO>> getUserOfRole(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                                                 @RequestParam(name = "onlyEmployee", required = false, defaultValue = "true") Boolean onlyEmployee) {
        return permissionQueryService.listEmployeeUserByRole(roleId, onlyEmployee);
    }

    /**
     * 获取用户的角色
     *
     * @param userId 用户ID
     * @return 角色
     */
    @ApiOperation(value = "获取用户的角色")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "userId", value = "用户ID", required = true, dataType = "int", paramType = "path")
    @GetMapping(value = "/user/{userId}/role")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取用户的角色"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<List<IdCodeNameCheckParam>> getRole(@PathVariable("userId") Long userId) {
        return permissionQueryService.getRoleByUserId(userId, false);
    }

    /**
     * 保存用户的角色
     *
     * @param userId 用户ID
     * @return 用户ID
     */
    @ApiOperation(value = "保存用户的角色")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true, dataType = "int", paramType = "path"),
            @ApiImplicitParam(name = "roleIds", value = "角色ID", required = true, dataType = "int", allowMultiple = true),
    })
    @PutMapping(value = "/user/{userId}/role")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存用户的角色"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> saveRole(@PathVariable("userId") Long userId, @RequestBody List<Long> roleIds) {
        return permissionMngService.saveUserRoles(userId, roleIds);
    }

    /**
     * 获取角色的菜单权限
     *
     * @return 菜单权限树
     */
    @ApiOperation(value = "获取角色的菜单权限")
    @ApiOperationSupport(order = 11)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "appCode", value = "应用编码", required = true),
            @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean"),
    })
    @GetMapping(value = "/role/menu")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取角色的菜单权限"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<List<RolePermissionRespVO>> getPermissionMenu(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                                                   @RequestParam(name = "appCode") @NotBlank(message = "应用编码为空") String appCode,
                                                                   @RequestParam(name = "tree", defaultValue = "true") Boolean tree) {
        return permissionQueryService.getPermissionMenuByRole(roleId, appCode, tree);
    }

    /**
     * 保存角色的菜单权限
     *
     * @param roleId     角色ID
     * @param saveVOList 权限信息
     * @return 保存结果
     */
    @ApiOperation(value = "保存角色的菜单权限")
    @ApiOperationSupport(order = 12)
    @ApiImplicitParam(name = "roleId", value = "角色ID", dataType = "int", required = true)
    @PostMapping(value = "/role/{roleId}/menu")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存角色的菜单权限"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> savePermissionMenu(@PathVariable("roleId") Long roleId,
                                              @RequestBody @Valid List<RoleAppPermissionSaveVO> saveVOList) {
        return permissionMngService.savePermissionMenu(roleId, saveVOList);
    }

    /**
     * 添加角色的菜单权限
     *
     * @param roleId 角色ID
     * @param saveVO 权限信息
     * @return 保存结果
     */
    @ApiOperation(value = "添加角色的菜单权限")
    @ApiOperationSupport(order = 13)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", dataType = "long", required = true),
    })
    @PostMapping(value = "/role/menu/dynamic")
    @BusinessObjectOperation(operationType = OperationTypeEnum.ADD, operationDescription = "添加角色的菜单权限"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> addPermissionMenu(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                             @RequestBody @Valid RoleAppPermissionSaveVO saveVO) {
        return permissionMngService.addPermissionMenu(roleId, saveVO);
    }

    @ApiOperation(value = "移除角色的菜单权限")
    @ApiOperationSupport(order = 14)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", dataType = "long", required = true),
    })
    @DeleteMapping(value = "/role/menu/dynamic")
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "移除角色的菜单权限"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> deletePermissionMenu(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                                @RequestBody @Valid RoleAppPermissionSaveVO saveVO) {
        return permissionMngService.removePermissionMenu(roleId, saveVO);
    }

    /**
     * 数据权限-数据范围
     *
     * @return 数据范围列表
     */
    @ApiOperation(value = "数据权限-数据范围")
    @ApiOperationSupport(order = 21)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "operationCode", value = "操作编码"),
            @ApiImplicitParam(name = "businessObjectCode", value = "业务对象编码"),
    })
    @GetMapping(value = "/data/ranges")
    public ApiResult<List<CodeNameParam>> getDprDataRanges(@RequestParam(name = "operationCode", required = false) String operationCode,
                                                           @RequestParam(name = "businessObjectCode", required = false) String businessObjectCode) {
        return permissionQueryService.dataPermissionRanges(operationCode, businessObjectCode);
    }

    /**
     * 数据权限-规则条件
     *
     * @return 规则条件列表
     */
    @ApiOperation(value = "数据权限-规则条件")
    @ApiOperationSupport(order = 22)
    @GetMapping(value = "/data/conditions")
    public ApiResult<List<CodeNameParam>> getDprRuleCondition() {
        return permissionQueryService.dataPermissionRuleCondition();
    }

    /**
     * 数据权限-规则的取值方式
     *
     * @return 规则的取值方式列表
     */
    @ApiOperation(value = "数据权限-规则的取值方式")
    @ApiOperationSupport(order = 23)
    @GetMapping(value = "/data/ruleValueMethod")
    public ApiResult<List<CodeNameParam>> getDprRuleValueMethod() {
        return permissionQueryService.dataPermissionRuleValueMethod();
    }

    @ApiOperation(value = "获取分配数据权限的菜单接口树")
    @ApiOperationSupport(order = 24)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleCode", value = "角色编码", required = true, dataType = "String"),
            @ApiImplicitParam(name = "type", value = "权限类型", required = true, dataType = "String", allowableValues = "MENU_OPERATION_RULE,BUSINESS_OPERATION_RULE,BUSINESS_OBJECT_RULE,BUSINESS_RESOURCE_RULE"),
            @ApiImplicitParam(name = "withRef", value = "是否显示关联的", dataType = "boolean"),
            @ApiImplicitParam(name = "tree", value = "是否返回树状数据", dataType = "boolean"),
    })
    @GetMapping(value = "/dataPermission/tree/v1")
    public ApiResult<List<DataPermissionTreeNodeRespVO>> getDataPermissionTree(@RequestParam(name = "roleCode") @NotBlank(message = "角色编码为空") String roleCode,
                                                                               @RequestParam(name = "type", defaultValue = "MENU_OPERATION_RULE") String type,
                                                                               @RequestParam(name = "withRef", required = false, defaultValue = "false") Boolean withRef,
                                                                               @RequestParam(name = "tree", required = false, defaultValue = "false") Boolean tree) {
        return permissionQueryService.dataPermissionTree(roleCode, type, withRef, null, tree);
    }

    @ApiOperation(value = "保存角色分配的数据权限")
    @ApiOperationSupport(order = 25)
    @PostMapping(value = "/data/rolePermission")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存角色分配的数据权限"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Boolean> saveDataPermission(@RequestBody @Valid DataPermissionSaveVO saveVO) {
        return permissionMngService.saveDataPermission(saveVO);
    }

    @ApiOperation(value = "获取角色分配的数据权限")
    @ApiOperationSupport(order = 26)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleCode", value = "角色编码", required = true),
            @ApiImplicitParam(name = "permissionType", value = "权限类型", required = true, allowableValues = "MENU_OPERATION_RULE,BUSINESS_OPERATION_RULE,BUSINESS_OBJECT_RULE,BUSINESS_RESOURCE_RULE"),
            @ApiImplicitParam(name = "menuCode", value = "菜单编码"),
            @ApiImplicitParam(name = "operationCode", value = "操作编码"),
            @ApiImplicitParam(name = "businessObjectCode", value = "业务对象编码"),
    })
    @GetMapping(value = "/data/rolePermission")
    public ApiResult<DataPermissionRespVO> getDataPermission(@RequestParam(name = "roleCode") @NotBlank(message = "角色编码为空") String roleCode,
                                                             @RequestParam(name = "permissionType") @NotBlank(message = "权限类型为空") String permissionType,
                                                             @RequestParam(name = "menuCode", required = false) String menuCode,
                                                             @RequestParam(name = "operationCode", required = false) String operationCode,
                                                             @RequestParam(name = "businessObjectCode", required = false) String businessObjectCode) {
        return permissionQueryService.getDataPermission(roleCode, permissionType, menuCode, operationCode, businessObjectCode);
    }

    /**
     * 保存角色分配的批量数据权限
     *
     * @param saveVO 保存信息
     * @return 角色ID
     */
    @ApiOperation(value = "保存角色分配的批量数据权限")
    @ApiOperationSupport(order = 42)
    @PostMapping(value = "/dataPermission/roleMenuApiBatch")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存角色分配的批量数据权限"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Boolean> saveApiDataPermission(@RequestBody @Valid SysDprBatchSaveVO saveVO) {
        return permissionMngService.saveApiDataPermission(saveVO);
    }

    /**
     * 根据权限编码获取权限信息
     *
     * @param code 权限编码
     * @return 权限信息
     */
    @GetMapping(value = "/{code}/detail")
    @ApiOperation(value = "根据权限编码获取权限信息")
    @ApiOperationSupport(order = 61)
    @ApiImplicitParam(name = "code", value = "权限编码")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "根据权限编码获取权限信息"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<PermissionDetailRespVO> permissionDetail(@PathVariable("code") String code) {
        return permissionQueryService.getPermissionDetailByCode(code);
    }

    /**
     * 获取菜单下的API
     *
     * @param menuCode 菜单编码
     * @return API列表
     * @deprecated 已废弃，现在菜单下挂载的是业务操作
     */
    @ApiOperation(value = "【已废弃】获取菜单下挂载的API")
    @ApiOperationSupport(order = 31)
    @ApiIgnore
    @ApiImplicitParam(name = "menuCode", value = "菜单编码", required = true)
    @GetMapping(value = "/data/menu/api")
    @Deprecated(since = "3.3.0")
    public ApiResult<List<SysMenuApiRespVO>> getApi(@RequestParam(name = "menuCode") @NotBlank(message = "菜单编码为空") String menuCode) {
        return permissionQueryService.getApi(menuCode);
    }

    /**
     * 保存菜单挂载的接口
     *
     * @param saveVO 保存信息
     * @return 结果
     * @deprecated 已废弃，现在菜单下挂载的是业务操作
     */
    @ApiIgnore
    @ApiOperation(value = "【已废弃】保存接口绑定的菜单")
    @ApiOperationSupport(order = 32)
    @PostMapping(value = "/api/menus")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存菜单挂载的接口"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    @Deprecated(since = "3.3.0")
    public ApiResult<Boolean> saveMenuApi(@RequestBody @Valid MenuApiSaveVO saveVO) {
        return permissionMngService.saveMenuApi(saveVO);
    }

    /**
     * 获取接口绑定的菜单
     *
     * @param apiId
     * @param appCode
     * @return
     * @deprecated 已废弃，现在菜单下挂载的是业务操作
     */
    @ApiIgnore
    @ApiOperation(value = "【已废弃】获取接口绑定的菜单")
    @ApiOperationSupport(order = 32)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "apiId", value = "接口ID", required = true),
            @ApiImplicitParam(name = "appCode", value = "应用编码"),
    })
    @GetMapping(value = "/api/menus")
    @Deprecated(since = "3.3.0")
    public ApiResult<List<Long>> getMenuApi(@RequestParam(name = "apiId") @NotNull(message = "接口ID为空") Long apiId,
                                            @RequestParam(name = "appCode", required = false) String appCode) {
        return permissionQueryService.getMenuIdOfApi(apiId, appCode);
    }

    /**
     * 获取接口绑定的菜单
     *
     * @param apiId
     * @param appCode
     * @return
     * @deprecated 已废弃，现在菜单下挂载的是业务操作
     */
    @ApiIgnore
    @ApiOperation(value = "【已废弃】获取接口绑定的菜单详细信息")
    @ApiOperationSupport(order = 33)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "apiId", value = "接口ID", required = true),
            @ApiImplicitParam(name = "appCode", value = "应用编码"),
    })
    @GetMapping(value = "/api/menusDetail")
    @Deprecated(since = "3.3.0")
    public ApiResult<List<PermissionMenuRespVO>> getMenuApiDetail(@RequestParam(name = "apiId") @NotNull(message = "接口ID为空") Long apiId,
                                                                  @RequestParam(name = "appCode", required = false) String appCode) {
        return permissionQueryService.getMenuDetailOfApi(apiId, appCode);
    }

    @ApiIgnore
    @ApiOperation(value = "【已废弃】获取分配数据权限的菜单接口树")
    @ApiOperationSupport(order = 24)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", required = true, dataType = "int"),
            @ApiImplicitParam(name = "tree", value = "是否返回树状数据", dataType = "boolean"),
    })
    @GetMapping(value = "/dataPermission/tree")
    @Deprecated(since = "3.3.0")
    public ApiResult<List<DataPermissionTreeNodeRespVO>> getDataPermissionTree(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                                                               @RequestParam(name = "tree", required = false, defaultValue = "false") Boolean tree) {
        return permissionQueryService.dataPermissionTree(roleId, null, tree);
    }

    /**
     * 获取角色分配的数据权限
     *
     * @param roleId   角色ID
     * @param menuCode 菜单编码
     * @param apiCode  API接口编码
     * @return 数据权限信息
     * @deprecated 已废弃，改用{@link #getDataPermission(String, String, String, String, String, String)}
     */
    @ApiIgnore
    @ApiOperation(value = "【已废弃】获取角色分配的数据权限")
    @ApiOperationSupport(order = 26)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", required = true),
            @ApiImplicitParam(name = "menuCode", value = "菜单编码", required = true),
            @ApiImplicitParam(name = "apiCode", value = "接口编码", required = true),
    })
    @GetMapping(value = "/data/roleMenuApi")
    @Deprecated(since = "3.3.0")
    public ApiResult<SysDprRespVO> getApiDataPermission(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                                        @RequestParam(name = "menuCode") @NotBlank(message = "菜单编码为空") String menuCode,
                                                        @RequestParam(name = "apiCode") @NotBlank(message = "接口编码为空") String apiCode) {
        return permissionQueryService.getApiDataPermission(roleId, menuCode, apiCode);
    }

    /**
     * 保存数据权限
     *
     * @param saveVO 保存信息
     * @return 角色ID
     * @deprecated 已废弃，改用 {@link #saveDataPermission(DataPermissionSaveVO)}
     */
    @ApiIgnore
    @ApiOperation(value = "【已废弃】保存角色分配的数据权限")
    @ApiOperationSupport(order = 25)
    @PostMapping(value = "/data/roleMenuApi")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存角色分配的数据权限"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    @Deprecated(since = "3.3.0")
    public ApiResult<Long> saveApiDataPermission(@RequestBody @Valid SysDprSaveVO saveVO) {
        return permissionMngService.saveApiDataPermission(saveVO);
    }
}
