package com.elitescloud.cloudt.system.controller.mng.sys;

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.model.vo.resp.role.GroupRoleRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RoleDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RoleGroupDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.SubUserDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.SubUserListRespVO;
import com.elitescloud.cloudt.system.model.vo.save.role.RoleGroupSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.role.RoleSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.sys.SubUserBatchSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.sys.SubUserSaveVO;
import com.elitescloud.cloudt.system.service.RoleGroupMngService;
import com.elitescloud.cloudt.system.service.RoleMngService;
import com.elitescloud.cloudt.system.service.RoleQueryService;
import com.elitescloud.cloudt.system.service.SubUserMngService;
import com.elitescloud.cloudt.system.service.common.constant.BelongType;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Set;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/27
 */
@Api(tags = "子账号管理")
@RestController
@RequestMapping(value = "/mng/subUser", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class SubUserMngController {

    private RoleMngService roleMngService;
    private RoleQueryService roleQueryService;
    private RoleGroupMngService roleGroupMngService;
    private SubUserMngService service;

    @ApiOperation(value = "角色及角色组列表")
    @ApiOperationSupport(order = 1)
    @GetMapping(value = "/roles")
    public ApiResult<List<GroupRoleRespVO>> listGroupRole() {
        return roleMngService.listGroupRole(true, BelongType.getBelongerPersonal());
    }

    @PostMapping(value = "/roleGroup/save")
    @ApiOperation(value = "保存角色组")
    @ApiOperationSupport(order = 2)
    public ApiResult<Long> upsertRoleGroup(@Valid @RequestBody RoleGroupSaveVO saveVO) {
        return roleGroupMngService.upsert(saveVO, BelongType.getBelongerPersonal());
    }

    @DeleteMapping(value = "/roleGroup/delete")
    @ApiOperation(value = "删除角色组")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParam(name = "id", value = "角色组ID", required = true, dataType = "int")
    public ApiResult<Long> deleteRoleGroup(@RequestParam(value = "id") @NotNull(message = "角色组ID为空") Long id) {
        return roleGroupMngService.delete(id);
    }

    @ApiOperation(value = "修改角色组名称")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "角色组ID", required = true, dataType = "long"),
            @ApiImplicitParam(name = "name", value = "角色组名称", required = true, dataType = "String"),
    })
    @PatchMapping(value = "/roleGroup/update/name")
    public ApiResult<Long> updateRoleGroupName(@RequestParam(name = "id") @NotNull(message = "角色组ID为空") Long id,
                                               @RequestParam(name = "name") @NotBlank(message = "角色组名称为空") String name) {
        return roleGroupMngService.updateName(id, name);
    }

    @GetMapping(value = "/roleGroup/get")
    @ApiOperation(value = "获取角色组详细信息")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "角色组ID", required = true, dataType = "int")
    public ApiResult<RoleGroupDetailRespVO> getRoleGroupDetail(@RequestParam(value = "id") @NotNull(message = "角色组ID为空") Long id) {
        return roleGroupMngService.getDetail(id);
    }

    @GetMapping(value = "/roleGroups")
    @ApiOperation(value = "查询角色分组列表")
    @ApiOperationSupport(order = 6)
    public ApiResult<List<IdCodeNameParam>> queryRoleGroupList() {
        return roleQueryService.listAllRoleGroup(BelongType.getBelongerPersonal());
    }

    @ApiOperation(value = "保存角色")
    @ApiOperationSupport(order = 11)
    @PostMapping(value = "/role/save")
    public ApiResult<Long> saveRole(@RequestBody @Valid RoleSaveVO saveVO) {
        saveVO.setBusinessKey(null);
        return roleMngService.save(saveVO, BelongType.getBelongerPersonal());
    }

    @ApiOperation(value = "获取角色详情")
    @ApiOperationSupport(order = 12)
    @ApiImplicitParam(name = "id", value = "角色ID", required = true, dataType = "int")
    @GetMapping(value = "/role/get/{id}")
    public ApiResult<RoleDetailRespVO> getRole(@PathVariable("id") Long id) {
        return roleMngService.get(id);
    }

    @ApiOperation(value = "修改角色名称")
    @ApiOperationSupport(order = 13)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "角色ID", required = true, dataType = "long"),
            @ApiImplicitParam(name = "name", value = "角色名称", required = true, dataType = "String"),
    })
    @PatchMapping(value = "/role/update/name")
    public ApiResult<Long> updateRoleName(@RequestParam(name = "id") @NotNull(message = "角色ID为空") Long id,
                                          @RequestParam(name = "name") @NotBlank(message = "角色名称为空") String name) {
        return roleMngService.updateName(id, name);
    }

    @ApiOperation(value = "删除角色")
    @ApiOperationSupport(order = 14)
    @ApiImplicitParam(name = "id", value = "角色ID", required = true, dataType = "int")
    @DeleteMapping(value = "/role/del/{id}")
    public ApiResult<Long> delete(@PathVariable("id") Long id) {
        return roleMngService.delete(id);
    }

    @ApiOperation(value = "保存子账号信息")
    @ApiOperationSupport(order = 21)
    @PostMapping(value = "/save")
    public ApiResult<Long> save(@RequestBody @Valid SubUserSaveVO saveVO) {
        return service.save(saveVO);
    }

    @ApiOperation(value = "批量添加子账号")
    @ApiOperationSupport(order = 22)
    @PostMapping(value = "/saveBatch")
    public ApiResult<List<Long>> saveBatch(@RequestBody @Valid SubUserBatchSaveVO saveVO) {
        return service.saveBatch(saveVO);
    }

    @ApiOperation(value = "查询子账号列表")
    @ApiOperationSupport(order = 23, ignoreParameters = {"belongType", "belongerId", "roleCode"})
    @ApiImplicitParams({
            @ApiImplicitParam(name = "belongType", value = "归属类型"),
            @ApiImplicitParam(name = "belongerId", value = "归属者ID"),
            @ApiImplicitParam(name = "roleCode", value = "角色编码"),
    })
    @GetMapping(value = "/list")
    public ApiResult<List<SubUserListRespVO>> querySubUserList(@RequestParam(name = "belongType", required = false) BelongType belongType,
                                                               @RequestParam(name = "belongerId", required = false) String belongerId,
                                                               @RequestParam(name = "roleCode", required = false) String roleCode) {
        return service.querySubUserList(belongType, belongerId, roleCode);
    }

    @ApiOperation(value = "删除子账号记录")
    @ApiOperationSupport(order = 24)
    @ApiImplicitParam(name = "delUser", value = "是否同时删除登录账号", dataType = "boolean")
    @DeleteMapping(value = "/delete")
    public ApiResult<Set<Long>> del(@RequestBody Set<Long> ids,
                                    @RequestParam(name = "delUser", required = false, defaultValue = "false") Boolean delUser) {
        return service.del(ids, delUser);
    }

    @ApiOperation(value = "获取子账号分配的详细信息")
    @ApiOperationSupport(order = 25)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "long")
    @GetMapping(value = "/edit")
    public ApiResult<SubUserDetailRespVO> get(@RequestParam(name = "id") @NotNull(message = "ID为空") Long id) {
        return service.get(id);
    }

    @ApiOperation(value = "获取子账号分配的角色")
    @ApiOperationSupport(order = 26)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "long")
    @GetMapping(value = "/user/roles")
    public ApiResult<List<IdCodeNameParam>> getRolesOfSubUser(@RequestParam(name = "id") @NotNull(message = "ID为空") Long id) {
        return service.getRolesOfSubUser(id);
    }

    @ApiOperation(value = "获取子账号分配的菜单")
    @ApiOperationSupport(order = 27)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "long")
    @GetMapping(value = "/user/menuCodes")
    public ApiResult<Set<String>> getMenuCodesOfSubUser(@RequestParam(name = "id") @NotNull(message = "ID为空") Long id) {
        return service.getMenuCodesOfSubUser(id);
    }

    @Autowired
    public void setRoleMngService(RoleMngService roleMngService) {
        this.roleMngService = roleMngService;
    }

    @Autowired
    public void setRoleGroupMngService(RoleGroupMngService roleGroupMngService) {
        this.roleGroupMngService = roleGroupMngService;
    }

    @Autowired
    public void setService(SubUserMngService service) {
        this.service = service;
    }

    @Autowired
    public void setRoleQueryService(RoleQueryService roleQueryService) {
        this.roleQueryService = roleQueryService;
    }
}
