package com.elitescloud.cloudt.system.modules.finereport.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.crypto.asymmetric.KeyType;
import cn.hutool.crypto.asymmetric.RSA;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.constant.AccountType;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.modules.finereport.common.FineReportConstants;
import com.elitescloud.cloudt.system.modules.finereport.model.ViewReportParamRespVO;
import com.elitescloud.cloudt.system.modules.finereport.service.FineReportService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.util.UriComponentsBuilder;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/6/21
 */
@Slf4j
@Service
public class FineReportServiceImpl implements FineReportService {

    @Autowired
    private SystemProperties systemProperties;

    @Override
    public ApiResult<String> getSsoToken() {
        var props = systemProperties.getFineReport();
        if (props.getAccountType() == null || CharSequenceUtil.isBlank(props.getPublicKey())) {
            return ApiResult.fail("获取SsoToken失败");
        }

        var ssoToken = this.generateToken(props);
        return ApiResult.ok(ssoToken);
    }

    @Override
    public ApiResult<ViewReportParamRespVO> viewReport(String viewlet, Boolean encoded) {
        var props = systemProperties.getFineReport();
        if (CharSequenceUtil.isBlank(props.getServerAddr())) {
            return ApiResult.fail("报表未配置");
        }
        if (CharSequenceUtil.isBlank(viewlet)) {
            viewlet = CharSequenceUtil.nullToDefault(props.getDefaultViewlet(), "");
        }

        ViewReportParamRespVO respVO = new ViewReportParamRespVO();
        respVO.setServerAddr(props.getServerAddr());
        respVO.setUri(obtainUriPath(viewlet));

        // 参数
        MultiValueMap<String, String> uriParams = new LinkedMultiValueMap<>();
        if (StringUtils.hasText(viewlet)) {
            uriParams.add(FineReportConstants.PARAM_VIEWLET, viewlet);
        }
        // token
        String token = this.generateToken(props);
        if (encoded == null || encoded) {
            token = URLEncoder.encode(token, StandardCharsets.UTF_8);
        }
        if (StringUtils.hasText(props.getPublicKey())) {
            // 提供公钥，则加密
            uriParams.add(FineReportConstants.PARAM_SSO_TOKEN, token);
        } else {
            uriParams.add(FineReportConstants.PARAM_FINE_USERNAME, token);
        }
        respVO.setUriParams(uriParams);

        var url = UriComponentsBuilder.fromUriString(respVO.getServerAddr())
                .path(respVO.getUri())
                .queryParams(uriParams)
                .build()
                .normalize()
                .toString();
        respVO.setUrl(url);
        log.info("请求报表：{}", url);

        return ApiResult.ok(respVO);
    }

    private String obtainUriPath(String viewlet) {
        if (viewlet.endsWith(FineReportConstants.SUFFIX_REPORT)) {
            return FineReportConstants.URI_REPORT;
        }
        if (viewlet.endsWith(FineReportConstants.SUFFIX_LARGE_SCREEN)) {
            return FineReportConstants.URI_LARGE_SCREEN;
        }
        if (viewlet.endsWith(FineReportConstants.SUFFIX_FORM)) {
            return FineReportConstants.URI_FORM;
        }

        return FineReportConstants.URI_REPORT;
    }

    private String generateToken(SystemProperties.FineReport props) {
        var accountType = props.getAccountType();
        Assert.notNull(accountType, "账号类型为空");

        // 获取账号
        var user = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        String account = null;
        if (accountType == AccountType.USERNAME) {
            account = user.getUsername();
        } else if (accountType == AccountType.EMAIL) {
            account = user.getUser().getEmail();
        } else if (accountType == AccountType.MOBILE) {
            account = user.getUser().getMobile();
        } else {
            throw new BusinessException("暂不支持的账号类型：" + accountType);
        }
        if (CharSequenceUtil.isBlank(account)) {
            throw new BusinessException("用户的" + accountType.getType() + "为空");
        }
        var publicKey = props.getPublicKey();
        if (CharSequenceUtil.isBlank(publicKey)) {
            return account;
        }

        // 加密
        RSA rsa = new RSA(null, publicKey);
        String plainTxt = null;
        if (Boolean.TRUE.equals(props.getTokenWithIssueTime())) {
            plainTxt = "{\"username\":\"" + account + "\",\"issueTime\":" + new Date().getTime() + "}";
        } else {
            plainTxt = account;
        }

        return rsa.encryptBase64(plainTxt, KeyType.PublicKey);
    }
}
