package com.elitescloud.cloudt.system.modules.message.repository;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.cloudt.system.constant.MsgSendStateEnum;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.modules.message.entity.QSysMsgSendRecordDO;
import com.elitescloud.cloudt.system.modules.message.entity.QSysMsgSendRecordDtlDO;
import com.elitescloud.cloudt.system.modules.message.entity.SysMsgSendRecordDtlDO;
import com.elitescloud.cloudt.system.param.SysMsgSentResultDTO;
import com.elitescloud.cloudt.system.param.SysMsgViewResultDTO;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.Projections;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/6/6
 */
@Repository
public class SysMsgSendRecordDtlRepoProc extends BaseRepoProc<SysMsgSendRecordDtlDO> {
    private static final QSysMsgSendRecordDtlDO QDO = QSysMsgSendRecordDtlDO.sysMsgSendRecordDtlDO;
    private static final QSysMsgSendRecordDO QDO_RECORD = QSysMsgSendRecordDO.sysMsgSendRecordDO;

    public SysMsgSendRecordDtlRepoProc() {
        super(QDO);
    }

    /**
     * 修改已读标识
     *
     * @param ids      记录ID
     * @param readFlag 已读状态
     */
    public void updateViewed(@NotEmpty Collection<Long> ids, boolean readFlag) {
        jpaQueryFactory.update(QDO)
                .set(QDO.readFlg, readFlag)
                .set(QDO.readTime, LocalDateTime.now().format(DatetimeUtil.FORMATTER_DATETIME))
                .where(QDO.id.in(ids))
                .execute();
    }

    /**
     * 修改发送结果
     *
     * @param batchId
     * @param sentEndMessage
     * @param sendState
     * @param sentErrMessage
     */
    public void updateSendResult(@NotBlank String batchId, @NotBlank String sentEndMessage, String sendState, String sentErrMessage) {
        jpaQueryFactory.update(QDO)
                .set(QDO.sendState, sendState)
                .set(QDO.sentErrMessage, sentErrMessage)
                .set(QDO.sentTimeEnd, LocalDateTime.now())
                .where(QDO.batchUuid.eq(batchId).and(QDO.sentEndMessage.eq(sentEndMessage)))
                .execute();
    }

    /**
     * 统计待阅的总数
     *
     * @param receiverUserCode 接收人账号
     * @return 数量
     */
    public long countToView(String receiverUserCode) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.recipientUserCode, receiverUserCode)
                .andEq(QDO.sentTypeCode, MsgSendTypeEnum.SYS_INTERIOR.name())
                .andEq(QDO.sendState, MsgSendStateEnum.OK.name())
                .andEq(QDO.readFlg, false)
                .build();
        return super.count(predicate);
    }

    /**
     * 查询发送结果
     *
     * @param batchId  发送批ID
     * @param sentType 发送类型
     * @return 发送结果
     */
    public List<SysMsgViewResultDTO> queryViewResult(@NotBlank String batchId, @NotNull MsgSendTypeEnum sentType) {
        return jpaQueryFactory.select(QDO.recipientUserId, QDO.recipientUserName, QDO.readFlg, QDO.readTime)
                .from(QDO)
                .where(QDO.batchUuid.eq(batchId).and(QDO.sentTypeCode.eq(sentType.name())).and(QDO.sendState.eq(MsgSendStateEnum.OK.name())))
                .fetch()
                .stream()
                .map(t -> {
                    SysMsgViewResultDTO dto = new SysMsgViewResultDTO();
                    if (StringUtils.hasText(t.get(QDO.recipientUserId))) {
                        dto.setUserId(Long.parseLong(t.get(QDO.recipientUserId)));
                    }
                    dto.setAccount(t.get(QDO.recipientUserName));
                    dto.setViewed(Boolean.TRUE.equals(t.get(QDO.readFlg)));
                    if (StringUtils.hasText(t.get(QDO.readTime))) {
                        dto.setViewTime(LocalDateTime.parse(t.get(QDO.readTime), DatetimeUtil.FORMATTER_DATETIME));
                    }

                    return dto;
                }).collect(Collectors.toList());
    }

    /**
     * 查询发送结果
     *
     * @param batchIds 发送批ID
     * @param sentType 发送类型
     * @return 发送结果
     */
    public List<SysMsgSentResultDTO> querySentResult(@NotEmpty Collection<String> batchIds, @NotNull MsgSendTypeEnum sentType) {
        return jpaQueryFactory.select(QDO.batchUuid, QDO.recipientUserId, QDO.recipientUserName, QDO.readFlg)
                .from(QDO)
                .where(QDO.batchUuid.in(batchIds).and(QDO.sentTypeCode.eq(sentType.name())).and(QDO.sendState.eq(MsgSendStateEnum.OK.name())))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.batchUuid)))
                .entrySet()
                .stream()
                .map(t -> {
                    SysMsgSentResultDTO sentResultDTO = new SysMsgSentResultDTO();
                    sentResultDTO.setRecordId(t.getKey());

                    List<String> viewedUserIds = new ArrayList<>();
                    List<String> viewedUserNames = new ArrayList<>();
                    List<String> toViewUserIds = new ArrayList<>();
                    List<String> toViewUserNames = new ArrayList<>();

                    for (Tuple tuple : t.getValue()) {
                        if (Boolean.TRUE.equals(tuple.get(QDO.readFlg))) {
                            viewedUserIds.add(tuple.get(QDO.recipientUserId));
                            viewedUserNames.add(tuple.get(QDO.recipientUserName));
                            continue;
                        }
                        toViewUserIds.add(tuple.get(QDO.recipientUserId));
                        toViewUserNames.add(tuple.get(QDO.recipientUserName));
                    }
                    sentResultDTO.setViewedUserIds(viewedUserIds);
                    sentResultDTO.setViewedUserAccounts(viewedUserNames);
                    sentResultDTO.setToViewUserIds(toViewUserIds);
                    sentResultDTO.setToViewUserAccounts(toViewUserNames);

                    return sentResultDTO;
                }).collect(Collectors.toList());
    }

    /**
     * 查询回执编码及接收人
     *
     * @param ids 记录ID
     * @return 回执编码及接收人
     */
    public Map<String, String> querySentEndMessage(Collection<Long> ids) {
        return jpaQueryFactory.select(QDO.sentEndMessage, QDO.recipientUserCode)
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.sentEndMessage), t -> t.get(QDO.recipientUserCode), (t1, t2) -> t1));
    }

    /**
     * 根据批次查询
     *
     * @param batchId
     * @return
     */
    public List<SysMsgSendRecordDtlDO> queryByBatchId(@NotBlank String batchId) {
        return super.getListByValue(QDO.batchUuid, batchId);
    }

    /**
     * 根据批次查询
     *
     * @param batchIds
     * @return
     */
    public List<SysMsgSendRecordDtlDO> queryByBatchId(@NotEmpty Collection<String> batchIds) {
        return super.getListByValue(QDO.batchUuid, batchIds);
    }
}
