package com.elitescloud.cloudt.system.modules.message.rpc;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.UUID;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.provider.TenantOrgDataIsolateProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.messenger.model.MessageAccountVO;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.constant.MsgTypeEnum;
import com.elitescloud.cloudt.system.dto.req.msg.MsgRecipientUserDTO;
import com.elitescloud.cloudt.system.dto.req.msg.template.TemplateAssignSendParamDTO;
import com.elitescloud.cloudt.system.dto.req.msg.template.TemplateEmployeeSendParamDTO;
import com.elitescloud.cloudt.system.dto.req.msg.template.TemplateTxtReplaceDTO;
import com.elitescloud.cloudt.system.dto.resp.SysMsgSentResultDTO;
import com.elitescloud.cloudt.system.model.vo.sbean.EmployeePagedRespBean;
import com.elitescloud.cloudt.system.modules.message.bo.SendTempLateMsgBO;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgTemplateConfigRepository;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgTemplateRepository;
import com.elitescloud.cloudt.system.modules.message.service.SysMsgSendRecordService;
import com.elitescloud.cloudt.system.modules.message.service.SysMsgTemplateService;
import com.elitescloud.cloudt.system.modules.message.service.impl.SysMsgSendCommonService;
import com.elitescloud.cloudt.system.modules.message.service.impl.SysMsgSendCommonServiceAbstract;
import com.elitescloud.cloudt.system.modules.message.rpc.manager.SysMsgTemplateSendRpcServiceServiceAbstract;
import com.elitescloud.cloudt.system.modules.message.vo.respose.SysMsgTemplateConfigVO;
import com.elitescloud.cloudt.system.param.SysSendResultParam;
import com.elitescloud.cloudt.system.service.SysMsgTemplateSendRpcService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.core.task.TaskExecutor;
import org.springframework.http.MediaType;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * &#064;description  : 消息服务消息发送接口
 * 自定义消息复用模板消息的代码和逻辑，有部分入参会转换成模板消息的对象。
 * 公共部分参见：SysMsgSendCommonService ，SysMsgSendCommonServiceAbstract
 * &#064;date  : 2023/5/25 11:01
 */
@RestController
@RequestMapping(value = SysMsgTemplateSendRpcService.URI, produces = MediaType.APPLICATION_JSON_VALUE)
@TenantOrgTransaction(useTenantOrg = false)
@AllArgsConstructor
@Slf4j
public class SysMsgTemplateSendRpcServiceImpl
        extends SysMsgTemplateSendRpcServiceServiceAbstract
        implements SysMsgTemplateSendRpcService {
    private final SysMsgTemplateRepository sysMsgTemplateRepository;
    private final SysMsgTemplateConfigRepository sysMsgTemplateConfigRepository;
    private final SysMsgSendCommonService sysMsgSendCommonService;
    private final SysMsgTemplateService msgTemplateService;
    private final TaskExecutor taskExecutor;
    private final TenantOrgDataIsolateProvider tenantOrgDataIsolateProvider;
    private final SysMsgSendRecordService msgRecordService;

    @Override
    public ApiResult<String> msgTemplateEmployeeSend(TemplateEmployeeSendParamDTO param) {
        // 批次ID 用于批量操作
        String uuidBatch = UUID.randomUUID().toString();
        this.compatibleTemplateParams(param);
        // 发送人 检查
        var sendUser = SysMsgSendCommonServiceAbstract.autoGetSender(param.getSendUser());
        param.setSendUser(sendUser);
        sysMsgSendCommonService.checkSendUser(sendUser);

        // 接收人集合
        List<EmployeePagedRespBean> recipientUserEmployeeList = this.obtainReceivers(param);

        // 查询抄送人和抄送人员工信息
        var carbonUserList = param.getCarbonUserList();
        List<EmployeePagedRespBean> carbonUserEmployeeList = new ArrayList<>();
        if (carbonUserList != null && !carbonUserList.isEmpty()) {
            carbonUserEmployeeList = sysMsgSendCommonService.getCheckEmployeePagedRespBeans(carbonUserList);
        }

        //查询模板配置，替换模板占位，保存主表明细表，rpc发送消息
        SendTempLateMsgBO sendBo=new SendTempLateMsgBO(param.getContentReplaceMap(),
                param.getTitleReplaceMap(),
                param.getMesJoinParamMap(),
                uuidBatch,
                sendUser,
                recipientUserEmployeeList,
                carbonUserEmployeeList,
                param.getFileCodes(),
                param.getTemplateCode());
        sendBo.setReceiptUserId(param.getReceiptUserId());
        sendTemplateRpcMsg(sendBo);
        return ApiResult.ok(uuidBatch);
    }

    @Override
    public ApiResult<List<SysSendResultParam>> msgTemplateEmployeeSend(List<TemplateEmployeeSendParamDTO> paramList) {
        List<SysSendResultParam> resultParamList = new ArrayList<>(paramList.size());
        for (var paramDTO : paramList) {
            try {
                var sendResult = this.msgTemplateEmployeeSend(paramDTO);
                resultParamList.add(new SysSendResultParam(sendResult.isSuccess(), sendResult.getMsg(), sendResult.getData()));
            } catch (Throwable e) {
                log.error("发送失败：", e);
                resultParamList.add(new SysSendResultParam(false, e.getMessage(), null));
            }
        }
        return ApiResult.ok(resultParamList);
    }

    @Override
    public ApiResult<String> msgTemplateMsgTypeParamSend(TemplateAssignSendParamDTO param) {
        //批次ID 用于批量操作
        String uuidBatch = UUID.randomUUID().toString();
        this.compatibleTemplateParams(param);
        //发送人 检查
        val sendUser = SysMsgSendCommonServiceAbstract.autoGetSender(param.getSendUser());
        param.setSendUser(sendUser);
        sysMsgSendCommonService.checkSendUser(sendUser);

        // 接收人集合
        List<EmployeePagedRespBean> recipientEmployeeList =
                param.getRecipientUserList().stream().map(templateAssignRecipientUserDTO -> {
                    var map = templateAssignRecipientUserDTO.getRecipientUserAccountDtoMap();
                    //根据接收人参数，发送类型的参数，封装入员工信息，后续逻辑可以和员工逻辑复用
                    return sysMsgSendCommonService.setEmployeeBeanByAssignRecipientUserDTO(map, templateAssignRecipientUserDTO);
                }).filter(ObjectUtil::isNotNull).collect(Collectors.toList());

        //查询抄送人和抄送人员工信息
        val carbonUserList = param.getContactUserList();
        List<EmployeePagedRespBean> carbonUserEmployeeList = null;
        if (carbonUserList != null && !carbonUserList.isEmpty()) {
            carbonUserEmployeeList =
                    carbonUserList.stream().map(templateAssignRecipientUserDTO -> {
                        var map = templateAssignRecipientUserDTO.getRecipientUserAccountDtoMap();
                        //根据接收人参数，发送类型的参数，封装入员工信息，后续逻辑可以和员工逻辑复用
                        return sysMsgSendCommonService.setEmployeeBeanByAssignRecipientUserDTO(map, templateAssignRecipientUserDTO);
                    }).filter(ObjectUtil::isNotNull).collect(Collectors.toList());
        }


        //模板编码 查询模板信息 检查
        val templateCode = param.getTemplateCode();
        sysMsgSendCommonService.checkTemplateCode(templateCode);


        //查询模板配置，替换模板占位，保存主表明细表，rpc发送消息查询发送状态的发送渠道类型
        SendTempLateMsgBO sendBo=new SendTempLateMsgBO(param.getContentReplaceMap(),
                param.getTitleReplaceMap(),
                param.getMesJoinParamMap(),
                uuidBatch,
                sendUser,
                recipientEmployeeList,
                carbonUserEmployeeList,
                param.getFileCodes(),
                templateCode);
        sendBo.setReceiptAccount(param.getReceiptAccount());
        sendTemplateRpcMsg(sendBo);
        return ApiResult.ok(uuidBatch);
    }

    @Override
    public ApiResult<List<SysSendResultParam>> msgTemplateMsgTypeParamSend(List<TemplateAssignSendParamDTO> paramList) {
        List<SysSendResultParam> resultParamList = new ArrayList<>(paramList.size());
        for (var paramDTO : paramList) {
            try {
                var sendResult = this.msgTemplateMsgTypeParamSend(paramDTO);
                resultParamList.add(new SysSendResultParam(sendResult.isSuccess(), sendResult.getMsg(), sendResult.getData()));
            } catch (Throwable e) {
                log.error("发送失败：", e);
                resultParamList.add(new SysSendResultParam(false, e.getMessage(), null));
            }
        }
        return ApiResult.ok(resultParamList);
    }

    @Override
    public ApiResult<String> msgTemplateSend(TemplateEmployeeSendParamDTO param) {
        //批次ID 用于批量操作
        String uuidBatch = UUID.randomUUID().toString();
        this.compatibleTemplateParams(param);
        //发送人 检查
        val sendUser = SysMsgSendCommonServiceAbstract.autoGetSender(param.getSendUser());
        param.setSendUser(sendUser);
        sysMsgSendCommonService.checkSendUser(sendUser);
        //接收人集合 检查
        val recipientUserList = param.getRecipientUserList();
        //接收人员工信息封装
        List<EmployeePagedRespBean> recipientEmployeeList = recipientUserList==null?new ArrayList<>():
                recipientUserList.stream().map(msgRecipientUserDTO -> {
                    EmployeePagedRespBean employeePagedRespBean=new EmployeePagedRespBean();
                    employeePagedRespBean.setUsername(msgRecipientUserDTO.getUserName());
                    return employeePagedRespBean;
                }).collect(Collectors.toList());

        //查询抄送人和抄送人员工信息
        val carbonUserList = param.getCarbonUserList();
        List<EmployeePagedRespBean> carbonUserEmployeeList = carbonUserList==null?new ArrayList<>():
                recipientUserList.stream().map(msgRecipientUserDTO -> {
                    EmployeePagedRespBean employeePagedRespBean=new EmployeePagedRespBean();
                    employeePagedRespBean.setUsername(msgRecipientUserDTO.getUserName());
                    return employeePagedRespBean;
                }).collect(Collectors.toList());

        //查询模板配置，替换模板占位，保存主表明细表，rpc发送消息查询发送状态的发送渠道类型
        SendTempLateMsgBO sendBo=new SendTempLateMsgBO( param.getContentReplaceMap(),
                param.getTitleReplaceMap(),
                param.getMesJoinParamMap(),
                uuidBatch,
                sendUser,
                recipientEmployeeList,
                carbonUserEmployeeList,
                param.getFileCodes(),
                param.getTemplateCode());
        sendBo.setReceiptUserId(param.getReceiptUserId());
        sendTemplateRpcMsg(sendBo);
        return ApiResult.ok(uuidBatch);
    }

    @Override
    public ApiResult<SysMsgSentResultDTO> getSentResult(String recordId) {
        return msgRecordService.querySentResult(recordId);
    }

    @Override
    public ApiResult<List<SysMsgSentResultDTO>> listSentResult(Set<String> recordIds) {
        return msgRecordService.querySentResults(recordIds);
    }

    @Override
    public ApiResult<String> retrySend(String recordId) {
        return msgRecordService.retrySend(recordId);
    }

    private void compatibleTemplateParams(TemplateEmployeeSendParamDTO sendParamDTO) {
        if (CollUtil.isEmpty(sendParamDTO.getTitleReplaceMap())) {
            // 取值之前的
            if (CollUtil.isNotEmpty(sendParamDTO.getMsgTemplateTitleReplaceDTOS())) {
                var replaceParams = sendParamDTO.getMsgTemplateTitleReplaceDTOS().stream()
                        .filter(t -> CollUtil.isNotEmpty(t.getReplaceMap()))
                        .findFirst()
                        .map(TemplateTxtReplaceDTO::getReplaceMap);
                sendParamDTO.setTitleReplaceMap(replaceParams.orElse(Collections.emptyMap()));
            }
        }
        if (CollUtil.isEmpty(sendParamDTO.getContentReplaceMap())) {
            // 取值之前的
            if (CollUtil.isNotEmpty(sendParamDTO.getMsgTemplateContentReplaceDTOS())) {
                var replaceParams = sendParamDTO.getMsgTemplateContentReplaceDTOS().stream()
                        .filter(t -> CollUtil.isNotEmpty(t.getReplaceMap()))
                        .findFirst()
                        .map(TemplateTxtReplaceDTO::getReplaceMap);
                sendParamDTO.setContentReplaceMap(replaceParams.orElse(Collections.emptyMap()));
            }
        }
    }

    @SuppressWarnings({"deprecation", "removal"})
    private void compatibleTemplateParams(TemplateAssignSendParamDTO sendParamDTO) {
        if (CollUtil.isEmpty(sendParamDTO.getTitleReplaceMap())) {
            // 取值之前的
            if (CollUtil.isNotEmpty(sendParamDTO.getMsgTemplateTitleReplaceDTOS())) {
                var replaceParams = sendParamDTO.getMsgTemplateTitleReplaceDTOS().stream()
                        .filter(t -> CollUtil.isNotEmpty(t.getReplaceMap()))
                        .findFirst()
                        .map(TemplateTxtReplaceDTO::getReplaceMap);
                sendParamDTO.setTitleReplaceMap(replaceParams.orElse(Collections.emptyMap()));
            }
        }
        if (CollUtil.isEmpty(sendParamDTO.getContentReplaceMap())) {
            // 取值之前的
            if (CollUtil.isNotEmpty(sendParamDTO.getMsgTemplateContentReplaceDTOS())) {
                var replaceParams = sendParamDTO.getMsgTemplateContentReplaceDTOS().stream()
                        .filter(t -> CollUtil.isNotEmpty(t.getReplaceMap()))
                        .findFirst()
                        .map(TemplateTxtReplaceDTO::getReplaceMap);
                sendParamDTO.setContentReplaceMap(replaceParams.orElse(Collections.emptyMap()));
            }
        }
    }

    /***查询模板配置，替换模板占位，保存主表明细表，rpc发送消息**/
    private void sendTemplateRpcMsg(SendTempLateMsgBO sendBo) {
        // 获取模板信息
        sysMsgSendCommonService.checkTemplateCode(sendBo.getTemplateCode());
        val templateDo = tenantOrgDataIsolateProvider.byNoTenantOrg(() -> sysMsgTemplateRepository.findByTemplateCode(sendBo.getTemplateCode()));
        Assert.notNull(templateDo, "消息服务，消息模板没有找到：" + sendBo.getTemplateCode());
        sendBo.setTemplateDo(templateDo);

        //获取模板配置
        val sysMsgTemplateConfigDTOList =
                sysMsgTemplateConfigRepository.findAllByTemplateIdAndSendTypeSwitch(sendBo.getTemplateDo().getId(), true);
        //消息模板明细配置，按照消息发送类型分组，并do转vo；
        Map<MsgSendTypeEnum, SysMsgTemplateConfigVO> groupedSendMap =
                sysMsgSendCommonService.sysMsgTemplateConfigDoToVoGroupBySendTypeCode(sysMsgTemplateConfigDTOList);
        sendBo.setGroupedSendMap(groupedSendMap);

        // 默认接收人
        var defaultReceiverMap = this.queryDefaultReceivers(sendBo.getTemplateCode());
        sendBo.setDefaultReceiverMap(defaultReceiverMap);

        // 事务添加消息主表和明细表数据，保障基础数据记录，根据接收人添加主表，根据模板发送类型配置，依次添加明细表。
        Map<MsgSendTypeEnum, List<MessageAccountVO>> sendPathMap =
                sysMsgSendCommonService.saveSendRecordToMsgSendTypeEnumListMap(sendBo, MsgTypeEnum.TEMPLATE, true);

        /**
         * 批量发送消息，根据发送类型批量发送
         * 获取发送类型对应的发送数据，sendPathMap获取，根据模板的内容和标题发送msgTemplateTxtReplaceDTOS
         * 发送完一个类型uuidBatch+sendType批量更新明细的发送状态
         * **/
        CompletableFuture.runAsync(() -> groupedSendMap.forEach((sendTypeEnum, msgTemplateTxtReplaceDTOS) -> {
                    sysMsgSendCommonService.sendBatchMsg(sendBo.getUuidBatch(), sendPathMap.get(sendTypeEnum), sendTypeEnum,
                            msgTemplateTxtReplaceDTOS, sendBo.getContentReplaceMap(), sendBo.getCarbonUserEmployeeList(), sendBo.getFileCodes(),
                            sendBo.getReceiptAccount());
                }
        ), taskExecutor).whenComplete((v, e) -> {
            if (e != null) {
                log.error("更新消息状态异常：", e);
            }
        });
    }

    private List<EmployeePagedRespBean> obtainReceivers(TemplateEmployeeSendParamDTO param) {
        if (CollUtil.isEmpty(param.getRecipientUserList())) {
            return Collections.emptyList();
        }

        var userIds = param.getRecipientUserList().stream()
                .map(MsgRecipientUserDTO::getUserId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        if (userIds.isEmpty()) {
            return Collections.emptyList();
        }

        // 查询员工信息
        return sysMsgSendCommonService.getCheckEmployeePagedRespBeans(userIds);
    }

    private Map<MsgSendTypeEnum, List<EmployeePagedRespBean>> queryDefaultReceivers(String templateCode) {
        // 查询接收人ID
        var defaultReceiverMap = msgTemplateService.queryDefaultReceivers(templateCode).computeData();
        if (defaultReceiverMap.isEmpty()) {
            return Collections.emptyMap();
        }

        // 查询员工信息
        var userIds = defaultReceiverMap.values().stream().flatMap(Collection::stream).filter(Objects::nonNull).collect(Collectors.toSet());
        if (userIds.isEmpty()) {
            return Collections.emptyMap();
        }
        var employeeMap = sysMsgSendCommonService.getCheckEmployeePagedRespBeans(userIds).stream()
                .collect(Collectors.toMap(EmployeePagedRespBean::getUserId, t -> t, (t1, t2) -> t1));

        Map<MsgSendTypeEnum, List<EmployeePagedRespBean>> result = new HashMap<>(defaultReceiverMap.size());
        for (Map.Entry<MsgSendTypeEnum, Set<Long>> entry : defaultReceiverMap.entrySet()) {
            var employeeList = entry.getValue().stream().map(employeeMap::get).filter(Objects::nonNull).collect(Collectors.toList());
            result.put(entry.getKey(), employeeList);
        }
        return result;
    }
}
