package com.elitescloud.cloudt.system.modules.message.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.zhxu.bs.BeanSearcher;
import cn.zhxu.bs.FieldOps;
import cn.zhxu.bs.util.MapUtils;
import com.elitescloud.boot.datasecurity.dpr.beansearcher.BeanSearcherFactory;
import com.elitescloud.boot.datasecurity.dpr.beansearcher.CloudBeanSearcherEnum;
import com.elitescloud.cloudt.messenger.model.MessageAccountVO;
import com.elitescloud.cloudt.system.constant.MsgSendStateEnum;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.constant.MsgTypeEnum;
import com.elitescloud.cloudt.system.constant.SysMsgReceiverTypeEnum;
import com.elitescloud.cloudt.system.dto.req.msg.MsgRecipientUserDTO;
import com.elitescloud.cloudt.system.model.vo.sbean.EmployeePagedRespBean;
import com.elitescloud.cloudt.system.modules.message.bo.SendTempLateMsgBO;
import com.elitescloud.cloudt.system.modules.message.entity.SysMsgSendRecordDtlDO;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgSendRecordDtlRepository;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgSendRecordRepository;
import com.elitescloud.cloudt.system.modules.message.vo.respose.SysMsgTemplateConfigVO;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description :
 * @date : 2023/6/2 09:24
 */
@Slf4j
@Service
public class SysMsgSendCommonService extends SysMsgSendCommonServiceAbstract {


    private final SysMsgSendRecordDtlRepository sysMsgSendRecordDtlRepository;
    private final SysMsgSendRecordRepository sysMsgSendRecordRepository;
    private final BeanSearcherFactory beanSearcherFactory;
    private final BeanSearcher beanSearcher;

    private final PlatformTransactionManager transactionManager;
    //消息发送
    private final SysMsgSendService sysMsgSendService;
    @Autowired
    private EmployeeRepoProc employeeRepoProc;

    protected SysMsgSendCommonService(
            SysMsgSendRecordDtlRepository sysMsgSendRecordDtlRepository,
            SysMsgSendRecordRepository sysMsgSendRecordRepository,
            BeanSearcherFactory beanSearcherFactory,
            PlatformTransactionManager transactionManager, SysMsgSendService sysMsgSendService) {
        this.sysMsgSendRecordDtlRepository = sysMsgSendRecordDtlRepository;
        this.sysMsgSendRecordRepository = sysMsgSendRecordRepository;
        this.beanSearcherFactory = beanSearcherFactory;
        this.beanSearcher = beanSearcherFactory.getBeanSearcherService(CloudBeanSearcherEnum.BS_TENANT);
        this.transactionManager = transactionManager;
        this.sysMsgSendService = sysMsgSendService;
    }

    /**
     * 根据用户接收人参数，查询对应的员工信息，发号对应的员工信息，并检查员工信息是不是找到了。打印没有找到的数据。
     *
     * @param recipientUserList
     * @return
     */
    public List<EmployeePagedRespBean> getCheckEmployeePagedRespBeans(List<MsgRecipientUserDTO> recipientUserList) {
        List<Long> userIds = recipientUserList.stream().map(MsgRecipientUserDTO::getUserId).collect(Collectors.toList());
        var wereSqlMapBuilder = MapUtils.builder().field(EmployeePagedRespBean::getUserId, userIds).op(FieldOps.InList);
        List<EmployeePagedRespBean> employeeList =
                beanSearcher.searchAll(EmployeePagedRespBean.class,
                        wereSqlMapBuilder.build());

        //检查参数的员工id和找到的员工差 日志提示
        checkFindMismatchedEmployeeIds(userIds, employeeList);
        return employeeList;
    }

    /**
     * 查询员工信息
     *
     * @param userIds
     * @return
     */
    public List<EmployeePagedRespBean> getCheckEmployeePagedRespBeans(Set<Long> userIds) {
        var wereSqlMapBuilder = MapUtils.builder().field(EmployeePagedRespBean::getUserId, userIds).op(FieldOps.InList);
        List<EmployeePagedRespBean> employeeList =
                beanSearcher.searchAll(EmployeePagedRespBean.class,
                        wereSqlMapBuilder.build());
        //检查参数的员工id和找到的员工差 日志提示
        checkFindMismatchedEmployeeIds(userIds, employeeList);
        return employeeList;
    }

    /**
     * 批量发送
     * sendTypeEnum 发送类型
     * msgTemplateTxtReplaceDTO 模板发送配置
     * templateParams 模板参数
     * sendPathMap 接收人
     * carbonUserEmployeeList 抄送人
     * fileCodes 附件编码
     **/
    public String sendBatchMsg(String uuidBatch, List<MessageAccountVO> sendPathMap,
                               MsgSendTypeEnum sendTypeEnum,
                               SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO,
                               Map<String, String> templateParams,
                               List<EmployeePagedRespBean> carbonUserEmployeeList,
                               List<String> fileCodes,
                               String receiptAccount) {

        switch (sendTypeEnum) {
            case EMAIL:
                return sysMsgSendService.sendBatchEmailMsg(uuidBatch, sendPathMap, msgTemplateTxtReplaceDTO,
                        carbonUserEmployeeList, fileCodes, receiptAccount);
            case MOBILE_SMS:
                return sysMsgSendService.sendBatchSmsMsg(uuidBatch, sendPathMap, msgTemplateTxtReplaceDTO, templateParams);
            case SYS_INTERIOR:
                return sysMsgSendService.sendBatchSiteMsg(uuidBatch, sendPathMap, msgTemplateTxtReplaceDTO);
            case APP:
                return sysMsgSendService.sendBatchAppMsg(uuidBatch, sendPathMap, msgTemplateTxtReplaceDTO);
            case WX_BOOT:
                return sysMsgSendService.sendBatchWxBootMsg(uuidBatch, msgTemplateTxtReplaceDTO);
            default:
                log.error("暂不支持的消息渠道：{}", sendTypeEnum);
                return "暂不支持";
        }
    }

    /***向发送map集合中清洗出来发送类型的发送路径对象list，用于后续批量发送信息 如果是非法信息，登记发送明细状态错误**/
    private void addSendPathMapByEmployeeSendType(Map<MsgSendTypeEnum, List<MessageAccountVO>> sendPathMap,
                                                  EmployeePagedRespBean msgRecipientUserDTO,
                                                  MsgSendTypeEnum sendTypeEnum, SysMsgSendRecordDtlDO sysMsgSendRecordDtlDO,
                                                  boolean isReceiver) {
        switch (sendTypeEnum) {
            case EMAIL:
                if (StringUtils.hasText(msgRecipientUserDTO.getEmail())) {
                    if (isReceiver) {
                        sendPathMap.computeIfAbsent(MsgSendTypeEnum.EMAIL, k -> new ArrayList<>())
                                .add(new MessageAccountVO(msgRecipientUserDTO.getEmail(), msgRecipientUserDTO.getLastName()));
                    }
                    sysMsgSendRecordDtlDO.setRecipientAccount(msgRecipientUserDTO.getEmail());
                } else {
                    sysMsgSendRecordDtlDO.setSendState(MsgSendStateEnum.FAILED.name());
                    sysMsgSendRecordDtlDO.setSentErrMessage("员工邮箱为空，跳过发送");
                }
                break;
            case MOBILE_SMS:
                if (StringUtils.hasText(msgRecipientUserDTO.getPhone())) {
                    sendPathMap.computeIfAbsent(MsgSendTypeEnum.MOBILE_SMS, k -> new ArrayList<>())
                            .add(new MessageAccountVO(msgRecipientUserDTO.getPhone(), msgRecipientUserDTO.getLastName()));
                    sysMsgSendRecordDtlDO.setRecipientAccount(msgRecipientUserDTO.getPhone());
                } else {
                    sysMsgSendRecordDtlDO.setSendState(MsgSendStateEnum.FAILED.name());
                    sysMsgSendRecordDtlDO.setSentErrMessage("员工手机号为空，跳过发送");
                }
                break;
            case WX_BOOT:
                sendPathMap.computeIfAbsent(MsgSendTypeEnum.WX_BOOT, t -> new ArrayList<>())
                        .add(new MessageAccountVO(msgRecipientUserDTO.getUsername()));
                sysMsgSendRecordDtlDO.setRecipientAccount(msgRecipientUserDTO.getUsername());
                sysMsgSendRecordDtlDO.setSendState(MsgSendStateEnum.OK.name());
                sysMsgSendRecordDtlDO.setSentTimeEnd(LocalDateTime.now());
                break;
            case SYS_INTERIOR:
                sendPathMap.computeIfAbsent(MsgSendTypeEnum.SYS_INTERIOR, t -> new ArrayList<>())
                        .add(new MessageAccountVO(msgRecipientUserDTO.getUsername()));
                sysMsgSendRecordDtlDO.setRecipientAccount(msgRecipientUserDTO.getUsername());
                sysMsgSendRecordDtlDO.setSendState(MsgSendStateEnum.OK.name());
                sysMsgSendRecordDtlDO.setSentTimeEnd(LocalDateTime.now());
                break;
            case APP:
                sendPathMap.computeIfAbsent(MsgSendTypeEnum.APP, t -> new ArrayList<>())
                        .add(new MessageAccountVO(msgRecipientUserDTO.getUsername()));
                sysMsgSendRecordDtlDO.setRecipientAccount(msgRecipientUserDTO.getUsername());
                sysMsgSendRecordDtlDO.setSendState(MsgSendStateEnum.OK.name());
                sysMsgSendRecordDtlDO.setSentTimeEnd(LocalDateTime.now());
                break;
            case MINI_PROGRAM:
            case OFFICIAL_ACCOUNT:

            default:
                // 处理其他类型或默认情况
                log.error("暂未实现的消息类型：{}", sendTypeEnum.name());
        }
    }

    /**
     * 消息发送记录保存同事返回按消息发送类型分组的发送参数对象
     * 1.对模板占位符数据替换，将替换结果。回设置到groupedSendMap 发送参数集合中
     * 2.发送前记录，事务添加消息主表和明细表数据，  根据接收人添加主表，根据模板发送类型配置，依次添加明细表。
     * 3.返回sendPathMap集合，清洗出来发送类型的发送路径对象list，用于后续批量发送信息 如果是非法信息，登记发送明细状态错误
     *
     * @param sendBo      消息发送业务对象
     * @param msgTypeEnum 模板发送/自定义
     * @return 向发送map集合中清洗出来发送类型的发送路径对象list
     */
    public Map<MsgSendTypeEnum, List<MessageAccountVO>> saveSendRecordToMsgSendTypeEnumListMap(SendTempLateMsgBO sendBo,
                                                                                               MsgTypeEnum msgTypeEnum,
                                                                                               boolean save) {
        if (msgTypeEnum.name().equals(MsgTypeEnum.TEMPLATE.name())) {
            // content占位替换配置 分组便于查找 , 占位符替换 根据入参msgTemplateTxtMap 更新groupedSendMap里面数据
            msgTemplateSetMesText(sendBo.getTemplateDo().getId(), sendBo.getContentReplaceMap(), sendBo.getGroupedSendMap());

            // title占位替换配置 分组便于查找 , 占位符替换 根据入参msgTemplateTxtMap 更新groupedSendMap里面数据
            msgTemplateSetMesTitle(sendBo.getTemplateDo().getId(), sendBo.getTitleReplaceMap(), sendBo.getGroupedSendMap());
        }
        // 回执账号
        if (sendBo.getReceiptUserId() != null) {
            sendBo.setReceiptAccount(employeeRepoProc.getEmailByUserId(sendBo.getReceiptUserId()));
        }

        TransactionTemplate transactionTemplate = new TransactionTemplate(transactionManager);
        return transactionTemplate.execute(status -> {
            try {
                String mesJoinParamJson = getMesJoinParamJson(sendBo);
                log.info("发送模板消息：{}，接收人数量：{}，默认接收人数量：{}，消息渠道数量：{}", sendBo.getTemplateDo().getTemplateCode(),
                        sendBo.getRecipientEmployeeList().size(), sendBo.getDefaultReceiverMap().size(), sendBo.getGroupedSendMap().size());
                // 执行事务操作
                //发送地址集合，可能是邮箱，可能是手机号
                EnumMap<MsgSendTypeEnum, List<MessageAccountVO>> sendPathMap = new EnumMap<>(MsgSendTypeEnum.class);
                //循环接收人，每个接收人都是一个主发送记录 发送前首先存储发送记录。状态发送中
                sendBo.getRecipientEmployeeList().forEach(msgRecipientUserDTO -> {
                    var sysMsgSendRecordDO = newSysMsgSendRecordDO(msgTypeEnum, msgRecipientUserDTO, sendBo);
                    sysMsgSendRecordDO.setCustomParamJson(mesJoinParamJson);
                    // sysMsgSendRecordDO.setSendParamJson(toJsonString(sendBo));
                    if (save) {
                        sysMsgSendRecordRepository.save(sysMsgSendRecordDO);
                    }

                    //批量循环添加明细 根据发送渠道
                    sendBo.getGroupedSendMap().forEach((sendTypeEnum, msgTemplateTxtReplaceDTO) -> {
                        var sysMsgSendRecordDtlDO = newSysMsgSendRecordDtlDO(sendBo, sysMsgSendRecordDO, sendTypeEnum, msgTemplateTxtReplaceDTO);
                        sysMsgSendRecordDtlDO.setCustomParamJson(mesJoinParamJson);
                        //根据员工发送 记录员工记录
                        sysMsgSendRecordDtlDO.setTemplateSendParamJson(toJsonString(msgRecipientUserDTO));
                        sysMsgSendRecordDtlDO.setExtend1(CollectionUtils.isEmpty(sendBo.getFileCodes()) ? null : toJsonString(sendBo.getFileCodes()));
                        sysMsgSendRecordDtlDO.setExtend2(CollectionUtils.isEmpty(sendBo.getCarbonUserEmployeeList()) ? null
                                : toJsonString(sendBo.getCarbonUserEmployeeList().stream().map(employeePagedRespBean -> {
                            return " UserId=" + employeePagedRespBean.getUserId() +
                                    " Username=" + employeePagedRespBean.getUsername() +
                                    " Phone=" + employeePagedRespBean.getPhone() +
                                    " Email=" + employeePagedRespBean.getEmail();
                        }).collect(Collectors.toList())));
                        //向发送map集合中清洗出来发送类型的发送路径对象list，用于后续批量发送信息 如果是非法信息，登记发送明细状态错误
                        addSendPathMapByEmployeeSendType(sendPathMap, msgRecipientUserDTO, sendTypeEnum, sysMsgSendRecordDtlDO, true);
                        sysMsgSendRecordDtlDO.setRecipientAccountType(SysMsgReceiverTypeEnum.RECEIVER.name());
                        sysMsgSendRecordDtlDO.setReceiptUserId(sendBo.getReceiptUserId() == null ? null : sendBo.getReceiptUserId().toString());
                        sysMsgSendRecordDtlDO.setReceiptAccount(sendBo.getReceiptAccount());
                        if (save) {
                            sysMsgSendRecordDtlRepository.save(sysMsgSendRecordDtlDO);
                        }
                    });
                });

                //  邮件抄送人
                if (CollUtil.isNotEmpty(sendBo.getCarbonUserEmployeeList()) && sendBo.getGroupedSendMap().containsKey(MsgSendTypeEnum.EMAIL)) {
                    sendBo.getCarbonUserEmployeeList().forEach(msgRecipientUserDTO -> {

                        var sysMsgSendRecordDO = newSysMsgSendRecordDO(msgTypeEnum, msgRecipientUserDTO, sendBo);
                        sysMsgSendRecordDO.setCustomParamJson(mesJoinParamJson);
                        if (save) {
                            sysMsgSendRecordRepository.save(sysMsgSendRecordDO);
                        }

                        // 批量循环添加明细 根据发送渠道
                        sendBo.getGroupedSendMap().forEach((sendTypeEnum, msgTemplateTxtReplaceDTO) -> {
                            if (sendTypeEnum != MsgSendTypeEnum.EMAIL) {
                                // 目前仅邮件支持抄送
                                return;
                            }
                            var sysMsgSendRecordDtlDO = newSysMsgSendRecordDtlDO(sendBo, sysMsgSendRecordDO, sendTypeEnum, msgTemplateTxtReplaceDTO);
                            sysMsgSendRecordDtlDO.setCustomParamJson(mesJoinParamJson);
                            //根据员工发送 记录员工记录
                            sysMsgSendRecordDtlDO.setTemplateSendParamJson(toJsonString(msgRecipientUserDTO));
                            sysMsgSendRecordDtlDO.setExtend1(CollectionUtils.isEmpty(sendBo.getFileCodes()) ? null : toJsonString(sendBo.getFileCodes()));
                            sysMsgSendRecordDtlDO.setExtend2(CollectionUtils.isEmpty(sendBo.getCarbonUserEmployeeList()) ? null
                                    : toJsonString(sendBo.getCarbonUserEmployeeList().stream().map(employeePagedRespBean -> {
                                return " UserId=" + employeePagedRespBean.getUserId() +
                                        " Username=" + employeePagedRespBean.getUsername() +
                                        " Phone=" + employeePagedRespBean.getPhone() +
                                        " Email=" + employeePagedRespBean.getEmail();
                            }).collect(Collectors.toList())));
                            //向发送map集合中清洗出来发送类型的发送路径对象list，用于后续批量发送信息 如果是非法信息，登记发送明细状态错误
                            addSendPathMapByEmployeeSendType(sendPathMap, msgRecipientUserDTO, sendTypeEnum, sysMsgSendRecordDtlDO, false);
                            sysMsgSendRecordDtlDO.setRecipientAccountType(SysMsgReceiverTypeEnum.CARBON_COPY.name());
                            if (save) {
                                sysMsgSendRecordDtlRepository.save(sysMsgSendRecordDtlDO);
                            }
                        });
                    });
                }

                // 默认接收人
                if (CollUtil.isNotEmpty(sendBo.getDefaultReceiverMap())) {
                    sendBo.getGroupedSendMap().forEach((sendTypeEnum, msgTemplateTxtReplaceDTO) -> {
                        var defaultReceiverList = sendBo.getDefaultReceiverMap().get(sendTypeEnum);
                        if (defaultReceiverList.isEmpty()) {
                            return;
                        }

                        for (EmployeePagedRespBean msgRecipientUserDTO : defaultReceiverList) {
                            var sysMsgSendRecordDO = newSysMsgSendRecordDO(msgTypeEnum, msgRecipientUserDTO, sendBo);
                            sysMsgSendRecordDO.setCustomParamJson(mesJoinParamJson);
                            if (save) {
                                sysMsgSendRecordRepository.save(sysMsgSendRecordDO);
                            }

                            var sysMsgSendRecordDtlDO = newSysMsgSendRecordDtlDO(sendBo, sysMsgSendRecordDO, sendTypeEnum, msgTemplateTxtReplaceDTO);
                            sysMsgSendRecordDtlDO.setCustomParamJson(mesJoinParamJson);
                            //根据员工发送 记录员工记录
                            sysMsgSendRecordDtlDO.setTemplateSendParamJson(toJsonString(msgRecipientUserDTO));
                            sysMsgSendRecordDtlDO.setExtend1(CollectionUtils.isEmpty(sendBo.getFileCodes()) ? null : toJsonString(sendBo.getFileCodes()));
                            sysMsgSendRecordDtlDO.setRecipientAccountType(SysMsgReceiverTypeEnum.DEFAULT.name());

                            //向发送map集合中清洗出来发送类型的发送路径对象list，用于后续批量发送信息 如果是非法信息，登记发送明细状态错误
                            addSendPathMapByEmployeeSendType(sendPathMap, msgRecipientUserDTO, sendTypeEnum, sysMsgSendRecordDtlDO, true);
                            if (save) {
                                sysMsgSendRecordDtlRepository.save(sysMsgSendRecordDtlDO);
                            }
                        }
                    });
                }

                // 手动提交事务
                status.flush();
                return sendPathMap;
            } catch (Exception e) {
                // 发生异常，手动回滚事务
                status.setRollbackOnly();
                log.error("保存消息记录明细异常：", e);
                throw e;
            }
        });

    }

    @Nullable
    private String getMesJoinParamJson(SendTempLateMsgBO sendBo) {
        //自定义参数转换JSON
        String mesJoinParamJson = toJsonString(sendBo.getMesJoinParamMap());

        if (mesJoinParamJson != null) {
            if (mesJoinParamJson.length() > 2000) {
                throw new RuntimeException("消息服务程序异常：自定义参数过长，超过2000.");
            }
        }
        return mesJoinParamJson;
    }


}
