package com.elitescloud.cloudt.system.modules.message.service.impl;


import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.zhxu.bs.BeanSearcher;
import cn.zhxu.bs.FieldOps;
import cn.zhxu.bs.SearchResult;
import cn.zhxu.bs.param.OrderBy;
import cn.zhxu.bs.util.MapUtils;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.datasecurity.dpr.beansearcher.BeanSearcherFactory;
import com.elitescloud.boot.datasecurity.dpr.beansearcher.CloudBeanSearcherEnum;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.messenger.Messenger;
import com.elitescloud.cloudt.messenger.common.MsgReceiveStatusEnum;
import com.elitescloud.cloudt.messenger.common.MsgSendStatusEnum;
import com.elitescloud.cloudt.messenger.model.dto.MsgResultDTO;
import com.elitescloud.cloudt.system.constant.MsgSendStateEnum;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.modules.message.entity.SysMsgSendRecordDtlDO;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgSendRecordDtlRepoProc;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgSendRecordDtlRepository;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgSendRecordRepository;
import com.elitescloud.cloudt.system.modules.message.sbean.SysMsgSendRecordDtlQueryBean;
import com.elitescloud.cloudt.system.modules.message.sbean.SysMsgSendRecordQueryBean;
import com.elitescloud.cloudt.system.modules.message.service.SysMsgSendRecordService;
import com.elitescloud.cloudt.system.modules.message.vo.request.SysMsgSendRecordDeleteVO;
import com.elitescloud.cloudt.system.modules.message.vo.request.SysMsgSendRecordDtlInteriorQueryVO;
import com.elitescloud.cloudt.system.modules.message.vo.request.SysMsgSendRecordQueryVO;
import com.elitescloud.cloudt.system.param.SysMsgSentResultDTO;
import com.elitescloud.cloudt.system.param.SysMsgViewResultDTO;
import com.elitescloud.cloudt.system.service.util.BeanSearcherUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.PostMapping;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description : 消息模板接口
 * @date : 2023/5/23 13:44
 */
@Service
@Slf4j
public class SysMsgSendRecordServiceImpl implements SysMsgSendRecordService {

    private final SysMsgSendRecordRepository sysMsgSendRecordRepository;
    private final SysMsgSendRecordDtlRepository sysMsgSendRecordDtlRepository;
    private final SysMsgSendRecordDtlRepoProc recordDtlRepoProc;
    private final BeanSearcher beanSearcher;
    private final BeanSearcherFactory beanSearcherFactory;
    private final TaskExecutor taskExecutor;

    @Autowired
    public SysMsgSendRecordServiceImpl(SysMsgSendRecordRepository sysMsgSendRecordRepository,
                                       SysMsgSendRecordDtlRepository sysMsgSendRecordDtlRepository,
                                       BeanSearcherFactory beanSearcherFactory,
                                       SysMsgSendRecordDtlRepoProc recordDtlRepoProc,
                                       TaskExecutor taskExecutor) {
        this.sysMsgSendRecordRepository = sysMsgSendRecordRepository;
        this.sysMsgSendRecordDtlRepository = sysMsgSendRecordDtlRepository;
        this.recordDtlRepoProc = recordDtlRepoProc;
        this.taskExecutor = taskExecutor;
        this.beanSearcherFactory = beanSearcherFactory;
        this.beanSearcher = this.beanSearcherFactory.getBeanSearcherService(CloudBeanSearcherEnum.BS_TENANT);
    }


    @Override
    public ApiResult<SearchResult<SysMsgSendRecordQueryBean>> sysMsgSendRecordQuery(
            SysMsgSendRecordQueryVO qvo) {
        var mapBuilder = MapUtils.builder()
                .page(qvo.getCurrent(), qvo.getSize())
                .field(SysMsgSendRecordQueryBean::getRecipientUserId, qvo.getRecipientUserId()).op(FieldOps.Equal)
                .field(SysMsgSendRecordQueryBean::getRecipientUserName, qvo.getRecipientUserName()).op(FieldOps.Equal)
                .field(SysMsgSendRecordQueryBean::getSendUserId, qvo.getSendUserId()).op(FieldOps.Equal)
                .field(SysMsgSendRecordQueryBean::getSendUserName, qvo.getSendUserName()).op(FieldOps.Equal)
                .field(SysMsgSendRecordQueryBean::getMsgType, qvo.getMsgTypeCode(), qvo.getSendUserName()).op(FieldOps.Equal)
                .field(SysMsgSendRecordQueryBean::getTemplateCode, qvo.getTemplateCode()).op(FieldOps.Equal)
                .field(SysMsgSendRecordQueryBean::getTemplateName, qvo.getTemplateName()).op(FieldOps.Equal)
                .field(SysMsgSendRecordQueryBean::getSendTime, qvo.getSendTimeStart(), qvo.getSendTimeEnd()).op(FieldOps.Between);
        //动态排序
        BeanSearcherUtils.setParamsOrderBy(qvo.getOrders(), mapBuilder, new OrderBy("sendTime", OrderBy.ORDER_ASC));
        var result = beanSearcher.search(SysMsgSendRecordQueryBean.class, mapBuilder.build());

        return ApiResult.ok(result);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Boolean> sysMsgSendRecordDelete(SysMsgSendRecordDeleteVO sysMsgSendRecordDeleteVO) {
        var recordIds = sysMsgSendRecordDeleteVO.getSysMsgSendRecordIdList();
        if (CollUtil.isNotEmpty(recordIds)) {
            sysMsgSendRecordRepository.deleteAllByIdInBatch(recordIds);
            recordIds.forEach(sysMsgSendRecordDtlRepository::deleteByMsgSendRecordId);
        }

        var dtlIds = sysMsgSendRecordDeleteVO.getRecordDtlIdList();
        if (CollUtil.isNotEmpty(dtlIds)) {
            sysMsgSendRecordDtlRepository.deleteAllById(dtlIds);
        }

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<List<SysMsgSentResultDTO>> querySentResult(Set<String> batchUuids) {
        Assert.notEmpty(batchUuids, "发送记录ID为空");

        var result = recordDtlRepoProc.querySentResult(batchUuids, MsgSendTypeEnum.SYS_INTERIOR);
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<SysMsgViewResultDTO>> getViewedResult(String batchUuid) {
        Assert.hasText(batchUuid, "发送记录ID为空");

        var resultList = recordDtlRepoProc.queryViewResult(batchUuid, MsgSendTypeEnum.SYS_INTERIOR);
        for (SysMsgViewResultDTO sysMsgViewResultDTO : resultList) {
            if (sysMsgViewResultDTO.getViewed() == null) {
                sysMsgViewResultDTO.setViewed(false);
            }
        }
        return ApiResult.ok(resultList);
    }

    @Override
    public ApiResult<com.elitescloud.cloudt.system.dto.resp.SysMsgSentResultDTO> querySentResult(String batchUuid) {
        if (CharSequenceUtil.isBlank(batchUuid)) {
            return ApiResult.fail("消息批次ID为空");
        }

        var recordDtlList = recordDtlRepoProc.queryByBatchId(batchUuid);
        if (recordDtlList.isEmpty()) {
            return ApiResult.fail("消息不存在");
        }

        var resultDTO = this.convertSentResult(recordDtlList).get(0);
        return ApiResult.ok(resultDTO);
    }

    @Override
    public ApiResult<List<com.elitescloud.cloudt.system.dto.resp.SysMsgSentResultDTO>> querySentResults(Set<String> batchUuidList) {
        if (CollUtil.isEmpty(batchUuidList)) {
            return ApiResult.fail("消息批次ID为空");
        }

        var recordDtlList = recordDtlRepoProc.queryByBatchId(batchUuidList);
        if (recordDtlList.isEmpty()) {
            return ApiResult.fail("消息不存在");
        }

        var resultDTO = this.convertSentResult(recordDtlList);
        return ApiResult.ok(resultDTO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> retrySend(String batchUuid) {
        if (CharSequenceUtil.isBlank(batchUuid)) {
            return ApiResult.fail("消息批次ID为空");
        }

        var recordDtlList = recordDtlRepoProc.queryByBatchId(batchUuid);
        if (recordDtlList.isEmpty()) {
            return ApiResult.fail("消息不存在");
        }

        var messageIds = recordDtlList.stream().filter(t -> MsgSendStateEnum.FAILED.name().equals(t.getSendState()) && CharSequenceUtil.isNotBlank(t.getSentEndMessage()))
                .map(SysMsgSendRecordDtlDO::getSentEndMessage).collect(Collectors.toSet());
        if (messageIds.isEmpty()) {
            return ApiResult.ok(batchUuid);
        }

        for (String messageId : messageIds) {
            try {
                Messenger.retrySend(messageId, true);
            } catch (Exception e) {
                log.error("重试消息异常：{}", messageId, e);
            }
        }

        // 查询消息状态
        List<MsgResultDTO> msgResultList = null;
        try {
            msgResultList = Messenger.querySendResult(messageIds);
        } catch (Exception e) {
            log.error("查询消息结果异常", e);
        }
        // 更新状态
        if (CollUtil.isNotEmpty(msgResultList)) {
            for (MsgResultDTO msgResultDTO : msgResultList) {
                String state;
                if (MsgSendStatusEnum.SENDING.name().equals(msgResultDTO.getStatusName())) {
                    state = MsgSendStateEnum.SENDING.name();
                } else if (MsgSendStatusEnum.SENT_FAIL.name().equals(msgResultDTO.getStatusName())) {
                    state = MsgSendStateEnum.FAILED.name();
                } else {
                    state = MsgSendStateEnum.OK.name();
                }
                recordDtlRepoProc.updateSendResult(batchUuid, msgResultDTO.getMessageId(), state, msgResultDTO.getFailMsg());
            }
        }
        return ApiResult.ok(batchUuid);
    }

    @Override
    public ApiResult<List<SysMsgSendRecordDtlQueryBean>> sysMsgSendRecordDtlByRecordId(Long sendRecordId) {
        var mapBuilder = MapUtils.builder()
                .field(SysMsgSendRecordDtlQueryBean::getMsgSendRecordId, sendRecordId).op(FieldOps.Equal)
                .orderBy(SysMsgSendRecordDtlQueryBean::getSentTimeStart).desc();

        var result = beanSearcher.search(SysMsgSendRecordDtlQueryBean.class, mapBuilder.build());
        return ApiResult.ok(result.getDataList());
    }

    @Override
    public ApiResult<SearchResult<SysMsgSendRecordDtlQueryBean>> sysMsgSendRecordDtlInteriorQuery(
            SysMsgSendRecordDtlInteriorQueryVO qvo) {
        var mapBuilder = MapUtils.builder()
                .selectExclude(SysMsgSendRecordDtlQueryBean::getMessageContent,
                        SysMsgSendRecordDtlQueryBean::getSentErrMessage)
                .page(qvo.getCurrent(), qvo.getSize())
                .field(SysMsgSendRecordDtlQueryBean::getRecipientUserId, qvo.getRecipientUserId()).op(FieldOps.Equal)
                .field(SysMsgSendRecordDtlQueryBean::getRecipientUserName, qvo.getRecipientUserName()).op(FieldOps.Equal)
                .field(SysMsgSendRecordDtlQueryBean::getRecipientUserCode, qvo.getRecipientUserCode()).op(FieldOps.Equal)
                .field(SysMsgSendRecordDtlQueryBean::getSentTypeCode, qvo.getMsgSendTypeEnum().name()).op(FieldOps.Equal)
                .field(SysMsgSendRecordDtlQueryBean::getSendState, qvo.getSendState()).op(FieldOps.Equal)
                .field(SysMsgSendRecordDtlQueryBean::getReadFlg, qvo.getReadFlag()).op(FieldOps.Equal)
                .field(SysMsgSendRecordDtlQueryBean::getTitleContent, qvo.getTitle()).op(FieldOps.Contain)
                .field(SysMsgSendRecordDtlQueryBean::getMessageContent, qvo.getContent()).op(FieldOps.Contain)
                .field(SysMsgSendRecordDtlQueryBean::getSentTimeStart, qvo.getSentTimeStart(),
                        qvo.getSentTimeStartEnd()).op(FieldOps.Between)
                //.field(SysMsgSendRecordDtlQueryBean::getSentEndMessage).op(FieldOps.NotEmpty)
                ;
        //动态排序
        BeanSearcherUtils.setParamsOrderBy(qvo.getOrders(), mapBuilder, new OrderBy("sentTimeStart", OrderBy.ORDER_ASC));
        var result = beanSearcher.search(SysMsgSendRecordDtlQueryBean.class, mapBuilder.build());
        if (CollUtil.isNotEmpty(result.getDataList())) {
            for (SysMsgSendRecordDtlQueryBean data : result.getDataList()) {
                if (StringUtils.hasText(data.getCustomParamJson())) {
                    data.setCustomParam(JSONUtil.json2Obj(data.getCustomParamJson(), new TypeReference<Map<String, Object>>() {
                    }));
                }
            }
        }
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<SysMsgSendRecordDtlQueryBean> sysMsgSendRecordDtlInteriorByDtlIdQuery(Long dtlId, Long userId) {
        var mapBuilder = MapUtils.builder()
                .field(SysMsgSendRecordDtlQueryBean::getId, dtlId).op(FieldOps.Equal);
        var result = beanSearcher.searchFirst(SysMsgSendRecordDtlQueryBean.class, mapBuilder.build());
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<Long> countToView() {
        var user = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        var num = recordDtlRepoProc.countToView(user.getUsername());
        return ApiResult.ok(num);
    }

    @Override
    public ApiResult<SysMsgSendRecordDtlQueryBean> getRecordDetail(String msgId) {
        Assert.hasText(msgId, "消息ID为空");
        var currentUser = SecurityContextUtil.currentUserName();
        var mapBuilder = MapUtils.builder()
                .field(SysMsgSendRecordDtlQueryBean::getSentEndMessage, msgId).op(FieldOps.Equal)
                .field(SysMsgSendRecordDtlQueryBean::getRecipientUserCode, currentUser).op(FieldOps.Equal);
        var result = beanSearcher.searchFirst(SysMsgSendRecordDtlQueryBean.class, mapBuilder.build());
        return ApiResult.ok(result);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> interiorReadUpdate(Long id, Boolean readFlg) {
        sysMsgSendRecordDtlRepository.findById(id).ifPresentOrElse(sysMsgSendRecordDtlDO -> {
                    sysMsgSendRecordDtlDO.setReadFlg(readFlg);
                    // 获取当前时间字符串，yyyy-MM-dd HH:mm:ss
                    sysMsgSendRecordDtlDO.setReadTime(DateUtil.now());
                    sysMsgSendRecordDtlRepository.save(sysMsgSendRecordDtlDO);

                    if (StringUtils.hasText(sysMsgSendRecordDtlDO.getSentEndMessage())) {
                        CompletableFuture.runAsync(() -> Messenger.updateReceiveStatus(sysMsgSendRecordDtlDO.getSentEndMessage(),
                                        sysMsgSendRecordDtlDO.getRecipientUserCode(), MsgReceiveStatusEnum.VIEWED), taskExecutor)
                                .whenComplete((v, e) -> {
                                    if (e != null) {
                                        log.error("更新消息状态异常：", e);
                                    }
                                });
                    }
                },
                () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateViewed(Set<Long> ids, Boolean viewed) {
        Assert.notEmpty(ids, "记录ID为空");
        viewed = viewed == null || viewed;

        recordDtlRepoProc.updateViewed(ids, viewed);

        var endMessageMap = recordDtlRepoProc.querySentEndMessage(ids);
        CompletableFuture.runAsync(() -> {
            for (Map.Entry<String, String> entry : endMessageMap.entrySet()) {
                Messenger.updateReceiveStatus(entry.getKey(),
                        entry.getValue(), MsgReceiveStatusEnum.VIEWED);
            }
        }, taskExecutor).whenComplete((v, e) -> {
            if (e != null) {
                log.error("更新消息状态异常：", e);
            }
        });

        return ApiResult.ok(true);
    }

    private List<com.elitescloud.cloudt.system.dto.resp.SysMsgSentResultDTO> convertSentResult(List<SysMsgSendRecordDtlDO> recordDtlBatchList) {
        // 消息服务状态查询
        var messageIds = recordDtlBatchList.stream().map(SysMsgSendRecordDtlDO::getSentEndMessage).filter(CharSequenceUtil::isNotBlank).collect(Collectors.toSet());
        List<MsgResultDTO> messageResultList = messageIds.isEmpty() ? Collections.emptyList() : Messenger.querySendResult(messageIds);
        Map<String, MsgResultDTO> messageResultMap = CollUtil.isEmpty(messageResultList) ? Collections.emptyMap()
                : messageResultList.stream().collect(Collectors.toMap(MsgResultDTO::getMessageId, Function.identity(), (t1, t2) -> t1));

        // 转换结果
        var typeRecordDtlMap = recordDtlBatchList.stream().collect(Collectors.groupingBy(SysMsgSendRecordDtlDO::getBatchUuid,
                Collectors.collectingAndThen(Collectors.toList(), t -> t.stream().collect(Collectors.groupingBy(SysMsgSendRecordDtlDO::getSentTypeCode)))));
        var resultDtoList = new ArrayList<com.elitescloud.cloudt.system.dto.resp.SysMsgSentResultDTO>();
        com.elitescloud.cloudt.system.dto.resp.SysMsgSentResultDTO resultDTO = null;
        for (Map.Entry<String, Map<String, List<SysMsgSendRecordDtlDO>>> typeEntry : typeRecordDtlMap.entrySet()) {
            var batchId = typeEntry.getKey();
            resultDTO = new com.elitescloud.cloudt.system.dto.resp.SysMsgSentResultDTO();
            var recordDtlMap = typeEntry.getValue();
            resultDTO.setRecordId(batchId);
            resultDTO.setResultMap(new HashMap<>(8));
            for (Map.Entry<String, List<SysMsgSendRecordDtlDO>> entry : recordDtlMap.entrySet()) {
                var sentTypeResult = new com.elitescloud.cloudt.system.dto.resp.SysMsgSentResultDTO.SentTypeResult();
                var recordDtl = entry.getValue().get(0);

                // 是否全部成功
                var successNum = entry.getValue().stream().filter(t -> MsgSendStateEnum.OK.name().equals(t.getSendState())).count();
                var isAllSucceed = successNum == entry.getValue().size();

                sentTypeResult.setSentTime(recordDtl.getSentTimeStart());
                sentTypeResult.setFinishTime(recordDtl.getSentTimeEnd());
                sentTypeResult.setSendState(MsgSendStateEnum.parse(recordDtl.getSendState()));
                sentTypeResult.setAllSucceed(isAllSucceed);
                sentTypeResult.setFailMsg(recordDtl.getSentErrMessage());
                sentTypeResult.setReceiverList(entry.getValue().stream().map(t -> {
                    var msgUser = new com.elitescloud.cloudt.system.dto.resp.SysMsgSentResultDTO.MsgUser();
                    msgUser.setUserId(t.getRecipientUserId());
                    msgUser.setUsername(t.getRecipientUserCode());
                    msgUser.setFullName(t.getRecipientUserName());
                    msgUser.setSendState(MsgSendStateEnum.parse(t.getSendState()));
                    msgUser.setAccount(t.getRecipientAccount());
                    msgUser.setAccountType(t.getRecipientAccountType());

                    return msgUser;
                }).collect(Collectors.toList()));

                // 判断消息服务状态
                if (CharSequenceUtil.isNotBlank(recordDtl.getSentEndMessage())) {
                    var messageResult = messageResultMap.get(recordDtl.getSentEndMessage());
                    if (MsgSendStatusEnum.SENT_FAIL.name().equals(messageResult.getStatusName())) {
                        sentTypeResult.setFailMsg(messageResult.getFailMsg());
                        sentTypeResult.setSendState(MsgSendStateEnum.FAILED);
                    } else if (MsgSendStatusEnum.SENT.name().equals(messageResult.getStatusName())) {
                        sentTypeResult.setFailMsg("发送成功");
                        sentTypeResult.setSendState(MsgSendStateEnum.OK);
                    } else if (MsgSendStatusEnum.SENDING.name().equals(messageResult.getStatusName())) {
                        sentTypeResult.setFailMsg("发送中");
                        sentTypeResult.setSendState(MsgSendStateEnum.SENDING);
                    }
                }

                resultDTO.getResultMap().put(MsgSendTypeEnum.parse(entry.getKey()), sentTypeResult);
            }

            resultDtoList.add(resultDTO);
        }

        return resultDtoList;
    }
}
