package com.elitescloud.cloudt.system.modules.message.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.cloudt.messenger.Messenger;
import com.elitescloud.cloudt.messenger.message.AbstractMessageSenderVO;
import com.elitescloud.cloudt.messenger.message.SmsMessageVO;
import com.elitescloud.cloudt.messenger.model.FileMessageAttachmentVO;
import com.elitescloud.cloudt.messenger.model.MessageAccountVO;
import com.elitescloud.cloudt.messenger.model.MessageAttachmentVO;
import com.elitescloud.cloudt.system.constant.MsgSendStateEnum;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.model.vo.sbean.EmployeePagedRespBean;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgSendRecordDtlRepository;
import com.elitescloud.cloudt.system.modules.message.vo.respose.SysMsgTemplateConfigVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description : 消息发送实现方法
 * @date : 2023/10/30 14:27
 */
@Service
@Slf4j
public class SysMsgSendService extends SysMsgSendCommonServiceAbstract {


    private final SysMsgSendRecordDtlRepository sysMsgSendRecordDtlRepository;
    private final RestTemplate restTemplate = RestTemplateFactory.instance();

    public SysMsgSendService(SysMsgSendRecordDtlRepository sysMsgSendRecordDtlRepository) {
        this.sysMsgSendRecordDtlRepository = sysMsgSendRecordDtlRepository;

    }

    /**
     * 批量发送微信BOOT消息
     **/
    public String sendBatchWxBootMsg(String uuidBatch, SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO) {
        String MSG_TYPE = "markdown";
        String wxBootUrl = msgTemplateTxtReplaceDTO.getExternalTemplateId();
        String title = msgTemplateTxtReplaceDTO.getMesTitle();
        String mesText = msgTemplateTxtReplaceDTO.getMesText();
        String content = "# " + title + " \n " + mesText;
        Map<String, Object> body = new HashMap<>();
        body.put("msgtype", MSG_TYPE);
        body.put(MSG_TYPE, Map.of("content", content));
        log.info("【企业微信】发送预警：{}, {}", wxBootUrl, content);

        MsgSendStateEnum state = MsgSendStateEnum.FAILED;
        String msg = "发送失败";
        String receipt = null;
        try {
            ResponseEntity<String> resp = restTemplate.exchange(wxBootUrl, HttpMethod.POST, new HttpEntity<>(body), String.class);
            receipt = resp.getBody();
            if (resp.getStatusCode().is2xxSuccessful()) {
                state = MsgSendStateEnum.OK;
                msg = null;
            }
        } catch (RestClientException e) {
            log.error("发送企业微信机器人消息异常：", e);
            msg = "发送异常：" + e.getMessage();
        }

        sysMsgSendRecordDtlRepository.updateByCondition(receipt, msg, state.name(), LocalDateTime.now(), uuidBatch, MsgSendTypeEnum.WX_BOOT.name());
        return msg;
    }

    /**
     * 批量发送站内信
     **/
    public String sendBatchSiteMsg(String uuidBatch,
                                   List<MessageAccountVO> accountList,
                                   SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO) {

        if (CollectionUtils.isEmpty(accountList)) {
            log.error("消息服务：没有一个接收人的账号，站内信消息将不会发送！");
            return "接收人或登录账号为空";
        }

        // 构建消息
        var message = Messenger.siteMessage()
                .setReceiverList(accountList)
                .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                .setContent(msgTemplateTxtReplaceDTO.getMesText())
                .setBusinessType(CharSequenceUtil.blankToDefault(msgTemplateTxtReplaceDTO.getBusinessType(), "default"))
                .setBusinessParams(msgTemplateTxtReplaceDTO.getBusinessData())
                .build();
        // 开始发送
        return this.executeSend(message, MsgSendTypeEnum.SYS_INTERIOR, uuidBatch);
    }

    /**
     * 批量发送App消息
     **/
    public String sendBatchAppMsg(String uuidBatch, List<MessageAccountVO> accountList,
                                  SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO) {
        if (CollectionUtils.isEmpty(accountList)) {
            log.error("消息服务：没有一个接收人的账号，App消息将不会发送！");
            return "接收人或接收人的登录账号为空";
        }

        // 构建消息
        var message = Messenger.appMessage()
                .setReceiverList(accountList)
                .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                .setContent(msgTemplateTxtReplaceDTO.getMesText())
                .setBusinessType(CharSequenceUtil.blankToDefault(msgTemplateTxtReplaceDTO.getBusinessType(), "default"))
                .setBusinessParams(msgTemplateTxtReplaceDTO.getBusinessData())
                .build();
        // 开始发送消息
        return this.executeSend(message, MsgSendTypeEnum.APP, uuidBatch);
    }

    /**
     * 批量发送短信
     **/
    public String sendBatchSmsMsg(String uuidBatch, List<MessageAccountVO> sendEmailList,
                                  SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO, Map<String, String> templateParams) {
        if (CollectionUtils.isEmpty(sendEmailList)) {
            log.error("消息服务：没有一个接收人的手机号，短信消息将不会发送！");
            return "接收人或接收人的手机号为空";
        }

        // 构建短信
        SmsMessageVO messageVO = null;
        if (StringUtils.hasText(msgTemplateTxtReplaceDTO.getExternalTemplateId())) {
            messageVO = Messenger.templateSms()
                    .setTemplateCode(msgTemplateTxtReplaceDTO.getExternalTemplateId())
                    .setSignName(CharSequenceUtil.blankToDefault(msgTemplateTxtReplaceDTO.getExternalSignName(), null))
                    .setTemplateParam(templateParams)
                    .setReceiverList(sendEmailList)
                    .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                    .setContent(msgTemplateTxtReplaceDTO.getMesText())
                    .build();
        } else {
            messageVO = Messenger.sms()
                    .setReceiverList(sendEmailList)
                    .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                    .setContent(msgTemplateTxtReplaceDTO.getMesText())
                    .setSignName(CharSequenceUtil.blankToDefault(msgTemplateTxtReplaceDTO.getExternalSignName(), null))
                    .build();
        }

        // 发送短信
        return this.executeSend(messageVO, MsgSendTypeEnum.MOBILE_SMS, uuidBatch);
    }

    /**
     * 批量发送邮件
     * uuidBatch批次号
     * sendEmailList 接收人
     * carbonUserEmployeeList 抄送人
     * fileCodes 附件编码
     **/
    public String sendBatchEmailMsg(String uuidBatch, List<MessageAccountVO> receiverList,
                                    SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO,
                                    List<EmployeePagedRespBean> carbonUserEmployeeList,
                                    List<String> fileCodes,
                                    String receiptAccount) {
        if (CollectionUtils.isEmpty(receiverList)) {
            log.error("消息服务：没有一个邮件接收人的邮箱，邮件消息将不会发送！");
            return "接收人或接收人的邮箱为空";
        }
        // 抄送人
        List<MessageAccountVO> ccList = CollUtil.isEmpty(carbonUserEmployeeList) ? Collections.emptyList() :
                carbonUserEmployeeList.stream().filter(t -> StringUtils.hasText(t.getEmail()))
                        .map(user -> {
                                    MessageAccountVO messageAccountVO = new MessageAccountVO();
                                    messageAccountVO.setAccount(user.getEmail());
                                    messageAccountVO.setNickName(user.getLastName());
                                    return messageAccountVO;
                                }
                        ).collect(Collectors.toList());
        // 附件
        List<MessageAttachmentVO> messageAttachmentVOS = CollUtil.isEmpty(fileCodes) ? Collections.emptyList() :
                fileCodes.stream().map(s -> {
                    FileMessageAttachmentVO fileMessageAttachmentVO = new FileMessageAttachmentVO();
                    fileMessageAttachmentVO.setFileCode(s);
                    return fileMessageAttachmentVO;
                }).collect(Collectors.toList());
        // 构建消息
        var message = Messenger.email()
                .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                .setContent(msgTemplateTxtReplaceDTO.getMesText())
                .setReceiverList(receiverList)
                .setCcList(ccList)
                .setAttachments(messageAttachmentVOS)
                .setRetryableOnFailure(true)
                .setReceiptAccount(receiptAccount)
                .build();

        // 发送消息
        return this.executeSend(message, MsgSendTypeEnum.EMAIL, uuidBatch);
    }

    private String executeSend(@NotNull AbstractMessageSenderVO messageSenderVO, @NotNull MsgSendTypeEnum sendTypeEnum,
                               String uuidBatch) {
        try {
            // 发送消息
            String messageId = messageSenderVO.sendSync();

            // 更新消息状态
            var status = MsgSendStateEnum.OK;
            var failMsg = "发送成功";
            if (CharSequenceUtil.isNotBlank(messageId)) {
                var sendResult = Messenger.querySendResult(messageId);
                if (MsgSendStateEnum.FAILED.name().equals(sendResult.getStatusName())) {
                    status = MsgSendStateEnum.FAILED;
                    failMsg = sendResult.getFailMsg();
                }
            }
            sysMsgSendRecordDtlRepository.updateByCondition(messageId, failMsg, status.name(),
                    LocalDateTime.now(), uuidBatch, sendTypeEnum.name());
            return status == MsgSendStateEnum.FAILED ? failMsg : null;
        } catch (Exception e) {
            //更新失败状态
            log.error("发送{}异常：{}", sendTypeEnum.getDescription(), uuidBatch, e);
            sysMsgSendRecordDtlRepository.updateByCondition(null, e.getMessage(), MsgSendStateEnum.FAILED.name(),
                    LocalDateTime.now(), uuidBatch, sendTypeEnum.name());
            return "发送异常：" + e.getMessage();
        }
    }
}
