package com.elitescloud.cloudt.system.provider.imports;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.extra.pinyin.PinyinUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.cloudt.constant.AreaType;
import com.elitescloud.cloudt.system.provider.imports.param.ImportAreaBO;
import com.elitescloud.cloudt.system.service.AreaMngService;
import com.elitescloud.cloudt.system.service.AreaQueryService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotBlank;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

/**
 * 省市县区域导入.
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/11
 */
@Component
public class AreaImportProvider implements DataImport<ImportAreaBO> {
    private static final String COUNTRY_CHINA_CODE = "CN";

    @Autowired
    private AreaMngService mngService;
    @Autowired
    private AreaQueryService queryService;

    @Override
    public String getTmplCode() {
        return "system_area_import";
    }

    @Override
    public List<String> executeImport(List<ImportAreaBO> dataList, int startRowIndex) {
        List<String> errorMsg = new ArrayList<>(dataList.size());
        int i = 0;

        Set<String> existsCode = new HashSet<>();
        for (ImportAreaBO importAreaBO : dataList) {
            try {
                importAreaBO = this.fillAreaInfo(importAreaBO, t -> existsCode.contains(t) || Boolean.TRUE.equals(queryService.existsCode(t).getData()));
            } catch (Exception e) {
                i++;
                errorMsg.add(e.getMessage());
                continue;
            }
            importAreaBO.setSortNo(startRowIndex + (i++));

            existsCode.add(importAreaBO.getCode());

            try {
                mngService.save(importAreaBO);
            } catch (Exception e) {
                errorMsg.add(e.getMessage());
            }
        }

        return errorMsg;
    }

    /**
     * 地区信息解析
     * <p>
     * 根据国家行政区域编码解析，数据来源：<a href="https://www.mca.gov.cn/mzsj/xzqh/2022/202201xzqh.html">2022年中华人民共和国县以上行政区划代码</a>，<p>
     * 代码从左到右的含义：
     * <li>第一、二位：表示省级（含省、自治区、直辖市、特别行政区）</li>
     * <li>第三、四位：表示地级（含省辖行政区、地级市、自治州、地区、盟及中央直辖市所属市辖区和县的汇总码和省或自治区直辖县级行政区划汇总码）。其中，01-20，51-70表示省直辖市；21-50表示地区（自治州、盟）。90表示省或自治区直辖县级行政区划汇总码</li>
     * <li>第五、六位：表示县级（市辖区、县级市、旗）。01-20表示市辖区或地区（自治州、盟）辖县级市；21-70表示县（旗）；81-99表示省辖县级市；71-80表示工业园区或者经济开发区。</li>
     * <li>第七、九位：表示乡、镇（街道办事处）。其中000-099表示街道办事处，100-199表示镇，200-299表示乡，400-479表示林场、牧场、科技城、园区，480-499表示林业管理局，500-579表示农场、团，580-599表示畜牧场。</li>
     *
     * @param areaBO 地区信息
     */
    public ImportAreaBO fillAreaInfo(ImportAreaBO areaBO, Predicate<String> predicateExistsArea) {
        return parseCode(areaBO.getCode(), areaBO.getName(), predicateExistsArea);
    }

    public static ImportAreaBO parseCode(@NotBlank String code, String name, Predicate<String> predicateExistsArea) {
        Assert.notBlank(code, "行政区域编码为空");
        code = code.trim();
        var codeArray = code2Array(code);
        Assert.isTrue(codeArray.length == 3, "暂不支持的行政区域编码");

        ImportAreaBO areaBO = new ImportAreaBO();
        areaBO.setCode(code);
        areaBO.setName(name.strip().replaceAll("\\u00A0", ""));
        areaBO.setPinyin(PinyinUtil.getPinyin(areaBO.getName()));

        if ("00".equals(codeArray[1])) {
            // 省、直辖市、自治区、特别行政区
            areaBO.setAreaType(AreaType.PROVINCE.getValue());
            areaBO.setParentCode(COUNTRY_CHINA_CODE);
            return areaBO;
        }

        if ("00".equals(codeArray[2])) {
            // 地级市
            areaBO.setAreaType(AreaType.CITY.getValue());
            areaBO.setParentCode(codeArray[0] + "00" + "00");
            return areaBO;
        }

        String parentCode = codeArray[0] + codeArray[1] + "00";
        if (predicateExistsArea.test(parentCode)) {
            // 县
            areaBO.setParentCode(parentCode);
            areaBO.setAreaType(AreaType.AREA.getValue());
            return areaBO;
        }
        // 县级市
        parentCode = codeArray[0] + "00" + "00";
        if (predicateExistsArea.test(parentCode)) {
            areaBO.setParentCode(parentCode);
            areaBO.setAreaType(AreaType.CITY.getValue());
            return areaBO;
        }

        throw new IllegalArgumentException("未知上级行政区域");
    }

    private static String[] code2Array(String code) {
        List<String> codeList = new ArrayList<>();

        int subIndex = 0;
        while (true) {
            String temp = CharSequenceUtil.sub(code, subIndex, subIndex + 2);
            if (CharSequenceUtil.isBlank(temp)) {
                break;
            }
            subIndex += 2;
            codeList.add(temp);
        }
        return codeList.toArray(String[]::new);
    }
}
