package com.elitescloud.cloudt.system.rpc.dpr;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprSysInternallyEnum;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.convert.SysDprRoleApiRuleConvert;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiDataRuleListQueryDTO;
import com.elitescloud.cloudt.system.dto.req.UnifyQueryQueryDTO;
import com.elitescloud.cloudt.system.model.vo.sbean.SysDprRoleApiDataRuleListQueryBean;
import com.elitescloud.cloudt.system.rpc.dpr.strategy.impl.DprSysInternallyDynamic;
import com.elitescloud.cloudt.system.service.SysUnifyQueryQueryService;
import com.elitescloud.cloudt.system.service.model.bo.DprRuleValueSysInternallyBo;
import com.elitescloud.cloudt.system.service.repo.BusinessObjectRepoProc;
import com.elitescloud.cloudt.system.service.repo.RoleDataPermissionRepoProc;
import com.elitescloud.cloudt.system.spi.DprSysRuleSysInternallyValueSpi;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.annotation.Order;
import org.springframework.core.task.TaskExecutor;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;


/**
 * @author : chen
 * <p>
 * 2022-11-24 14:22
 */
@Slf4j
public class RoleAppApiDataPermissionRpcServiceUtil {
    public static final String NO_PERMISSION = "-1";


    /**
     * @param userInfo 当前用户信息
     *                 valueServiceSpi 接口
     *                 sysBean 角色规则对象
     * @return SysDprRoleApiDataRuleListQueryDTO 角色规则对象包括规则值
     * @description 设置系统内置规则值 会根据不同的规则策略动态获取。
     */
    public static List<SysDprRoleApiDataRuleListQueryDTO> setSysSysDprRoleApiRuleValue(DprSysRuleSysInternallyValueSpi valueServiceSpi,
                                                                                       GeneralUserDetails userInfo,
                                                                                       List<SysDprRoleApiDataRuleListQueryBean> sysBean) {
        Map<String, DprRuleValueSysInternallyBo> dprSysInternallyGroupMap = new HashMap<>(16);
        List<SysDprRoleApiDataRuleListQueryDTO> sysDprRoleApiDataRuleList =
                sysBean.stream().map(sysDprRoleApiDataRuleListQueryBean -> {
                    var ruleDTO = SysDprRoleApiRuleConvert.INSTANCE.beanToDto(sysDprRoleApiDataRuleListQueryBean);
                    String dprSysInternally = CharSequenceUtil.blankToDefault(ruleDTO.getDprRuleValue(), ruleDTO.getDprSysInternally());
                    var dsiEnum = parseSysInternal(dprSysInternally);
                    //本人-下级等内置策略,不同的菜单权限只需要查询一次，没有必要多次循环，已经查询过了存储起来，下次直接使用。
                    DprRuleValueSysInternallyBo valueBo;
                    if (dprSysInternallyGroupMap.containsKey(dsiEnum.name())) {
                        valueBo = dprSysInternallyGroupMap.get(dsiEnum.name());
                    } else {
                        valueBo = valueServiceSpi.getSysInternallyRuleFieldRuleValue(userInfo, ruleDTO, dsiEnum);
                        dprSysInternallyGroupMap.put(dsiEnum.name(), valueBo);
                    }
                    ruleDTO.setDprRuleValue(valueBo.getDprRuleValue());
                    ruleDTO.setDprRuleFieldType(valueBo.getDprRuleFieldType());
                    return ruleDTO;
                }).collect(Collectors.toList());
        return sysDprRoleApiDataRuleList;
    }

    /**
     * 系统内置的值
     *
     * @param valueServiceSpi
     * @param userInfo
     * @param ruleList
     */
    public static void setRuleValueForInternal(DprSysRuleSysInternallyValueSpi valueServiceSpi,
                                               GeneralUserDetails userInfo,
                                               List<SysDprRoleApiDataRuleListQueryDTO> ruleList) {
        if (CollUtil.isEmpty(ruleList)) {
            return;
        }
        Map<String, DprRuleValueSysInternallyBo> dprSysInternallyGroupMap = new HashMap<>(16);
        for (SysDprRoleApiDataRuleListQueryDTO ruleDTO : ruleList) {
            String dprSysInternally = ruleDTO.getDprRuleValue();
            var dsiEnum = parseSysInternal(dprSysInternally);
            //本人-下级等内置策略,不同的菜单权限只需要查询一次，没有必要多次循环，已经查询过了存储起来，下次直接使用。
            DprRuleValueSysInternallyBo valueBo;
            if (dprSysInternallyGroupMap.containsKey(dsiEnum.name())) {
                valueBo = dprSysInternallyGroupMap.get(dsiEnum.name());
            } else {
                valueBo = valueServiceSpi.getSysInternallyRuleFieldRuleValue(userInfo, ruleDTO, dsiEnum);
                dprSysInternallyGroupMap.put(dsiEnum.name(), valueBo);
            }
            ruleDTO.setDprRuleValue(valueBo.getDprRuleValue());
            ruleDTO.setDprRuleFieldType(valueBo.getDprRuleFieldType());
        }
    }

    /**
     * 业务数据集设置规则值
     */
    public static List<SysDprRoleApiDataRuleListQueryDTO> setBusinessSysDprRoleApiRuleValue(GeneralUserDetails userInfo,
                                                                                            List<SysDprRoleApiDataRuleListQueryBean> businessBean) {
        var beanValueList = businessBean.stream().map(sysDprRoleApiDataRuleListQueryBean -> {
            var ruleDTO = SysDprRoleApiRuleConvert.INSTANCE.beanToDto(sysDprRoleApiDataRuleListQueryBean);
            Assert.hasText(ruleDTO.getDprRuleValue(), "值空，请检查是否角色中配置了自定义值。Business规则：" + ruleDTO.getDprRuleId());
            //设置自定义规则的规则值
            ruleDTO.setDprRuleValue(ruleDTO.getDprRuleValue());
            return ruleDTO;
        }).collect(Collectors.toList());
        return beanValueList;
    }

    public static void setRuleValueForBusinessSet(GeneralUserDetails userInfo, List<SysDprRoleApiDataRuleListQueryDTO> ruleList) {
        if (CollUtil.isEmpty(ruleList)) {
            return;
        }

        List<CompletableFuture<Void>> completableFutureList = new ArrayList<>(ruleList.size());
        TaskExecutor taskExecutor = SpringContextHolder.getBean(TaskExecutor.class);
        for (SysDprRoleApiDataRuleListQueryDTO rule : ruleList) {
            if (Boolean.FALSE.equals(rule.getRefResource())) {
                continue;
            }

            completableFutureList.add(CompletableFuture.runAsync(() -> {
                log.info("query public resource permission start ...");
                long start = System.currentTimeMillis();
                String value = queryRefBusinessResourceValue(rule, userInfo);
                log.info("public resource value:{}, {}, {}, {}", rule.getRoleCode(), rule.getDprRuleField(), rule.getRefBusinessObject(), value);
                log.info("query public resource permission end, cost {}ms", System.currentTimeMillis() - start);
                rule.setDprRuleValue(value);
            }, taskExecutor));
        }
        if (completableFutureList.isEmpty()) {
            return;
        }
        CompletableFuture.allOf(completableFutureList.toArray(CompletableFuture[]::new)).join();
    }

    /**
     * 自定义数据集设置规则值
     */
    public static List<SysDprRoleApiDataRuleListQueryDTO> setCustomSysDprRoleApiRuleValue(GeneralUserDetails userInfo,
                                                                                          List<SysDprRoleApiDataRuleListQueryBean> businessBean) {
        var beanValueList = businessBean.stream().map(sysDprRoleApiDataRuleListQueryBean -> {
            var ruleDTO = SysDprRoleApiRuleConvert.INSTANCE.beanToDto(sysDprRoleApiDataRuleListQueryBean);
            Assert.hasText(ruleDTO.getDprRuleValue(), "值空，请检查是否角色中配置了自定义值。Custom规则：" + ruleDTO.getDprRuleId());
            //设置自定义规则的规则值
            ruleDTO.setDprRuleValue(ruleDTO.getDprRuleValue());
            return ruleDTO;
        }).collect(Collectors.toList());
        return beanValueList;
    }

    public static String normalizeRuleValue(Collection<?> values) {
        if (CollUtil.isEmpty(values)) {
            // 没有有效值，则返回-1，标识无权限
            return RoleAppApiDataPermissionRpcServiceUtil.NO_PERMISSION;
        }
        return StringUtils.collectionToDelimitedString(values, ",");
    }

    /**
     * 设置权限规则中的值
     *
     * @param ruleList
     * @param userInfo
     */
    public static void fillDataPermissionValue(List<SysDprRoleApiDataRuleListQueryDTO> ruleList, GeneralUserDetails userInfo) {
        if (ruleList.isEmpty()) {
            return;
        }

        var valueServiceSpi = SpringContextHolder.getBean(DprSysRuleSysInternallyValueSpi.class);
        var valueTypeMap = ruleList.stream().collect(Collectors.groupingBy(SysDprRoleApiDataRuleListQueryDTO::getDprRuleValueType));
        for (Map.Entry<String, List<SysDprRoleApiDataRuleListQueryDTO>> entry : valueTypeMap.entrySet()) {
            var type = DprRuleValueTypeEnum.parse(entry.getKey());
            Assert.notNull(type, "未知规则值类型：" + entry.getKey());
            switch (type) {
                case DPR_RULE_VALUE_TYPE_SYS:
                    // 系统内置
                    RoleAppApiDataPermissionRpcServiceUtil.setRuleValueForInternal(valueServiceSpi, userInfo, entry.getValue());
                    break;
                case DPR_RULE_VALUE_TYPE_CUSTOM:
                    // 自定义
                    // 原始值，无需处理
                    break;
                case DPR_RULE_VALUE_TYPE_BUSINESS:
                    RoleAppApiDataPermissionRpcServiceUtil.setRuleValueForBusinessSet(userInfo, entry.getValue());
                    break;
                default:
                    throw new BusinessException("暂不支持的值类型：" + entry.getKey());
            }
        }
    }

    private static String queryRefBusinessResourceValue(SysDprRoleApiDataRuleListQueryDTO ruleDTO, GeneralUserDetails userInfo) {
        // 获取关联资源的权限规则
        Assert.hasText(ruleDTO.getRefBusinessObject(), "存在关联的业务对象为空：" + ruleDTO.getRoleCode());
        Assert.hasText(ruleDTO.getRoleCode(), "存在权限配置的角色编码为空");
        Assert.hasText(ruleDTO.getRefField(), "存在管理的业务对象字段为空");
        var refRuleList = SpringContextHolder.getBean(RoleDataPermissionRepoProc.class).queryForRefBusinessObjectResourceByRole(ruleDTO.getRoleCode(), ruleDTO.getRefBusinessObject());
        if (refRuleList.isEmpty()) {
            // 公共资源未配置，则无权限
            return NO_PERMISSION;
        }
        fillDataPermissionValue(refRuleList, userInfo);

        // 获取条件
        Map<String, List<String>> refRuleValues = new HashMap<>(refRuleList.size());
        Map<String, String> fieldTypeMap = new HashMap<>(refRuleList.size());
        for (SysDprRoleApiDataRuleListQueryDTO refRule : refRuleList) {
            if (CharSequenceUtil.isBlank(refRule.getDprRuleField()) || CharSequenceUtil.isBlank(refRule.getDprRuleValue())
                    || CharSequenceUtil.isBlank(refRule.getDprRuleFieldType())) {
                log.warn("存在数据权限规则字段或值为空：{}，{}，{}，{}", refRule.getRoleCode(), refRule.getDprRuleField(), refRule.getDprRuleFieldType(),
                        refRule.getDprRuleValue());
                continue;
            }
            if (NO_PERMISSION.equals(refRule.getDprRuleValue())) {
                if (DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name().equals(refRule.getDprRuleValueType()) ||
                        DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS.name().equals(refRule.getDprRuleValueType())) {
                    // 无权限
                    return NO_PERMISSION;
                }
            }
            fieldTypeMap.put(refRule.getDprRuleField(), refRule.getDprRuleFieldType());

            List<String> values = refRuleValues.getOrDefault(refRule.getDprRuleField(), new ArrayList<>(0));
            if (refRule.getDprRuleValue().contains(",")) {
                values.addAll(Arrays.stream(refRule.getDprRuleValue().split(",")).collect(Collectors.toList()));
            } else {
                values.add(refRule.getDprRuleValue());
            }
            refRuleValues.put(refRule.getDprRuleField(), values);
        }
        Map<String, Object> conditions = new HashMap<>(refRuleValues.size());
        for (Map.Entry<String, List<String>> entry : refRuleValues.entrySet()) {
            if (entry.getValue().contains(NO_PERMISSION)) {
                return NO_PERMISSION;
            }
            if (entry.getValue().size() == 1) {
                conditions.put(entry.getKey(), ObjUtil.convertBasicValue(entry.getValue().get(0), fieldTypeMap.get(entry.getKey())));
                continue;
            }
            conditions.put(entry.getKey(), entry.getValue().stream().map(t -> ObjUtil.convertBasicValue(t, fieldTypeMap.get(entry.getKey()))).collect(Collectors.toList()));
        }

        UnifyQueryQueryDTO queryDTO = new UnifyQueryQueryDTO();
        queryDTO.setBusinessObjectCode(ruleDTO.getRefBusinessObject());
        queryDTO.setQueryFields(List.of(ruleDTO.getRefField()));
        queryDTO.setConditions(conditions);

        var supportTenant = SpringContextHolder.getBean(BusinessObjectRepoProc.class).getSupportTenant(queryDTO.getBusinessObjectCode());
        var unifyQueryService = SpringContextHolder.getBean(SysUnifyQueryQueryService.class);
        var unifyQueryData = Boolean.TRUE.equals(supportTenant) ? unifyQueryService.query(queryDTO).computeData() :
                SpringContextHolder.getBean(TenantDataIsolateProvider.class).byDefaultDirectly(() -> unifyQueryService.query(queryDTO)).computeData();
        Assert.isTrue(unifyQueryData.getSuccess(), unifyQueryData.getFailMsg());
        return obtainValueFromUnifyQueryData(unifyQueryData.getData());
    }

    private static String obtainValueFromUnifyQueryData(Object data) {
        if (data == null) {
            return NO_PERMISSION;
        }
        if (data instanceof List) {
            var tempDataList = (List<?>) data;
            var element = CollUtil.isEmpty(tempDataList) ? null : tempDataList.get(0);
            if (element == null) {
                return NO_PERMISSION;
            }
            if (element instanceof Map) {
                var tempDataMap = (Map<?, ?>) element;
                return CollUtil.isEmpty(tempDataMap) ? NO_PERMISSION : normalizeRuleValue(tempDataMap.values());
            }
        }
        log.error("解析权限配置数据：{}", JSONUtil.toJsonString(data));
        throw new BusinessException("解析权限配置异常");
    }

    /**
     * @param @param null
     * @return
     * @description SPI策略获取 依赖@Order
     */
    private static ServiceLoader.Provider<DprSysRuleSysInternallyValueSpi> getValueServiceSpi(List<ServiceLoader.Provider<DprSysRuleSysInternallyValueSpi>> valueServiceSpiList) {
        if (valueServiceSpiList.size() == 1) {
            return valueServiceSpiList.get(0);
        } else {
            int valueIndex = 0;
            int minOrder = 0;
            for (int i = 0; i < valueServiceSpiList.size(); i++) {
                var annotation = valueServiceSpiList.get(i).type().getDeclaredAnnotation(Order.class);
                if (annotation != null) {
                    if (minOrder == 0) {
                        minOrder = annotation.value();
                    } else {
                        if (minOrder > annotation.value()) {
                            minOrder = annotation.value();
                            valueIndex = i;
                        }
                    }
                }
            }
            return valueServiceSpiList.get(valueIndex);
        }
    }

    private static DprSysInternallyEnum parseSysInternal(String value) {
        if (value.startsWith(DprSysInternallyDynamic.PREFIX)) {
            return DprSysInternallyEnum.DPR_SYS_INTERNALLY_DYNAMIC;
        }

        return DprSysInternallyEnum.valueOf(value);
    }
}
