package com.elitescloud.cloudt.system.service;

import com.elitescloud.boot.common.param.IdCodeNameCheckParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeUpsertDTO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmployeePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpOrgRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeeDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeePagedRespVO;
import com.elitescloud.cloudt.system.model.vo.save.org.EmployeeOrgInfoSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.org.EmployeeOrgSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.org.EmployeeSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.user.UserSaveVO;
import com.elitescloud.cloudt.system.provider.imports.param.ImportEmployeeBO;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/8
 */
public interface EmployeeMngService {

    /**
     * 保存员工信息
     * <p>
     * 包含用户、组织等详细信息
     *
     * @param saveVO 员工信息
     * @return 保存结果
     */
    ApiResult<Long> upsert(EmployeeSaveVO saveVO);

    /**
     * 员工创建与修改
     * <p>
     * 根据id或code确定员工唯一性，如果存在则修改，否则新增
     *
     * @param upsertDTO 员工信息
     * @return 员工编号
     */
    ApiResult<String> upsert(EmployeeUpsertDTO upsertDTO);

    /**
     * 检查员工信息
     *
     * @param upsertDTOList 员工信息列表
     * @return 校验结果
     */
    ApiResult<List<String>> checkForUpsert(List<EmployeeUpsertDTO> upsertDTOList);

    /**
     * 导入员工
     * <p>
     * 导入时必须是不存在的员工和账号，如果存在，则报错
     *
     * @param employeeBO 员工信息
     * @return 员工ID
     */
    ApiResult<Long> importEmployee(ImportEmployeeBO employeeBO);

    /**
     * 保存员工用户
     *
     * @param orgId  组织ID
     * @param saveVO 用户信息
     * @return 保存结果
     */
    ApiResult<Long> upsertUser(Long orgId, UserSaveVO saveVO);

    /**
     * 保存员工信息
     *
     * @param saveVO 员工信息
     * @return 保存结果
     */
    ApiResult<Long> upsertOrg(EmployeeOrgInfoSaveVO saveVO);

    /**
     * 更新启用状态
     *
     * @param id 员工ID
     * @return 更新结果
     */
    ApiResult<Long> updateEnabled(Long id);

    /**
     * 更新启用状态
     *
     * @param code    员工编号
     * @param enabled 启用状态
     * @return 更新结果
     */
    ApiResult<String> updateEnabledByCode(@NotBlank String code, Boolean enabled);


    /**
     * 更新手机号
     *
     * @param id         用户ID
     * @param mobile     手机号
     * @param updateUser 是否更新账号里的手机号
     * @return 用户ID
     */
    ApiResult<Long> updateMobile(@NotNull Long id, String mobile, Boolean updateUser);

    /**
     * 更新邮箱
     *
     * @param id         用户ID
     * @param email      邮箱
     * @param updateUser 是否更新账号里的邮箱
     * @return 用户ID
     */
    ApiResult<Long> updateEmail(@NotNull Long id, String email, Boolean updateUser);

    /**
     * 保存员工的组织
     * <p>
     * 员工的所有组织
     *
     * @param id            员工ID
     * @param orgSaveVoList 组织列表
     * @return 员工ID
     */
    ApiResult<Long> saveEmployeeOrg(Long id, List<EmployeeOrgSaveVO> orgSaveVoList);

    /**
     * 删除员工
     *
     * @param id       员工ID
     * @param keepUser 是否保留账号信息
     * @return 删除结果
     */
    ApiResult<Long> delete(Long id, Boolean keepUser);

    /**
     * 保存员工的角色
     *
     * @param id      员工ID
     * @param roleIds 角色ID
     * @return 保存结果
     */
    ApiResult<Long> saveRole(Long id, List<Long> roleIds);

    /**
     * 绑定员工的业务员
     *
     * @param id         员工ID
     * @param salesmanId 业务员ID，为空则是解绑
     * @return 绑定结果
     */
    ApiResult<Boolean> bindSalesman(@NotNull Long id, String salesmanId);

    /**
     * 获取员工详细信息
     *
     * @param id 员工ID
     * @return 详细信息
     */
    ApiResult<EmployeeDetailRespVO> getDetail(Long id);

    /**
     * 获取员工的所在组织
     *
     * @param id 员工ID
     * @return 组织列表
     */
    ApiResult<List<EmpOrgRespVO>> getEmployeeOrgs(Long id);

    /**
     * 获取员工的组织
     *
     * @param ids 员工ID
     * @return 员工ID与组织
     */
    ApiResult<Map<Long, List<IdCodeNameParam>>> getEmployeeOrgs(List<Long> ids);

    /**
     * 获取员工的角色
     *
     * @param userIds 员工的用户ID
     * @return 员工ID与角色
     */
    ApiResult<Map<Long, List<IdCodeNameParam>>> getEmployeeRolesByUserIds(List<Long> userIds);

    /**
     * 获取员工的角色
     *
     * @param id 员工ID
     * @return 角色列表
     */
    ApiResult<List<IdCodeNameCheckParam>> getRole(Long id);

    /**
     * 分页查询员工信息
     *
     * @param queryVO 查询参数
     * @return 员工列表
     */
    ApiResult<PagingVO<EmployeePagedRespVO>> page(EmployeePageQueryVO queryVO);

    /**
     * 处理数据
     *
     * @param pageData
     * @param userMap
     */
    void fillPage(@NotNull List<EmployeePagedRespVO> pageData, Map<Long, SysUserBasicDTO> userMap);

    ApiResult testUpdate(EmployeeSaveVO saveVO);
}
