package com.elitescloud.cloudt.system.service.alert;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.digest.MD5;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.cloudt.system.constant.SysAlertType;
import com.elitescloud.cloudt.system.model.vo.resp.extend.AlertConfigWxWorkRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.AlertConfigWxWorkSaveVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 企业微信.
 * <p>
 * <a href = "https://developer.work.weixin.qq.com/document/path/99110#markdown%E7%B1%BB%E5%9E%8B">文档地址</a>
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/26
 */
@Slf4j
public class WxWorkAlertProvider implements AlertProvider<AlertConfigWxWorkSaveVO, AlertConfigWxWorkRespVO> {

    private static final String MSG_TYPE = "markdown";
    private final RedisUtils redisUtils;
    private final RestTemplate restTemplate;

    public WxWorkAlertProvider(RedisUtils redisUtils) {
        this.redisUtils = redisUtils;
        this.restTemplate = RestTemplateFactory.instance();
    }

    @Override
    public SysAlertType alertType() {
        return SysAlertType.WX_WORK;
    }

    @Override
    public String toString(AlertConfigWxWorkSaveVO saveVO) {
        if (saveVO == null) {
            return null;
        }
        Assert.notEmpty(saveVO.getWebhookUrls(), "地址不能为空");

        var cfg = new Cfg(saveVO.getWebhookUrls(), saveVO.getTmplContent());

        // 去重配置
        boolean deduplicate = ObjectUtil.defaultIfNull(saveVO.getDeduplicate(), false);
        cfg.setDeduplicate(deduplicate);
        cfg.setDeduplicateFields(saveVO.getDeduplicateFields());
        cfg.setDeduplicateIntervals(ObjectUtil.defaultIfNull(saveVO.getDeduplicateIntervals(), 0));
        if (deduplicate) {
            Assert.notEmpty(cfg.getDeduplicateFields(), "请选择去重字段");
            Assert.isTrue(cfg.getDeduplicateIntervals() > 0, "去重间隔必须大于0");
        }

        return JSONUtil.toJsonString(cfg);
    }

    @Override
    public AlertConfigWxWorkRespVO parse(String cfgJson) {
        if (CharSequenceUtil.isBlank(cfgJson)) {
            return null;
        }

        var cfg = this.parseCfg(cfgJson);

        AlertConfigWxWorkRespVO respVO = new AlertConfigWxWorkRespVO();
        respVO.setWebhookUrls(cfg.getWebhookUrlMap());
        respVO.setTmplContent(cfg.getTmplContent());

        // 去重
        boolean deduplicate = ObjectUtil.defaultIfNull(cfg.getDeduplicate(), false);
        respVO.setDeduplicate(deduplicate);
        respVO.setDeduplicateFields(ObjectUtil.defaultIfNull(cfg.getDeduplicateFields(), Collections.emptyList()));
        respVO.setDeduplicateIntervals(ObjectUtil.defaultIfNull(cfg.getDeduplicateIntervals(), 0));

        return respVO;
    }

    @Override
    public boolean send(String cfgJson, String content, String category) {
        if (CharSequenceUtil.isBlank(cfgJson)) {
            log.info("企业微信配置内容为空");
            return false;
        }
        var cfg = this.parseCfg(cfgJson);

        // 发送
        return this.execute(cfg, content, category);
    }

    @Override
    public boolean sendByTmpl(String cfgJson, Map<String, Object> tmplParams, String category) {
        if (CharSequenceUtil.isBlank(cfgJson)) {
            log.info("企业微信配置内容为空");
            return false;
        }
        var cfg = this.parseCfg(cfgJson);

        var tmplContent = cfg.getTmplContent();
        if (CharSequenceUtil.isBlank(tmplContent)) {
            log.info("模板内容未配置：{}", cfgJson);
            return false;
        }

        // 判重
        if (isDuplicate(cfg, tmplParams, category)) {
            log.info("消息重复：{}", JSONUtil.toJsonString(cfgJson));
            return false;
        }

        var content = StrUtil.format(tmplContent, tmplParams, true);

        // 发送
        return this.execute(cfg, content, category);
    }

    private boolean isDuplicate(Cfg cfg, Map<String, Object> tmplParams, String category) {
        if (!Boolean.TRUE.equals(cfg.getDeduplicate())) {
            // 不需要去重
            return false;
        }

        // 拼接去重字段
        StringBuffer txt = new StringBuffer();
        for (String field : cfg.getDeduplicateFields()) {
            var key = this.normalizeTmplField(field);
            var value = ObjectUtil.defaultIfNull(tmplParams.get(key), "").toString();
            if (CharSequenceUtil.isBlank(value)) {
                // 空值忽略
                continue;
            }
            txt.append(value + ":");
        }

        if (txt.length() == 0) {
            // 没有有效值
            return false;
        }

        var digestHex = MD5.create().digestHex(ObjectUtil.defaultIfNull(category, "default") + ":" + txt, StandardCharsets.UTF_8);
        var exists = redisUtils.getRedisTemplate().opsForValue().setIfAbsent(digestHex, "1", Duration.ofMinutes(cfg.getDeduplicateIntervals()));
        log.info("去重：{}，{}", txt, exists);
        return exists != null && !exists;
    }

    private String normalizeTmplField(String tmplField) {
        return CharSequenceUtil.strip(tmplField, "{", "}");
    }

    private boolean execute(Cfg cfg, String content, String category) {
        // 根据分类获取
        List<String> urls = CollUtil.isEmpty(cfg.getWebhookUrlMap()) ? null :
                cfg.getWebhookUrlMap().get(CharSequenceUtil.blankToDefault(category, CATEGORY_DEFAULT));
        if (urls != null) {
            urls = urls.stream().filter(StringUtils::hasText).collect(Collectors.toList());
        }

        if (CollUtil.isEmpty(urls)) {
            log.info("企业微信地址配置未配置：{}，{}", category, JSONUtil.toJsonString(cfg));
            return false;
        }

        // 企业微信有限制字数
        content = CharSequenceUtil.sub(content, 0, 3000);

        for (String webhookUrl : urls) {
            this.execute(webhookUrl, content);
        }

        return true;
    }

    private void execute(String url, String content) {
        Map<String, Object> body = new HashMap<>();
        body.put("msgtype", MSG_TYPE);
        body.put("markdown", Map.of("content", content));

        log.info("【企业微信】发送预警：{}, {}", url, content);

        ResponseEntity<String> resp = null;
        try {
            resp = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(body), String.class);
            if (resp.getStatusCode().is2xxSuccessful()) {
                log.info("发送成功！");
                return;
            }
        } catch (RestClientException e) {
            log.error("发送预警异常：", e);
            return;
        }
        log.info("发送失败：{}, {}", resp.getStatusCodeValue(), resp.getBody());
    }

    private Cfg parseCfg(String cfgJson) {
        var cfg = JSONUtil.json2Obj(cfgJson, Cfg.class);
        if (CollUtil.isEmpty(cfg.getWebhookUrlMap()) && CollUtil.isNotEmpty(cfg.getWebhookUrls())) {
            // 兼容性升级
            cfg.setWebhookUrlMap(Map.of(CATEGORY_DEFAULT, cfg.getWebhookUrls()));
        }
        return cfg;
    }

    static class Cfg implements Serializable {
        private static final long serialVersionUID = 4537456626636877932L;
        private List<String> webhookUrls;
        private Map<String, List<String>> webhookUrlMap;
        private String tmplContent;
        /**
         * 是否去重
         */
        private Boolean deduplicate;

        /**
         * 去重字段
         */
        private List<String> deduplicateFields;

        /**
         * 去重间隔
         * <p>
         * 单位分钟
         */
        private Integer deduplicateIntervals;

        public Cfg(Map<String, List<String>> webhookUrlMap, String tmplContent) {
            this.webhookUrlMap = webhookUrlMap;
            this.tmplContent = tmplContent;
        }

        public Cfg() {
        }

        public Map<String, List<String>> getWebhookUrlMap() {
            return webhookUrlMap;
        }

        public void setWebhookUrlMap(Map<String, List<String>> webhookUrlMap) {
            this.webhookUrlMap = webhookUrlMap;
        }

        public List<String> getWebhookUrls() {
            return webhookUrls;
        }

        public void setWebhookUrls(List<String> webhookUrls) {
            this.webhookUrls = webhookUrls;
        }

        public String getTmplContent() {
            return tmplContent;
        }

        public void setTmplContent(String tmplContent) {
            this.tmplContent = tmplContent;
        }

        public Boolean getDeduplicate() {
            return deduplicate;
        }

        public void setDeduplicate(Boolean deduplicate) {
            this.deduplicate = deduplicate;
        }

        public List<String> getDeduplicateFields() {
            return deduplicateFields;
        }

        public void setDeduplicateFields(List<String> deduplicateFields) {
            this.deduplicateFields = deduplicateFields;
        }

        public Integer getDeduplicateIntervals() {
            return deduplicateIntervals;
        }

        public void setDeduplicateIntervals(Integer deduplicateIntervals) {
            this.deduplicateIntervals = deduplicateIntervals;
        }
    }
}
