package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.constant.CurrencyRateCalcMethod;
import lombok.extern.slf4j.Slf4j;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 汇率.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
@Slf4j
public class CurrencyRateInitProvider extends AbstractBasicDataInitProvider {

    @Override
    public String typeName() {
        return "汇率";
    }

    @Override
    public String tableName() {
        return "sys_currency_rate";
    }

    @Override
    public List<String> fields() {
        return List.of("from_curr", "to_curr", "ratio", "cal_method", "enabled", "valid_from", "valid_to", "std_version",
                "create_time", "modify_time");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("from_curr", "自货币码");
        titles.put("to_curr", "至货币码");
        titles.put("ratio", "汇率");
        titles.put("cal_method", "计算方法");
        titles.put("cal_method_name", "计算方法名称");
        titles.put("valid_from", "起始时间");
        titles.put("valid_to", "截止时间");
        titles.put("enabled", "状态");
        titles.put("create_time", "创建时间");
        titles.put("modify_time", "最后修改时间");
        return titles;
    }

    @Override
    public boolean isTenantData() {
        return true;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.desc("create_time"));
    }

    @Override
    public String fieldAppCode() {
        return "";
    }

    @Override
    public Map<String, Object> convertForExport(Map<String, Object> data, List<Map<String, Object>> preData) {
        // 过滤启用的数据
        var enabled = getBooleanValue(data.get("enabled"), null);
        if (enabled == null || !enabled) {
            return null;
        }

        // 转换类型
        var type = CurrencyRateCalcMethod.parse(ObjectUtil.defaultIfNull(data.get("call_method"), "").toString());
        if (type != null) {
            data.put("call_method_name", type.getDescription());
        }
        return data;
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        var from_curr = getStringValue(data, "from_curr", null);
        Assert.notBlank(from_curr, "缺少from_curr");
        var to_curr = getStringValue(data, "to_curr", null);
        Assert.notBlank(to_curr, "缺少to_curr");
        this.normalizeBooleanValue(data, "enabled");

        if (currentData.isEmpty()) {
            return UpdateType.ADD;
        }

        var existsData = currentData.parallelStream()
                .filter(t -> from_curr.equals(t.get("from_curr")) && to_curr.equals(t.get("to_curr"))
                ).findFirst()
                .orElse(null);
        if (existsData == null) {
            this.normalizeForAdd(data);
            return UpdateType.ADD;
        }
        this.normalizeForUpdate(data, existsData);
        return UpdateType.UPDATE;
    }

    private void normalizeForAdd(Map<String, Object> data) {
        data.put("app_state", this.getBooleanValue(data.get("app_state"), true));
    }

    private void normalizeForUpdate(Map<String, Object> data, Map<String, Object> dataExists) {
        data.put("id", dataExists.get("id"));
        data.put("app_state", ObjectUtil.defaultIfNull(dataExists.get("app_state"), this.getBooleanValue(data.get("app_state"), true)));
    }
}
