package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.system.cacheable.SysCacheAppRpcService;
import com.elitescloud.cloudt.system.service.SysTenantBasicDataService;
import com.elitescloud.cloudt.system.service.common.constant.PlatformAppTypeEnum;
import org.springframework.core.task.TaskExecutor;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 平台app.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
public class PlatformAppInitProvider extends AbstractBasicDataInitProvider {

    @Override
    public String typeName() {
        return "应用";
    }

    @Override
    public String tableName() {
        return "sys_platform_app";
    }

    @Override
    public List<String> fields() {
        return List.of("create_time", "modify_time", "app_code", "app_name", "app_type", "app_describe", "app_state", "app_order", "std_version");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("app_code", "应用编码");
        titles.put("app_name", "应用名称");
        titles.put("app_type", "应用类型");
        titles.put("app_type_name", "应用类型名称");
        titles.put("app_order", "顺序");
        titles.put("app_state", "状态");
        titles.put("app_describe", "描述");
        titles.put("create_time", "创建时间");
        titles.put("modify_time", "最后修改时间");
        return titles;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.desc("app_order"));
    }

    @Override
    public String fieldAppCode() {
        return "app_code";
    }

    @Override
    public Map<String, Object> convertForExport(Map<String, Object> data, List<Map<String, Object>> preData) {
        // 过滤启用的数据
        var state = getBooleanValue(data.get("app_state"), null);
        if (state == null || !state) {
            return null;
        }

        // 转换类型
        var type = PlatformAppTypeEnum.parse(ObjectUtil.defaultIfNull(data.get("app_type"), "").toString());
        if (type != null) {
            data.put("app_type_name", type.getDescription());
        }
        return data;
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        var app_code = this.getStringValue(data, "app_code", null);
        Assert.notBlank(app_code, "应用编码为空");

        this.normalizeBooleanValue(data, "app_state");

        if (currentData.isEmpty()) {
            return UpdateType.ADD;
        }

        var existsData = currentData.stream()
                .filter(t -> app_code.equals(t.get("app_code")))
                .findFirst()
                .orElse(null);

        if (existsData == null) {
            this.normalizeForAdd(data);
            return UpdateType.ADD;
        }
        this.normalizeForUpdate(data, existsData);
        return UpdateType.UPDATE;
    }

    @Override
    public void afterExecute(boolean export, Throwable e) {
        if (e == null || export) {
            return;
        }

        SpringContextHolder.getBean(TaskExecutor.class).execute(() -> {
            SpringContextHolder.getBean(SysCacheAppRpcService.class).clearCacheForAllTenant();
        });
    }

    private void normalizeForAdd(Map<String, Object> data) {
        data.put("app_state", this.getBooleanValue(data.get("app_state"), true));
    }

    private void normalizeForUpdate(Map<String, Object> data, Map<String, Object> dataExists) {
        data.put("id", dataExists.get("id"));
        data.put("app_state", ObjectUtil.defaultIfNull(dataExists.get("app_state"), getBooleanValue(data.get("app_state"), true)));
    }
}
