package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.system.service.SysTenantBasicDataService;
import org.springframework.core.task.TaskExecutor;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 发号规则明细.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
public class PlatformNumberRuleDtlInitProvider extends AbstractBasicDataInitProvider {

    @Override
    public String typeName() {
        return "发号规则明细";
    }

    @Override
    public String tableName() {
        return "sys_platform_number_ruledtl";
    }

    @Override
    public List<String> fields() {
        return List.of("create_time", "modify_time", "app_code", "rule_id", "seq", "number_type",
                "number_pattern", "nn_len", "std_version");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("app_code", "应用编码");
        titles.put("rule_id", "规则ID");
        titles.put("seq", "顺序");
        titles.put("number_type", "取号类型");
        titles.put("number_pattern", "取号模式");
        titles.put("nn_len", "序号宽度");
        titles.put("create_time", "创建时间");
        return titles;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.asc("app_code"), OrderItem.desc("rule_id")
                , OrderItem.asc("seq"));
    }

    @Override
    public String fieldAppCode() {
        return "app_code";
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        var rule_id_original = this.getLongValue(data, "rule_id", null);
        Assert.notNull(rule_id_original, "规则ID为空");

        Assert.notEmpty(lastStepData, "发号规则为空");
        var rule_id = lastStepData.parallelStream()
                .filter(t -> rule_id_original.longValue() == ObjUtil.defaultIfNull((Long) t.get("original_id"), -1L))
                .findFirst()
                .map(t -> t.get("id"))
                .orElse(null);
        if (rule_id == null) {
            return UpdateType.IGNORE;
        }

        data.put("rule_id", rule_id);
        if (currentData.isEmpty()) {
            return UpdateType.ADD;
        }

        var exists = currentData.stream()
                .anyMatch(t -> rule_id == ObjUtil.defaultIfNull((Long) t.get("rule_id"), -1L));
        if (exists) {
            // 已存在则忽略
            return UpdateType.IGNORE;
        }
        return UpdateType.ADD;
    }

    @Override
    public void afterExecute(boolean export, Throwable e) {
        if (e == null || export) {
            return;
        }

        SpringContextHolder.getBean(TaskExecutor.class).execute(() -> {
            SysTenantBasicDataService tenantBasicDataService = SpringContextHolder.getBean(SysTenantBasicDataService.class);
            SpringContextHolder.getBean(TenantClientProvider.class).getAllTenants().forEach(tenant -> tenantBasicDataService.syncSequence(tenant.getId()));
        });
    }
}
