package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.common.constant.MimeTypeConstant;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.CloudtIdCreator;
import com.elitescloud.boot.util.FileUtil;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.system.service.impl.SystemDataServiceImpl;
import com.elitescloud.cloudt.system.service.manager.ResourceByteMngManager;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 导入导出模板.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
@Slf4j
public class TmplInitProvider extends AbstractBasicDataInitProvider {
    private static final String DIR = "tmpl";
    private Map<String, ResourceInfo> tmplData = new HashMap<>();
    private File workDir;
    private FileService fileService;
    private CloudtIdCreator idCreator;
    private NamedParameterJdbcTemplate jdbcTemplate;
    private SystemDataServiceImpl.ImportedData importedData;

    @Override
    public String typeName() {
        return "导入导出模板";
    }

    @Override
    public String tableName() {
        return "sys_tmpl";
    }

    @Override
    public List<String> fields() {
        return List.of("name", "code_", "file_code", "file_name", "export", "head_row", "field_type_row",
                "enabled", "data_limit_per", "async_threshold", "concurrent_limit", "attribute_list", "export_sheet_limit",
                "export_sheet_strategy", "app_code", "std_version", "create_time", "modify_time");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("app_code", "应用编码");
        titles.put("name", "模板名称");
        titles.put("code_", "编码");
        titles.put("file_name", "文件名称");
        titles.put("export", "是否是导出模板");
        titles.put("head_row", "头部行数");
        titles.put("field_type_row", "字段类型所在行");
        titles.put("data_limit_per", "数据量限制");
        titles.put("async_threshold", "异步阀值");
        titles.put("concurrent_limit", "并发量限制");
        titles.put("export_sheet_limit", "导出时sheet数据量限制");
        titles.put("export_sheet_strategy", "导出时超出sheet后策略");
        titles.put("attribute_list", "属性");
        titles.put("file_code", "模板文件");
        titles.put("create_time", "创建时间");
        titles.put("modify_time", "最后修改时间");
        return titles;
    }

    @Override
    public void prepareExport(File workDir) {
        this.workDir = workDir;
        this.fileService = SpringContextHolder.getBean(FileService.class);

        // 查询模板数据
        var jdbcTemplate = SpringContextHolder.getBean(NamedParameterJdbcTemplate.class);
        Map<String, Object> sqlParams = new HashMap<>();
        sqlParams.put("delete_flag", 0);
        sqlParams.put("tenant_id", TenantConstant.DEFAULT_TENANT_ID);
        sqlParams.put("resource_type", ResourceByteMngManager.BUSINESS_TYPE_TMPL);
        var tmplResourceList = jdbcTemplate.query("select resource_key, resource, mime_type, suffix, resource_name, size from sys_resource_byte " +
                        "where delete_flag = :delete_flag and tenant_id = :tenant_id and resource_type = :resource_type",
                sqlParams, new BeanPropertyRowMapper<>(ResourceInfo.class));
        if (CollUtil.isNotEmpty(tmplResourceList)) {
            for (var resource : tmplResourceList) {
                tmplData.put(resource.getResourceKey(), resource);
            }
        }
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.asc("app_code"), OrderItem.desc("create_time"));
    }

    @Override
    public String fieldAppCode() {
        return "app_code";
    }

    @Override
    public Map<String, Object> convertForExport(Map<String, Object> data, List<Map<String, Object>> preData) {
        // 过滤启用的数据
        var enabled = getBooleanValue(data.get("enabled"), null);
        if (enabled == null || !enabled) {
            return null;
        }

        // 模板文件
        var fileCode = data.get("file_code").toString();
        var resourceInfo = tmplData.get(fileCode);
        byte[] resourceBytes = null;
        if (resourceInfo != null) {
            resourceBytes = resourceInfo.resource;
        } else {
            HttpEntity<StreamingResponseBody> body = fileService.download(fileCode, null);
            if (body.hasBody() && body.getBody() != null) {
                ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
                try {
                    body.getBody().writeTo(outputStream);
                } catch (IOException e) {
                    log.error("下载模板文件{}异常：", fileCode, e);
                }
                resourceBytes = outputStream.toByteArray();
            }
        }
        if (ArrayUtil.isNotEmpty(resourceBytes)) {
            File tmplDir = new File(workDir, DIR);
            if (!tmplDir.exists()) {
                tmplDir.mkdirs();
            }
            File tmplFile = new File(tmplDir, fileCode + ".xlsx");
            FileUtil.writeBytes(resourceBytes, tmplFile);
        }

        return data;
    }

    @Override
    public void prepareForImport(SystemDataServiceImpl.ImportedData importData, Long tenantId) {
        this.idCreator = SpringContextHolder.getBean(CloudtIdCreator.class);
        this.importedData = importData;
        this.jdbcTemplate = SpringContextHolder.getBean(NamedParameterJdbcTemplate.class);
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        var code = this.getStringValue(data, "code_", null);
        Assert.notBlank(code, "模板编码为空");
        var fileCode = this.getStringValue(data, "file_code", null);
        Assert.notBlank(fileCode, "模板文件编码为空");

        // 保存模板文件
        var tmplFile = this.saveResource(fileCode, data);
        if (tmplFile == null) {
            return UpdateType.IGNORE;
        }

        this.normalizeBooleanValue(data, "export");
        this.normalizeIntegerValue(data, "head_row");
        this.normalizeIntegerValue(data, "field_type_row");
        this.normalizeIntegerValue(data, "data_limit_per");
        this.normalizeIntegerValue(data, "async_threshold");
        this.normalizeIntegerValue(data, "concurrent_limit");
        this.normalizeIntegerValue(data, "export_sheet_limit");
        this.normalizeIntegerValue(data, "export_sheet_limit");

        if (currentData.isEmpty()) {
            return UpdateType.ADD;
        }

        var existsData = currentData.stream()
                .filter(t -> code.equals(t.get("code_")))
                .findFirst()
                .orElse(null);

        if (existsData == null) {
            return UpdateType.ADD;
        }
        this.normalizeForUpdate(data, existsData);
        return UpdateType.UPDATE;
    }

    private void normalizeForUpdate(Map<String, Object> data, Map<String, Object> dataExists) {
        data.put("id", dataExists.get("id"));
    }

    private File saveResource(String fileCode, Map<String, Object> data) {
        var filename = this.getStringValue(data, "file_name", null);
        if (CharSequenceUtil.isBlank(filename)) {
            filename = this.getStringValue(data, "name", fileCode) + ".xlsx";
        }
        var suffix = FileUtil.getSuffix(filename);
        var file = importedData.getOtherFiles().get(DIR + File.separator + fileCode + "." + suffix);
        if (file == null || !file.exists()) {
            log.info("模板文件不存在：{}", fileCode);
            return null;
        }

        jdbcTemplate.update("delete from sys_resource_byte where resource_key = :resource_key and resource_type = :resource_type",
                Map.of("resource_key", fileCode, "resource_type", ResourceByteMngManager.BUSINESS_TYPE_TMPL));

        String sql = "insert into sys_resource_byte(id, resource_type, resource_key, resource, mime_type, suffix, " +
                "resource_name, show_name, size, temp, create_time) values (:id, :resource_type, :resource_key, :resource, :mime_type," +
                ":suffix, :resource_name, :show_name, :size, :temp, :create_time);";
        Map<String, Object> sqlParams = new HashMap<>(32);
        sqlParams.put("id", idCreator.create());
        sqlParams.put("resource_type", ResourceByteMngManager.BUSINESS_TYPE_TMPL);
        sqlParams.put("resource_key", fileCode);
        sqlParams.put("resource", FileUtil.readBytes(file));
        sqlParams.put("mime_type", "xlsx".equals(suffix) ? MimeTypeConstant.XLSX.toString() : MimeTypeConstant.XLS.toString());
        sqlParams.put("suffix", suffix);
        sqlParams.put("resource_name", filename);
        sqlParams.put("show_name", filename);
        sqlParams.put("size", file.length());
        sqlParams.put("temp", 0);
        sqlParams.put("create_time", LocalDateTime.now());

        jdbcTemplate.update(sql, sqlParams);

        return file;
    }

    static class ResourceInfo {
        private String resourceKey;
        private byte[] resource;
        private String mimeType;
        private String suffix;
        private String resourceName;
        private Long size;

        public String getResourceKey() {
            return resourceKey;
        }

        public void setResourceKey(String resourceKey) {
            this.resourceKey = resourceKey;
        }

        public byte[] getResource() {
            return resource;
        }

        public void setResource(byte[] resource) {
            this.resource = resource;
        }

        public String getMimeType() {
            return mimeType;
        }

        public void setMimeType(String mimeType) {
            this.mimeType = mimeType;
        }

        public String getSuffix() {
            return suffix;
        }

        public void setSuffix(String suffix) {
            this.suffix = suffix;
        }

        public String getResourceName() {
            return resourceName;
        }

        public void setResourceName(String resourceName) {
            this.resourceName = resourceName;
        }

        public Long getSize() {
            return size;
        }

        public void setSize(Long size) {
            this.size = size;
        }
    }
}
