package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.pinyin.PinyinUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.constant.AreaType;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.convert.SysPlatformAreaConvert;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAreaDO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.AreaDetailRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.AreaMngTreeRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.save.PlatformAreaSaveVO;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAreaRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAreaRepoProc;
import com.elitescloud.cloudt.system.convert.AreaConvert;
import com.elitescloud.cloudt.system.provider.imports.param.ImportAreaBO;
import com.elitescloud.cloudt.system.service.AreaMngService;
import com.elitescloud.cloudt.system.service.manager.AreaManager;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantAreaDO;
import com.elitescloud.cloudt.system.service.repo.TenantAreaRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.Comparator;
import java.util.List;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/30
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class AreaMngServiceImpl extends BaseServiceImpl implements AreaMngService {
    private static final SysPlatformAreaConvert CONVERT = SysPlatformAreaConvert.INSTANCE;
    @Autowired
    private TenantAreaRepoProc tenantAreaRepoProc;
    @Autowired
    private SysPlatformAreaRepo areaRepo;
    @Autowired
    private SysPlatformAreaRepoProc areaRepoProc;

    @Autowired
    private AreaManager areaManager;

    @Override
    public ApiResult<Boolean> getEnabledPlatform() {
        var enabled = this.enabledPlatform();
        return ApiResult.ok(enabled);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateEnabledPlatform() {
        areaManager.clearCache();

        // 获取当前状态
        var tenantId = super.currentTenantId();
        var enabled = tenantAreaRepoProc.getEnabled(tenantId);
        if (enabled == null) {
            // 尚未设置，则新增
            SysTenantAreaDO tenantAreaDO = new SysTenantAreaDO();
            tenantAreaDO.setSysTenantId(tenantId);
            tenantAreaDO.setEnabled(tenantId != TenantConstant.DEFAULT_TENANT_ID.longValue());
            tenantAreaRepoProc.save(tenantAreaDO);

            return ApiResult.ok(tenantAreaDO.getEnabled());
        }

        // 修改
        enabled = !enabled;
        tenantAreaRepoProc.updateEnabled(tenantId, enabled);
        return ApiResult.ok(enabled);
    }

    @Override
    public ApiResult<List<AreaMngTreeRespVO>> tree(Boolean tree) {
        var enabled = this.enabledPlatform();
        List<AreaMngTreeRespVO> areaList = null;
        if (enabled || TenantConstant.DEFAULT_TENANT_ID.longValue() == super.currentTenantId()) {
            areaList = tenantDataIsolateProvider.byDefaultDirectly(() -> areaRepoProc.getTree(null, null, null, this::do2TreeRespVO));
        } else {
            areaList = areaRepoProc.getTree(null, null, null, this::do2TreeRespVO);
        }

        // 是否转树
        if (!ObjectUtil.defaultIfNull(tree, true) || areaList.isEmpty()) {
            return ApiResult.ok(areaList);
        }
        TreeDataUtil<AreaMngTreeRespVO> treeDataUtil = new TreeDataUtil<AreaMngTreeRespVO>(areaList, AreaMngTreeRespVO::getId,
                AreaMngTreeRespVO::getParentId, AreaMngTreeRespVO::setChildren, Comparator.comparingInt(AreaMngTreeRespVO::getSortNo));
        var result = (List<AreaMngTreeRespVO>) treeDataUtil.getRoots();
        return ApiResult.ok(result);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(PlatformAreaSaveVO saveVO) {
        // 校验是否有操作权限
        this.validatePermission();

        SysPlatformAreaDO areaDO = null;
        try {
            areaDO = saveVO.getId() == null ? this.checkForInsert(saveVO) : this.checkForUpdate(saveVO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        areaManager.clearCache();

        // 默认值处理
        Long parentId = StringUtils.hasText(areaDO.getParentAreaCode()) ? areaRepoProc.getId(areaDO.getParentAreaCode()) : null;
        if (areaDO.getEnabled() == null) {
            areaDO.setEnabled(true);
        }
        if (areaDO.getSortNo() == null) {
            areaDO.setSortNo(areaRepoProc.findMaxSortNo(parentId) + 1);
        }

        String oldCodePath = saveVO.getId() == null ? null : areaRepoProc.getCodePath(saveVO.getId());
        areaRepo.save(areaDO);

        // 保存树节点信息
        areaRepoProc.saveTreeNode(areaDO, parentId, areaDO.getSortNo());
        // 如果codePath有调整，则修改下级
        if (oldCodePath != null && !StrUtil.equals(oldCodePath, areaDO.getCodePath())) {
            var idAndCodePathMap = areaRepoProc.getIdAndCodePathByCodePath(oldCodePath);
            var len = oldCodePath.length();
            for (Map.Entry<Long, String> entry : idAndCodePathMap.entrySet()) {
                String newPath = areaDO.getCodePath() + entry.getValue().substring(len);
                areaRepoProc.updateCodePath(entry.getKey(), newPath);
            }
        }

        return ApiResult.ok(areaDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(ImportAreaBO areaBO) {
        // 校验是否有操作权限
        this.validatePermission();

        SysPlatformAreaDO areaDO = null;
        try {
            areaDO = this.checkForInsert(areaBO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        areaManager.clearCache();

        // 默认值处理
        Long parentId = StringUtils.hasText(areaDO.getParentAreaCode()) ? areaRepoProc.getId(areaDO.getParentAreaCode()) : null;
        if (areaDO.getEnabled() == null) {
            areaDO.setEnabled(true);
        }
        if (areaDO.getSortNo() == null) {
            areaDO.setSortNo(areaRepoProc.findMaxSortNo(parentId) + 1);
        }

        areaRepo.save(areaDO);

        // 保存树节点信息
        areaRepoProc.saveTreeNode(areaDO, parentId, areaDO.getSortNo());

        return ApiResult.ok(areaDO.getId());
    }

    @Override
    public ApiResult<AreaDetailRespVO> get(Long id) {
        var enabled = this.enabledPlatform();

        var respVO = enabled || TenantConstant.DEFAULT_TENANT_ID.longValue() == super.currentTenantId() ?
                tenantDataIsolateProvider.byDefaultDirectly(() -> this.getDetailRespVO(id)) : this.getDetailRespVO(id);
        return ApiResult.ok(respVO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        // 校验是否有操作权限
        this.validatePermission();

        // 获取节点信息
        var data = areaRepoProc.get(id);
        if (data == null) {
            return ApiResult.ok(id);
        }

        areaManager.clearCache();

        // 移除树节点
        areaRepoProc.removeTreeNode(data);
        // 删除数据
        areaRepoProc.delete(id);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> rebuildTree(Long rootId) {
        // 获取编码与id关系
        var idCodeMap = areaRepoProc.getIdAndCode();
        if (idCodeMap.isEmpty()) {
            return ApiResult.ok();
        }

        areaRepoProc.rebuildTree(rootId, t -> StringUtils.hasText(t.getParentAreaCode()) ? idCodeMap.get(t.getParentAreaCode()) : null);
        return ApiResult.ok(true);
    }

    private boolean enabledPlatform() {
        var tenantId = super.currentTenantId();
        var enabled = tenantAreaRepoProc.getEnabled(tenantId);
        return ObjectUtil.defaultIfNull(enabled, true);
    }

    private void validatePermission() {
        var tenantId = super.currentTenantId();
        if (TenantConstant.DEFAULT_TENANT_ID.longValue() != tenantId) {
            // 非默认租户，则使用平台树时无权修改
            var enabled = this.enabledPlatform();
            if (enabled) {
                throw new BusinessException("无权限修改平台默认配置");
            }
        }
    }

    private AreaMngTreeRespVO do2TreeRespVO(SysPlatformAreaDO t) {
        var respVO = new AreaMngTreeRespVO();
        respVO.setEnabled(t.getEnabled());
        respVO.setPinyin(t.getPinyin());
        respVO.setId(t.getId());
        respVO.setCode(t.getAreaCode());
        respVO.setName(t.getAreaName());
        respVO.setSortNo(t.getSortNo());
        respVO.setParentId(t.getPId());
        respVO.setParentCode(t.getParentAreaCode());

        return respVO;
    }

    private SysPlatformAreaDO checkForInsert(ImportAreaBO saveBO) {
        var areaDO = AreaConvert.INSTANCE.bo2Do(saveBO);

        // 判断编码是否已存在
        var exists = areaRepoProc.existsAreaCode(areaDO.getAreaCode(), null);
        Assert.isTrue(!exists, "地区编码已存在");

        // 上级地区
        if (StringUtils.hasText(areaDO.getParentAreaCode())) {
            exists = areaRepoProc.existsAreaCode(areaDO.getParentAreaCode(), null);
            Assert.isTrue(exists, "上级地区编码不存在");

            String parentCodePath = areaRepoProc.getCodePath(areaDO.getParentAreaCode());
            areaDO.setCodePath(parentCodePath + SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        } else {
            areaDO.setCodePath(SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        }

        // 拼音
        if (CharSequenceUtil.isBlank(saveBO.getPinyin())) {
            areaDO.setPinyin(PinyinUtil.getPinyin(saveBO.getName()));
        }

        return areaDO;
    }

    private SysPlatformAreaDO checkForInsert(PlatformAreaSaveVO saveVO) {
        var areaDO = CONVERT.saveVo2Do(saveVO);

        // 判断编码是否已存在
        var exists = areaRepoProc.existsAreaCode(areaDO.getAreaCode(), null);
        Assert.isTrue(!exists, "地区编码已存在");

        // 上级地区
        if (StringUtils.hasText(areaDO.getParentAreaCode())) {
            exists = areaRepoProc.existsAreaCode(areaDO.getParentAreaCode(), null);
            Assert.isTrue(exists, "上级地区编码不存在");

            String parentCodePath = areaRepoProc.getCodePath(areaDO.getParentAreaCode());
            areaDO.setCodePath(parentCodePath + SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        } else {
            areaDO.setCodePath(SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        }

        // 拼音
        areaDO.setPinyin(PinyinUtil.getPinyin(saveVO.getAreaName()));

        return areaDO;
    }

    private SysPlatformAreaDO checkForUpdate(PlatformAreaSaveVO saveVO) {
        var areaDO = areaRepoProc.get(saveVO.getId());
        Assert.notNull(areaDO, "修改的数据不存在");

        // 判断编码是否已存在
        if (!StrUtil.equals(saveVO.getAreaCode(), areaDO.getAreaCode())) {
            var exists = areaRepoProc.existsAreaCode(saveVO.getAreaCode(), null);
            Assert.isTrue(!exists, "地区编码已存在");
        }

        if (StrUtil.isBlank(saveVO.getParentAreaCode())) {
            // 上级为空
            areaDO.setCodePath(SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        } else if (!StrUtil.equals(saveVO.getParentAreaCode(), areaDO.getParentAreaCode())) {
            // 上级有调整
            var exists = areaRepoProc.existsAreaCode(saveVO.getParentAreaCode(), null);
            Assert.isTrue(exists, "上级地区编码不存在");

            String parentCodePath = areaRepoProc.getCodePath(areaDO.getParentAreaCode());
            areaDO.setCodePath(parentCodePath + SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        }

        // 拼音
        if (!StrUtil.equals(saveVO.getAreaName(), areaDO.getAreaName()) || StrUtil.isBlank(areaDO.getAreaName())) {
            areaDO.setPinyin(PinyinUtil.getPinyin(saveVO.getAreaName()));
        }

        CONVERT.saveVo2Do(saveVO, areaDO);
        return areaDO;
    }

    private AreaDetailRespVO getDetailRespVO(long id) {
        return areaRepoProc.getOptional(id)
                .map(t -> {
                    var respVO = CONVERT.do2DetailRespVO(t);
                    // 父节点
                    if (StringUtils.hasText(t.getParentAreaCode())) {
                        respVO.setParentAreaName(areaRepoProc.getAreaName(t.getParentAreaCode()));
                    }
                    // 节点类型
                    if (StringUtils.hasText(t.getAreaType())) {
                        respVO.setAreaTypeName(super.udcValue(new AreaType(t.getAreaType())));
                    }
                    return respVO;
                }).orElse(null);
    }
}
