package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.boot.common.annotation.businessobject.BusinessParmObjectTypeEnum;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.annotation.businessobject.OperationScopeEnum;
import com.elitescloud.boot.common.annotation.businessobject.OperationStateEnum;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.datasecurity.config.DataSecurityProperties;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.model.dto.SysBusinessOperationDTO;
import com.elitescloud.boot.provider.PlatformAppProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.swagger.openapi.model.BusinessObjectInfo;
import com.elitescloud.boot.swagger.openapi.model.BusinessObjectOperationInfo;
import com.elitescloud.boot.swagger.openapi.model.BusinessObjectParamInfo;
import com.elitescloud.boot.swagger.openapi.service.impl.OpenApiServiceUtil;
import com.elitescloud.boot.swagger.openapi.swagger3.models.Operation;
import com.elitescloud.boot.swagger.openapi.swagger3.models.media.Schema;
import com.elitescloud.boot.swagger.openapi.swagger3.models.parameters.Parameter;
import com.elitescloud.boot.util.ArrayUtil;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.LockUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.service.repo.MenuOperationRepoProc;
import com.elitescloud.cloudt.system.cacheable.SysCacheBusinessObjectRpcService;
import com.elitescloud.cloudt.system.common.BusinessObjectsConstant;
import com.elitescloud.cloudt.system.constant.SysCacheConstant;
import com.elitescloud.cloudt.system.convert.BusinessObjectConvert;
import com.elitescloud.cloudt.system.model.bo.BusinessObjectConvertBO;
import com.elitescloud.cloudt.system.model.bo.BusinessOperationParamBO;
import com.elitescloud.cloudt.system.model.vo.query.businessobject.BusinessObjectPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.businessobject.BusinessOperationPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.businessobject.*;
import com.elitescloud.cloudt.system.model.vo.save.businessobject.BusinessObjectSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.businessobject.BusinessOperationSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.businessobject.UpdateFieldPermissionFilterParam;
import com.elitescloud.cloudt.system.service.BusinessObjectMngService;
import com.elitescloud.cloudt.system.service.common.constant.*;
import com.elitescloud.cloudt.system.service.manager.BusinessObjectManager;
import com.elitescloud.cloudt.system.service.model.entity.*;
import com.elitescloud.cloudt.system.service.repo.*;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/6
 */
@Slf4j
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class BusinessObjectMngServiceImpl implements BusinessObjectMngService {
    @Autowired
    private OpenApiInfoRepoProc openApiInfoRepoProc;
    @Autowired
    private OpenApiOperationRepoProc openApiOperationRepoProc;
    @Autowired
    private OpenApiResourceRepoProc resourceRepoProc;
    @Autowired
    private OpenApiSchemaRepoProc openApiSchemaRepoProc;
    @Autowired
    private BusinessObjectRepoProc businessObjectRepoProc;
    @Autowired
    private BusinessObjectRefRepoProc refRepoProc;
    @Autowired
    private BusinessOperationRepoProc businessOperationRepoProc;
    @Autowired
    private BusinessOperationParamRepoProc businessOperationParamRepoProc;
    @Autowired
    private BusinessParamRepoProc businessParamRepoProc;
    @Autowired
    private PlatformAppProvider appProvider;
    @Autowired
    private MenuOperationRepoProc menuOperationRepoProc;

    @Autowired
    private BusinessObjectManager businessObjectManager;

    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private DataSecurityProperties dataSecurityProperties;

    @Autowired
    private PlatformTransactionManager transactionManager;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private SysCacheBusinessObjectRpcService cacheBusinessObjectRpcService;

    @Override
    public ApiResult<List<BusinessOperationParamBO>> queryBusinessOperationParam(String operationCode, BusinessObjectParamInEnum businessObjectParamInEnum) {
        List<BusinessOperationParamBO> operationParams = businessOperationParamRepoProc.listBoByOperationCode(operationCode, businessObjectParamInEnum);
        if (operationParams.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }
        List<BusinessOperationParamBO> paramVoList = businessObjectManager.convertParam(operationParams, null);
        return ApiResult.ok(filterResponseData(paramVoList));
    }

    private List<BusinessOperationParamBO> filterResponseData(List<BusinessOperationParamBO> paramList) {
        if (CollUtil.isEmpty(paramList)) {
            return Collections.emptyList();
        }
        for (var operationParam : paramList) {
            if (operationParam.isApiResultData()) {
                // 判断是否有分页查询的
                var pageData = businessObjectManager.attemptObtainPagingData(operationParam.getChildrenParams());
                return CollUtil.defaultIfEmpty(pageData, operationParam.getChildrenParams());
            }
        }
        return paramList;
    }

    @Override
    public ApiResult<Boolean> updateBusinessOperationFieldPermissionFilter(UpdateFieldPermissionFilterParam param) {
        var businessOperationParamDO = businessOperationParamRepoProc.get(param.getBusinessOperationParamId());
        businessOperationParamDO.setFieldPermissionFilter(param.getFieldPermissionFilterEnabled());
        businessOperationParamRepoProc.save(businessOperationParamDO);
        return ApiResult.ok(param.getFieldPermissionFilterEnabled());
    }

    @Override
    public ApiResult<Boolean> testAnalyzeOpenApi(Long openApiId, Long openApiOperationId, Boolean save) {
        // 查询OpenAPI记录
        var openApiDO = openApiInfoRepoProc.get(openApiId);
        if (openApiDO == null) {
            return ApiResult.fail("OpenApi记录不存在，请刷新后再试");
        }

        if (openApiOperationId == null) {
            // 测试分析整个OpenApi
            var result = this.convertOpenApi2BusinessObject(openApiDO);
            log.info("结果：{}", result.size());
        } else {
            // 测试分析单个操作
            var operationDO = openApiOperationRepoProc.get(openApiOperationId);
            if (operationDO == null) {
                return ApiResult.fail("OpenApi的操作不存在");
            }
            RootContext context = new RootContext(openApiDO);
            this.analyzeBusinessObject(context, operationDO);
            log.info("结果：{}", context.businessObjects.size());

            if (Boolean.TRUE.equals(save)) {
                this.saveBusinessObject(openApiDO, OpenApiSourceEnum.GATHER_PULL, new ArrayList<>(context.businessObjects.values()));
            }
        }
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> generateByOpenApi(Long openApiId) {
        Assert.notNull(openApiId, "OpenApiId为空");

        // 查询OpenAPI记录
        var openApiDO = openApiInfoRepoProc.get(openApiId);
        if (openApiDO == null) {
            return ApiResult.fail("OpenApi记录不存在，请刷新后再试");
        }

        // 开始生成openApi
        this.generateBusinessObject(openApiDO);

        return ApiResult.ok(openApiId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveBusinessObject(BusinessObjectSaveVO saveVO) {
        if (saveVO == null || saveVO.getId() == null) {
            return ApiResult.fail("业务对象ID为空");
        }

        var businessObjectDO = businessObjectRepoProc.get(saveVO.getId());
        if (businessObjectDO == null) {
            return ApiResult.fail("业务对象不存在");
        }
        BusinessObjectConvert.INSTANCE.copySaveVO(saveVO, businessObjectDO);
        businessObjectRepoProc.save(businessObjectDO);

        taskExecutor.execute(this::cacheBusinessObjects);
        return ApiResult.ok(saveVO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveBusinessOperation(BusinessOperationSaveVO saveVO) {
        if (saveVO == null || saveVO.getId() == null) {
            return ApiResult.fail("业务对象ID为空");
        }

        var businessOperationDO = businessOperationRepoProc.get(saveVO.getId());
        if (businessOperationDO == null) {
            return ApiResult.fail("功能操作不存在");
        }
        BusinessObjectConvert.INSTANCE.copySaveVO(saveVO, businessOperationDO);
        businessOperationRepoProc.save(businessOperationDO);

        taskExecutor.execute(this::cacheBusinessObjects);
        return ApiResult.ok(saveVO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Set<Long>> saveBusinessOperation(List<BusinessOperationSaveVO> saveVOList) {
        if (CollUtil.isEmpty(saveVOList)) {
            return ApiResult.fail("保存信息为空");
        }

        var saveVoMap = saveVOList.stream()
                .filter(t -> t.getId() != null)
                .collect(Collectors.toMap(BusinessOperationSaveVO::getId, Function.identity(), (t1, t2) -> t1));
        if (saveVoMap.isEmpty()) {
            return ApiResult.fail("保存信息的ID为空");
        }
        var businessOperations = businessOperationRepoProc.get(saveVoMap.keySet());

        for (var businessOperation : businessOperations) {
            BusinessObjectConvert.INSTANCE.copySaveVO(saveVoMap.get(businessOperation.getId()), businessOperation);
        }
        businessOperationRepoProc.save(businessOperations);

        return ApiResult.ok(saveVoMap.keySet());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateDataPermissionEnabledOfParam(Long businessParamId, Boolean enabled) {
        Assert.notNull(businessParamId, "业务参数ID为空");
        if (enabled == null) {
            enabled = false;
        }

        businessParamRepoProc.updateDataPermissionEnabled(businessParamId, enabled);
        return ApiResult.ok(businessParamId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateFieldPermissionEnabledOfParam(Long businessParamId, Boolean enabled) {
        Assert.notNull(businessParamId, "业务参数ID为空");
        if (enabled == null) {
            enabled = false;
        }

        businessParamRepoProc.updateFieldPermissionEnabled(businessParamId, enabled);
        return ApiResult.ok(businessParamId);
    }

    @Override
    public ApiResult<PagingVO<BusinessObjectPageRespVO>> pageObject(BusinessObjectPageQueryVO queryVO) {
        PagingVO<BusinessObjectPageRespVO> pageData = businessObjectRepoProc.pageMng(queryVO).map(t -> {
            var vo = BusinessObjectConvert.INSTANCE.do2PageRespVO(t);
            vo.setName(CharSequenceUtil.blankToDefault(t.getCustomName(), t.getName()));
            return vo;
        });
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        var businessObjectIds = pageData.stream().map(BusinessObjectPageRespVO::getId).collect(Collectors.toList());
        // 查询业务对象下的操作数量
        var businessOperationCountMap = businessOperationRepoProc.countByBusinessObject(businessObjectIds);
        // 应用信息
        var appMap = appProvider.all().stream().collect(Collectors.toMap(CodeNameParam::getCode, CodeNameParam::getName, (t1, t2) -> t1));

        pageData.each(t -> {
            t.setAppName(appMap.get(t.getAppCode()));
            t.setOperationNum(businessOperationCountMap.getOrDefault(t.getId(), 0L));
        });
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<BusinessObjectDetailRespVO> getDetail(Long id) {
        Assert.notNull(id, "ID为空");

        var businessObjectDO = businessObjectRepoProc.get(id);
        if (businessObjectDO == null) {
            return ApiResult.fail("业务对象不存在");
        }

        // 获取基本信息
        var respVO = BusinessObjectConvert.INSTANCE.do2DetailVO(businessObjectDO);
        respVO.setName(CharSequenceUtil.blankToDefault(businessObjectDO.getCustomName(), businessObjectDO.getName()));
        var appMap = appProvider.all().stream().collect(Collectors.toMap(CodeNameParam::getCode, CodeNameParam::getName, (t1, t2) -> t1));
        respVO.setAppName(appMap.get(respVO.getAppCode()));

        // 业务对象参数
        respVO.setParamList(this.listBusinessParams(id, BusinessObjectConvert.INSTANCE::do2DetailVO));

        // 获取操作列表
        var operationList = businessOperationRepoProc.listByBusinessObjectId(id);
        respVO.setOperationList(operationList);
        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<BusinessOperationRespVO> getBusinessOperation(Long businessOperationId) {
        Assert.notNull(businessOperationId, "ID为空");

        var operationDO = businessOperationRepoProc.get(businessOperationId);
        if (operationDO == null) {
            return ApiResult.fail("业务操作不存在");
        }

        var respVO = BusinessObjectConvert.INSTANCE.do2DetailVO(operationDO);
        respVO.setOperationName(CharSequenceUtil.blankToDefault(operationDO.getCustomName(), operationDO.getOperationDescription()));
        respVO.setDefaultName(operationDO.getOperationDescription());
        // 操作类型
        if (StringUtils.hasText(respVO.getOperationType())) {
            var operationType = OperationTypeEnum.parse(respVO.getOperationType());
            respVO.setOperationTypeName(operationType == null ? null : operationType.getDescription());
        }
        // 操作范围
        if (StringUtils.hasText(respVO.getScope())) {
            var scope = OperationScopeEnum.parse(respVO.getScope());
            respVO.setScopeName(scope == null ? null : scope.getDescription());
        }
        // 操作状态
        if (StringUtils.hasText(respVO.getState())) {
            var state = OperationStateEnum.parse(respVO.getState());
            respVO.setStateName(state == null ? null : state.getDescription());
        }

        // 出入参
        var paramList = businessOperationParamRepoProc.listBoByOperationId(businessOperationId, null);
        respVO.setRequestParams(this.obtainRequestParams(paramList));
        respVO.setResponseParams(this.obtainResponseParams(paramList));

        // 业务对象参数
        var businessParams = this.listBusinessParams(operationDO.getBusinessObjectId(), this::convert2Vo);
        respVO.setBusinessParamList(businessParams);

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<PagingVO<BusinessOperationPageRespVO>> pageOperation(BusinessOperationPageQueryVO queryVO) {
        PagingVO<BusinessOperationPageRespVO> pageData = businessOperationRepoProc.pageMng(queryVO)
                .map(t -> {
                    var respVO = BusinessObjectConvert.INSTANCE.do2PageRespVO(t);
                    respVO.setOperationName(CharSequenceUtil.blankToDefault(t.getCustomName(), t.getOperationDescription()));
                    respVO.setDefaultName(t.getOperationDescription());
                    return respVO;
                });
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        // 查询关联的菜单数量
        var businessObjectIds = pageData.stream().map(BusinessOperationPageRespVO::getBusinessObjectId).collect(Collectors.toList());
        var menuNumMap = menuOperationRepoProc.countMenuOfOperationByBusinessObjectId(businessObjectIds);

        pageData.each(t -> {
            // 操作类型
            if (StringUtils.hasText(t.getOperationType())) {
                var operationType = OperationTypeEnum.parse(t.getOperationType());
                t.setOperationTypeName(operationType == null ? null : operationType.getDescription());
            }
            // 关联菜单数量
            t.setMenuNum(menuNumMap.getOrDefault(t.getOperationCode(), 0L));
        });
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<List<BusinessPermissionParamRespVO>> listPermissionFields(Long id) {
        // 查询参数
        var params = businessParamRepoProc.listByBusinessObjectId(id, null);
        if (params.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }
        var permissionParamList = this.listBusinessParams(id, t -> {
            var respVO = BusinessObjectConvert.INSTANCE.do2PermissionVO(t);
            var javaType = JavaTypeEnum.parse(t.getFieldJavaType());
            respVO.setFieldType(javaType == null ? "对象" : javaType.getDescription());

            String[] values = StringUtils.hasText(t.getValuesJson()) ? JSONUtil.json2Obj(t.getValuesJson(), new TypeReference<String[]>() {
            }) : new String[0];
            respVO.setValues(values);

            return respVO;
        });

        return ApiResult.ok(permissionParamList);
    }

    @Override
    public ApiResult<List<BusinessPermissionParamRespVO>> listOperationPermissionFields(Long operationId) {
        if (operationId == null) {
            return ApiResult.fail("业务操作ID为空");
        }

        // 查询操作
        var operationDO = businessOperationRepoProc.get(operationId);
        if (operationDO == null) {
            return ApiResult.fail("业务操作不存在");
        }

        return this.listPermissionFields(operationDO.getBusinessObjectId());
    }

    private void cacheBusinessObjects() {
        // 查询启用状态的操作
        var businessMap = businessOperationRepoProc.allDtoList(true).stream()
                .collect(Collectors.toMap(SysBusinessOperationDTO::getOperationCode, Function.identity(), (t1, t2) -> t1));

        // 先清空现有的
        redisUtils.del(SysCacheConstant.BUSINESS_OPERATION_ALL_HASH);

        // 存入新的
        if (!businessMap.isEmpty()) {
            redisUtils.hmset(SysCacheConstant.BUSINESS_OPERATION_ALL_HASH, businessMap);
        }

        cacheBusinessObjectRpcService.clearCache();
    }

    private boolean filterBusinessParam(SysBusinessParamDO paramDO) {
        if (Boolean.FALSE.equals(paramDO.getBaseField())) {
            return true;
        }

        return dataSecurityProperties.getBaseFields().contains(paramDO.getFieldName()) ||
                dataSecurityProperties.getFieldPermission().getBaseFields().contains(paramDO.getFieldName());
    }

    private <T> List<T> listBusinessParams(long businessObjectId, Function<SysBusinessParamDO, T> mapper) {
        return businessParamRepoProc.listByBusinessObjectId(businessObjectId).stream()
                .filter(t -> SysBusinessParamDO.DEFAULT_ID == t.getBusinessOperationId() && this.filterBusinessParam(t))
                .sorted(Comparator.comparing(SysBusinessParamDO::getSortNo, Comparator.nullsLast(Integer::compareTo)))
                .map(mapper)
                .collect(Collectors.toList());
    }

    private BusinessObjectParamRespVO convert2Vo(SysBusinessParamDO paramDO) {
        var respVO = BusinessObjectConvert.INSTANCE.do2DetailVO(paramDO);
        if (StringUtils.hasText(respVO.getFieldJavaType())) {
            ObjUtil.ifNotNull(JavaTypeEnum.parse(respVO.getFieldJavaType()), t -> respVO.setFieldJavaType(t.getDescription()));
        }
        return respVO;
    }

    private List<BusinessOperationRespVO.OperationParam> obtainRequestParams(List<BusinessOperationParamBO> paramList) {
        Set<BusinessObjectParamInEnum> supportParamIn = Set.of(BusinessObjectParamInEnum.PATH, BusinessObjectParamInEnum.QUERY, BusinessObjectParamInEnum.REQUEST_BODY);

        return businessObjectManager.convertParam(paramList, t -> {
            var paramIn = BusinessObjectParamInEnum.parse(t.getParamIn());
            return paramIn != null && supportParamIn.contains(paramIn);
        }, BusinessObjectConvert.INSTANCE::bo2DetailVO, BusinessOperationRespVO.OperationParam::setChildren);
    }

    private List<BusinessOperationRespVO.OperationParam> obtainResponseParams(List<BusinessOperationParamBO> paramList) {
        return businessObjectManager.convertParam(paramList, t -> {
            var paramIn = BusinessObjectParamInEnum.parse(t.getParamIn());
            return paramIn == BusinessObjectParamInEnum.RESPONSE_BODY;
        }, BusinessObjectConvert.INSTANCE::bo2DetailVO, BusinessOperationRespVO.OperationParam::setChildren);
    }

    private void generateBusinessObject(SysOpenApiInfoDO openApiDO) {
        // 先判断是否可以生成
        var lockKey = "CloudtSystem:BusinessObject:Update:" + openApiDO.getId();
        LockUtil.executeByLock(lockKey, () -> {
            this.checkForGenerate(openApiDO);

            // 更新状态
            openApiInfoRepoProc.updateBusinessObjectStart(openApiDO.getId());
            return null;
        });

        // 转换成业务对象并保存
        CompletableFuture.runAsync(() -> {
            log.info("开始转换OpenApi至业务对象：{}", openApiDO.getId());
            var businessObjectBoList = this.convertOpenApi2BusinessObject(openApiDO);

            log.info("OpenApi转换至业务对象成功，开始持久化");
            DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            transactionDefinition.setName("OpenApi2BusinessObject");
            var transaction = transactionManager.getTransaction(transactionDefinition);
            try {
                this.saveBusinessObject(openApiDO, OpenApiSourceEnum.GATHER_PULL, businessObjectBoList);
                transactionManager.commit(transaction);
            } catch (Exception e) {
                if (!transaction.isCompleted()) {
                    transactionManager.rollback(transaction);
                }
                throw e;
            }

            taskExecutor.execute(this::cacheBusinessObjects);
        }, taskExecutor).whenComplete((v, e) -> {
            if (e == null) {
                openApiInfoRepoProc.updateBusinessObjectEnd(openApiDO.getId(), OpenApiStatusEnum.FINISHED, null);
                return;
            }
            log.error("OpenApi转换业务对象异常：{}", openApiDO.getId(), e);
            openApiInfoRepoProc.updateBusinessObjectEnd(openApiDO.getId(), OpenApiStatusEnum.FAILED, e.getMessage());
        });
    }

    private void saveBusinessObject(@NotNull SysOpenApiInfoDO apiInfoDO, @NotNull OpenApiSourceEnum source, List<BusinessObjectConvertBO> businessObjectBoList) {
        if (CollUtil.isEmpty(businessObjectBoList)) {
            // 新的没有了，则清空原来自动同步的
            this.deleteGatherForGatherByOpenApiCode(apiInfoDO.getOpenApiCode());
            return;
        }

        // 查询出OpenApi关联的业务对象信息
        Map<String, SysBusinessObjectDO> businessObjectMap = businessObjectRepoProc.listByOpenApiCode(apiInfoDO.getOpenApiCode()).stream().collect(Collectors.toMap(SysBusinessObjectDO::getCode, t -> t, (t1, t2) -> t1));
        // 查询OpenApi关联的操作
        Map<String, Map<String, SysBusinessObjectRefDO>> businessObjectRefMap = refRepoProc.listByOpenApiCode(apiInfoDO.getOpenApiCode()).stream().collect(
                Collectors.groupingBy(SysBusinessObjectRefDO::getBusinessObjectCode,
                        Collectors.mapping(Function.identity(), Collectors.toMap(t -> t.getRefType() + ":" + t.getRef(), Function.identity())))
        );
        // 查询OpenApi关联的业务对象参数信息，并根据业务对象分组
        Map<String, Map<String, SysBusinessParamDO>> businessParamMap = businessParamRepoProc.listByOpenApiCode(apiInfoDO.getOpenApiCode()).stream().collect(
                Collectors.groupingBy(SysBusinessParamDO::getBusinessObjectCode,
                        Collectors.mapping(Function.identity(), Collectors.toMap(SysBusinessParamDO::getFieldName, Function.identity())))
        );
        // 查询OpenApi关联的业务操作，并根据业务对象分组
        Map<String, Map<String, SysBusinessOperationDO>> businessOperationMap = businessOperationRepoProc.listByOpenApiCode(apiInfoDO.getOpenApiCode()).stream().collect(
                Collectors.groupingBy(SysBusinessOperationDO::getBusinessObjectCode,
                        Collectors.mapping(Function.identity(), Collectors.toMap(SysBusinessOperationDO::getOperationCode, Function.identity())))
        );
        // 查询OpenApi关联的业务操作的参数，并根据业务操作分组
        Map<String, Map<String, SysBusinessOperationParamDO>> businessOperationParamMap = businessOperationParamRepoProc.listByOpenApiCode(apiInfoDO.getOpenApiCode()).stream().collect(
                Collectors.groupingBy(SysBusinessOperationParamDO::getOpenApiOperationCode,
                        Collectors.mapping(Function.identity(), Collectors.toMap(t -> t.getParamIn() + ":" + t.getParamPath(), Function.identity()))
                )
        );

        var businessObjectSaved = new HashSet<String>(businessObjectBoList.size());
        for (BusinessObjectConvertBO businessObjectBO : businessObjectBoList) {
            // 保存业务对象
            this.saveOrUpdateBusinessObject(businessObjectBO, businessObjectMap, businessObjectRefMap,
                    businessParamMap, businessOperationMap, businessOperationParamMap, source);

            businessObjectSaved.add(businessObjectBO.getCode());
        }
        // 删除没有的业务对象
        var toDelBusinessObjectIds = businessObjectMap.values().stream()
                .filter(t -> !businessObjectSaved.contains(t.getCode()))
                .map(SysBusinessObjectDO::getId)
                .collect(Collectors.toList());
        if (!toDelBusinessObjectIds.isEmpty()) {
            businessObjectRepoProc.delete(toDelBusinessObjectIds);
            refRepoProc.deleteByBusinessObjectIds(toDelBusinessObjectIds);
            businessOperationRepoProc.deleteByBusinessObjectId(toDelBusinessObjectIds);
            businessOperationParamRepoProc.deleteByBusinessObjectId(toDelBusinessObjectIds);
            businessParamRepoProc.deleteByBusinessObject(toDelBusinessObjectIds);
        }
    }

    private void saveOrUpdateBusinessObject(BusinessObjectConvertBO businessObjectBO, Map<String, SysBusinessObjectDO> businessObjectMap,
                                            Map<String, Map<String, SysBusinessObjectRefDO>> businessObjectRefMap,
                                            Map<String, Map<String, SysBusinessParamDO>> businessParamMap,
                                            Map<String, Map<String, SysBusinessOperationDO>> businessOperationMap,
                                            Map<String, Map<String, SysBusinessOperationParamDO>> businessOperationParamMap,
                                            OpenApiSourceEnum source) {
        Assert.hasText(businessObjectBO.getCode(), "业务对象编码为空");

        var businessObjectDO = businessObjectMap.getOrDefault(businessObjectBO.getCode(), new SysBusinessObjectDO());
        BusinessObjectConvert.INSTANCE.bo2Do(businessObjectBO, businessObjectDO);
        if (businessObjectDO.getId() == null) {
            // 新增时
            businessObjectDO.setDataSource(source.name());
        }
        businessObjectRepoProc.save(businessObjectDO);

        // 保存业务对象的参数
        this.saveOrUpdateBusinessParam(businessObjectBO.getBusinessParamMap(), businessObjectDO, businessParamMap);

        // 保存业务对象的依赖
        this.saveOrUpdateBusinessObjectRef(businessObjectBO.getRefBusinessObjectOperations(), businessObjectDO, businessObjectRefMap, source);

        // 保存业务对象下的操作
        this.saveOrUpdateBusinessOperation(businessObjectBO.getOperationMap(), businessObjectDO, businessParamMap,
                businessOperationMap, businessOperationParamMap, source);
    }

    private void saveOrUpdateBusinessObjectRef(String[] refBos, SysBusinessObjectDO businessObjectDO,
                                               Map<String, Map<String, SysBusinessObjectRefDO>> businessObjectRefMap,
                                               OpenApiSourceEnum source) {
        if (ArrayUtil.isEmpty(refBos)) {
            refRepoProc.deleteByBusinessObjectIds(businessObjectDO.getId());
            return;
        }

        var refDoMap = businessObjectRefMap.getOrDefault(businessObjectDO.getCode(), Collections.emptyMap());
        var refSaved = new HashSet<String>();

        int i = 0;
        var type = BusinessObjectRefTypeEnum.BUSINESS_OPERATION;
        for (String ref : refBos) {
            if (!StringUtils.hasText(ref) || refSaved.contains(type.name() + ":" + ref)) {
                continue;
            }
            refSaved.add(type.name() + ":" + ref);

            var refDO = refDoMap.getOrDefault(type.name() + ":" + ref, new SysBusinessObjectRefDO());
            refDO.setOpenApiCode(businessObjectDO.getOpenApiCode());
            refDO.setBusinessObjectId(businessObjectDO.getId());
            refDO.setBusinessObjectCode(businessObjectDO.getCode());
            refDO.setRef(ref);
            refDO.setRefType(type.name());
            refDO.setSortNo(i++);
            if (refDO.getId() == null) {
                refDO.setDataSource(source.name());
            }

            refRepoProc.save(refDO);
        }

        // 删除业务对象下已不存在的
        var toDelRefIds = refDoMap.values().stream()
                .filter(t -> !refSaved.contains(t.getRefType() + ":" + t.getRef()))
                .map(SysBusinessObjectRefDO::getId)
                .collect(Collectors.toList());
        if (!toDelRefIds.isEmpty()) {
            refRepoProc.delete(toDelRefIds);
        }
    }

    private void saveOrUpdateBusinessParam(Map<String, BusinessObjectConvertBO.BusinessParam> businessParamBoMap, SysBusinessObjectDO businessObjectDO,
                                           Map<String, Map<String, SysBusinessParamDO>> businessParamMap) {
        if (CollUtil.isEmpty(businessParamBoMap)) {
            // 不存在参数了，则删除已存在的
            businessParamRepoProc.deleteByBusinessObject(businessObjectDO.getId(), SysBusinessParamDO.DEFAULT_ID);
            return;
        }

        var businessParamDoMap = businessParamMap.getOrDefault(businessObjectDO.getCode(), Collections.emptyMap());
        var businessParamSaved = new HashSet<String>(businessParamBoMap.size());
        int i = 0;
        for (BusinessObjectConvertBO.BusinessParam businessParamBO : businessParamBoMap.values()) {
            Assert.hasText(businessParamBO.getFieldName(), () -> "业务对象[" + businessObjectDO.getCode() + "]存在参数名称为空");
            var businessParamDO = businessParamDoMap.getOrDefault(businessParamBO.getFieldName(), new SysBusinessParamDO());
            BusinessObjectConvert.INSTANCE.bo2Do(businessParamBO, businessParamDO);
            businessParamDO.setValuesJson(ArrayUtil.isEmpty(businessParamBO.getValues()) ? null : JSONUtil.toJsonString(businessParamBO.getValues()));
            businessParamDO.setBusinessObjectId(businessObjectDO.getId());
            businessParamDO.setBusinessOperationId(SysBusinessParamDO.DEFAULT_ID);
            businessParamDO.setBusinessOperationParamId(SysBusinessParamDO.DEFAULT_ID);
            businessParamDO.setEnabled(true);
            businessParamDO.setEntityField(true);
            businessParamDO.setSortNo(i++);

            businessParamSaved.add(businessParamBO.getFieldName());
            businessParamRepoProc.save(businessParamDO);
        }

        // 删除业务对象下已不存在的
        var toDelBusinessParamIds = businessParamDoMap.values().stream()
                .filter(t -> !businessParamSaved.contains(t.getFieldName()))
                .map(SysBusinessParamDO::getId)
                .collect(Collectors.toList());
        if (!toDelBusinessParamIds.isEmpty()) {
            businessParamRepoProc.delete(toDelBusinessParamIds);
        }
    }

    private void saveOrUpdateBusinessOperation(Map<String, BusinessObjectConvertBO.Operation> operationBoMap, SysBusinessObjectDO businessObjectDO,
                                               Map<String, Map<String, SysBusinessParamDO>> businessParamMap,
                                               Map<String, Map<String, SysBusinessOperationDO>> businessOperationMap,
                                               Map<String, Map<String, SysBusinessOperationParamDO>> businessOperationParamMap,
                                               OpenApiSourceEnum source) {
        if (CollUtil.isEmpty(operationBoMap)) {
            // 不存在操作，则删除已存在的操作
            if (businessOperationMap.containsKey(businessObjectDO.getCode())) {
                businessOperationRepoProc.deleteByBusinessObjectId(businessObjectDO.getId());
                businessOperationParamRepoProc.deleteByBusinessObjectId(businessObjectDO.getId());
                businessParamRepoProc.deleteForOperation(businessObjectDO.getId());
                return;
            }
        }

        // 业务对象下的操作
        var businessOperationDoMap = businessOperationMap.getOrDefault(businessObjectDO.getCode(), Collections.emptyMap());
        var businessOperationSaved = new HashSet<String>(businessOperationMap.size());
        for (BusinessObjectConvertBO.Operation operationBO : operationBoMap.values()) {
            Assert.hasText(operationBO.getOperationCode(), () -> "[" + businessObjectDO.getCode() + "]存在操作的编码为空");
            var businessOperationDO = businessOperationDoMap.getOrDefault(operationBO.getOperationCode(), new SysBusinessOperationDO());
            BusinessObjectConvert.INSTANCE.bo2Do(operationBO, businessOperationDO);
            businessOperationDO.setBusinessObjectId(businessObjectDO.getId());
            if (businessOperationDO.getId() == null) {
                // 新增时
                businessOperationDO.setDataSource(source.name());
            }
            businessOperationRepoProc.save(businessOperationDO);

            // 保存操作的参数
            this.saveOrUpdateBusinessOperationParam(operationBO.getParamList(), businessOperationDO, businessParamMap,
                    businessOperationParamMap, source);
            businessOperationSaved.add(operationBO.getOperationCode());
        }

        // 删除业务对象下已不存在的操作
        var toDelBusinessOperationIds = businessOperationDoMap.values().stream()
                .filter(t -> !businessOperationSaved.contains(t.getOperationCode()))
                .map(SysBusinessOperationDO::getId)
                .collect(Collectors.toList());
        if (!toDelBusinessOperationIds.isEmpty()) {
            businessOperationRepoProc.delete(toDelBusinessOperationIds);
            // 删除操作对应的参数
            businessOperationParamRepoProc.deleteByBusinessOperationId(toDelBusinessOperationIds);
            businessParamRepoProc.deleteForOperation(toDelBusinessOperationIds);
        }
    }

    private void saveOrUpdateBusinessOperationParam(List<BusinessObjectConvertBO.OperationParam> paramBoList, SysBusinessOperationDO operationDO,
                                                    Map<String, Map<String, SysBusinessParamDO>> businessParamMap,
                                                    Map<String, Map<String, SysBusinessOperationParamDO>> businessOperationParamMap,
                                                    OpenApiSourceEnum source) {
        if (CollUtil.isEmpty(paramBoList)) {
            // 不存在参数，则删除已存在的参数
            if (businessOperationParamMap.containsKey(operationDO.getOperationCode())) {
                businessOperationParamRepoProc.deleteByBusinessOperationId(operationDO.getId());
                businessParamRepoProc.deleteByBusinessObject(operationDO.getBusinessObjectId(), operationDO.getId());
                return;
            }
        }

        // 目前已存在的参数
        var businessOperationParamDoMap = businessOperationParamMap.getOrDefault(operationDO.getOperationCode(), Collections.emptyMap());
        paramBoList = this.expandOperationParam(new ArrayList<>(), paramBoList);
        var paramSaved = new HashSet<String>(64);
        // BO转DO
        var paramDoList = paramBoList.stream().map(opp -> {
            Assert.hasText(opp.getParamIn(), () -> "[" + operationDO.getOperationCode() + "]存在参数paramIn为空");
            Assert.notNull(opp.getParamPath(), () -> "[" + operationDO.getOperationCode() + "]存在参数paramPath为空");
            var paramDO = businessOperationParamDoMap.getOrDefault(opp.getParamIn() + ":" + opp.getParamPath(), new SysBusinessOperationParamDO());
            BusinessObjectConvert.INSTANCE.bo2Do(opp, paramDO);
            paramDO.setBusinessObjectId(operationDO.getBusinessObjectId());
            paramDO.setBusinessOperationId(operationDO.getId());
            if (paramDO.getId() == null) {
                // 新增时
                paramDO.setDataSource(source.name());
            }

            paramSaved.add(opp.getParamIn() + ":" + opp.getParamPath());
            return paramDO;
        }).collect(Collectors.toList());
        businessOperationParamRepoProc.save(paramDoList);

        // 删除不存在的
        var toDelBusinessOperationParamIds = businessOperationParamDoMap.values().stream().
                filter(t -> !paramSaved.contains(t.getParamIn() + ":" + t.getParamPath()))
                .map(SysBusinessOperationParamDO::getId)
                .collect(Collectors.toList());
        if (!toDelBusinessOperationParamIds.isEmpty()) {
            businessOperationParamRepoProc.delete(toDelBusinessOperationParamIds);
        }

        // 业务参数
        var businessParamDoMap = businessParamMap.getOrDefault(operationDO.getBusinessObjectCode(), Collections.emptyMap());
        if (!businessParamDoMap.isEmpty()) {
            // 需要删除的
            for (SysBusinessParamDO businessParamDO : businessParamDoMap.values()) {
                if (toDelBusinessOperationParamIds.contains(businessParamDO.getBusinessOperationParamId())) {
                    businessParamRepoProc.delete(businessParamDO.getId());
                }
            }
        }
    }

    private void deleteGatherForGatherByOpenApiCode(String openApiCode) {
        log.info("根据OpenApiId删除自动抓取的业务对象：{}", openApiCode);
        businessObjectRepoProc.deleteForGatherByOpenApiCode(openApiCode);
        businessOperationRepoProc.deleteForGatherByOpenApiCode(openApiCode);
        businessOperationParamRepoProc.deleteForGatherByOpenApiCode(openApiCode);
        businessParamRepoProc.deleteForGatherByOpenApiCode(openApiCode);
    }

    private List<BusinessObjectConvertBO.OperationParam> expandOperationParam(List<BusinessObjectConvertBO.OperationParam> paramAll, List<BusinessObjectConvertBO.OperationParam> paramRoot) {
        if (paramAll == null) {
            paramAll = new ArrayList<>();
        }

        if (paramRoot != null) {
            for (BusinessObjectConvertBO.OperationParam param : paramRoot) {
                paramAll.add(param);
                this.expandOperationParam(paramAll, param.getItemList());
            }
        }

        return paramAll;
    }

    private List<BusinessObjectConvertBO> convertOpenApi2BusinessObject(SysOpenApiInfoDO openApi) {
        // 查询含有业务对象的api接口
        var openApiOperationDoList = openApiOperationRepoProc.listByOpenApiId(openApi.getId()).stream()
                .filter(t -> {
                            if (Boolean.FALSE.equals(t.getEnabled())) {
                                // 已禁用
                                return false;
                            }
                            if (!Boolean.TRUE.equals(t.getBusinessOperation())) {
                                // 支持业务操作
                                return false;
                            }

                            // 仅支持自动的，手动的不更新
                            return OpenApiSourceEnum.GATHER_PULL.name().equals(t.getDataSource()) || OpenApiSourceEnum.GATHER_PUSH.name().equals(t.getDataSource());
                        }
                )
                .collect(Collectors.toList());
        if (openApiOperationDoList.isEmpty()) {
            // 不包含支持业务对象的api接口
            return Collections.emptyList();
        }

        // 遍历api的方法
        RootContext context = new RootContext(openApi);
        openApiOperationDoList.forEach(t -> analyzeBusinessObject(context, t));

        return new ArrayList<>(context.businessObjects.values());
    }

    private void analyzeBusinessObject(@NotNull BusinessObjectMngServiceImpl.RootContext context, @NotNull SysOpenApiOperationDO apiOperationDO) {
        // 先解析业务对象信息
        if (context.businessObjects.isEmpty()) {
            context.businessObjects.putAll(this.queryBusinessObject(context.openApiDO));
        }
        if (context.businessObjects.isEmpty()) {
            // 业务对象为空，就没必要再查下业务操作了
            return;
        }

        // 反序列化OpenApi的API操作信息
        var apiOperation = apiOperationDO.getResourceId() == null ? null :
                JSONUtil.jsonBytes2Obj(resourceRepoProc.getResourceBytes(apiOperationDO.getResourceId()), Operation.class);
        var businessObjectContext = new BusinessOperationContext(context, apiOperationDO, apiOperation);

        // 解析业务操作
        this.obtainBusinessOperation(businessObjectContext);
    }

    private Map<String, BusinessObjectConvertBO> queryBusinessObject(SysOpenApiInfoDO openApiDO) {
        if (openApiDO.getBusinessResourceId() == null) {
            return Collections.emptyMap();
        }

        // 查询序列化后的业务对象资源
        var resources = resourceRepoProc.getResourceBytes(openApiDO.getBusinessResourceId());
        if (ArrayUtil.isEmpty(resources)) {
            throw new BusinessException("未查询到业务对象资源：" + openApiDO.getId());
        }

        // 反序列化为业务对象
        Map<String, BusinessObjectInfo> businessObjectInfoMap = JSONUtil.jsonBytes2Obj(resources, new TypeReference<>() {
        }, false, () -> "从OpenAPI中读取业务对象信息失败");
        if (CollUtil.isEmpty(businessObjectInfoMap)) {
            return Collections.emptyMap();
        }

        // 业务对象转换
        Map<String, BusinessObjectConvertBO> result = new HashMap<>(businessObjectInfoMap.size());
        BusinessObjectConvertBO businessObjectBO = null;
        String businessObjectCode = null;
        for (Map.Entry<String, BusinessObjectInfo> entry : businessObjectInfoMap.entrySet()) {
            businessObjectCode = entry.getKey().split(":")[0];
            if (result.containsKey(businessObjectCode)) {
                continue;
            }

            businessObjectBO = new BusinessObjectConvertBO();
            businessObjectBO.setOpenApiCode(openApiDO.getOpenApiCode());
            businessObjectBO.setAppCode(openApiDO.getAppCode());
            businessObjectBO.setCode(businessObjectCode);

            var tempArray = entry.getValue().getBusinessType().split(":");
            if (tempArray.length > 1) {
                businessObjectBO.setName(tempArray[1]);
            }
            businessObjectBO.setDescription(entry.getValue().getBusinessDescription());
            businessObjectBO.setDomain(entry.getValue().getDomain());
            businessObjectBO.setEnabled(true);
            businessObjectBO.setRefBusinessObjectOperations(entry.getValue().getRefBusinessObjectOperations());
            businessObjectBO.setPublicResource(entry.getValue().getPublicResource());
            businessObjectBO.setSupportTenant(entry.getValue().getSupportTenant());
            var businessParams = this.convertBusinessParam(openApiDO.getOpenApiCode(), businessObjectBO.getCode(), entry.getValue().getParamInfoList());
            businessObjectBO.setBusinessParamMap(businessParams);
            if (!businessParams.isEmpty()) {
                businessObjectBO.setEntityClassName(businessParams.values().stream().findFirst().map(BusinessObjectConvertBO.BusinessParam::getEntityClassName).orElse(null));
            }
            businessObjectBO.setOperationMap(new HashMap<>(32));

            result.put(businessObjectCode, businessObjectBO);
        }
        return result;
    }

    private Map<String, BusinessObjectConvertBO.BusinessParam> convertBusinessParam(String openApiCode, String businessObjectCode, List<BusinessObjectParamInfo> paramInfoList) {
        if (CollUtil.isEmpty(paramInfoList)) {
            return Collections.emptyMap();
        }

        Map<String, BusinessObjectConvertBO.BusinessParam> paramMap = new LinkedHashMap<>(64);
        BusinessObjectConvertBO.BusinessParam paramBO = null;
        for (BusinessObjectParamInfo paramInfo : paramInfoList) {
            if (CollUtil.isEmpty(paramInfo.getParamList())) {
                continue;
            }
            for (BusinessObjectParamInfo.Param param : paramInfo.getParamList()) {
                Assert.hasText(param.getFieldName(), () -> "OpenApi[" + openApiCode + "]的业务对象存在参数名称为空");
                if (paramMap.containsKey(param.getFieldName())) {
                    continue;
                }
                paramBO = new BusinessObjectConvertBO.BusinessParam();
                paramBO.setOpenApiCode(openApiCode);
                paramBO.setBusinessObjectCode(businessObjectCode);
                paramBO.setEntityClassName(paramInfo.getClassName());
                paramBO.setTableName(paramInfo.getTableName());
                paramBO.setFieldName(param.getFieldName());
                paramBO.setColumnName(param.getColumnName());
                paramBO.setFieldDescription(param.getFieldComment());
                paramBO.setFieldJavaType(param.getJavaType());
                paramBO.setSingle(param.isSingle());
                paramBO.setEnumeration(param.isEnumeration());
                paramBO.setEnumType(param.getEnumType());
                paramBO.setRelatedBusinessObject(param.getRelatedBusinessObject());
                paramBO.setRelatedField(param.getRelatedField());
                paramBO.setValues(param.getValues());
                paramBO.setUserType(param.getUserType());
                paramBO.setDataPermissionEnabled(param.isDataPermissionEnabled());
                paramBO.setFieldPermissionEnabled(param.isFieldPermissionEnabled());
                paramBO.setBaseField(param.isBaseField());
                paramBO.setDeprecated(param.isDeprecated());

                paramMap.put(param.getFieldName(), paramBO);
            }
        }

        return paramMap;
    }

    private BusinessObjectConvertBO.Operation obtainBusinessOperation(@NotNull BusinessOperationContext operationContext) {
        // 从扩展配置中获取业务对象的操作信息
        BusinessObjectOperationInfo businessObjectOperationInfo = JSONUtil.convertObj(operationContext.openApiOperationExtensions.get(OpenApiServiceUtil.X_BUSINESS_OBJECT_OPERATION), BusinessObjectOperationInfo.class, true);
        if (businessObjectOperationInfo == null) {
            businessObjectOperationInfo = new BusinessObjectOperationInfo();
            businessObjectOperationInfo.setEnabled(false);
            businessObjectOperationInfo.setBusinessObjectType("1:1");
            businessObjectOperationInfo.setBusinessParamObjectType(BusinessParmObjectTypeEnum.ALL);
            businessObjectOperationInfo.setBusinessParamValueKey("");
            businessObjectOperationInfo.setOperationCode("");
            businessObjectOperationInfo.setOperationType(OperationTypeEnum.ADD);
            businessObjectOperationInfo.setOperationDescription("");
            businessObjectOperationInfo.setDetail("");
            businessObjectOperationInfo.setBusinessParam("");
            businessObjectOperationInfo.setOperationScopeEnum(OperationScopeEnum.PUBLIC);
            businessObjectOperationInfo.setOperationStateEnum(OperationStateEnum.RELEASED);
            businessObjectOperationInfo.setLogEnabled(false);
            businessObjectOperationInfo.setArgsJsonEnabled(false);
            businessObjectOperationInfo.setResultJsonEnabled(false);
            businessObjectOperationInfo.setAllowRepeatRequest(false);
            businessObjectOperationInfo.setRepeatIntervalMillis(0);
            businessObjectOperationInfo.setDataPermissionEnabled(false);
            businessObjectOperationInfo.setFieldPermissionAutoFilter(false);
            businessObjectOperationInfo.setPermissionRef("");

        }

        var apiOperationDO = operationContext.openApiOperationDO;
        Assert.notNull(businessObjectOperationInfo, () -> "OpenApi的操作[" + apiOperationDO.getId() + "]的扩展信息中业务操作不存在");
        var operationCode = apiOperationDO.getOperationCode();
        Assert.hasText(operationCode, () -> "OpenApi的操作[" + apiOperationDO.getId() + "]的业务对象操作编码为空");
        var businessObjectCode = businessObjectOperationInfo.getBusinessObjectType().split(":")[0];
        var businessObjectBO = operationContext.rootContext.businessObjects.get(businessObjectCode);
        Assert.notNull(businessObjectBO, "业务对象[" + businessObjectCode + "]不存在");
        operationContext.businessObjectBO = businessObjectBO;

        BusinessObjectOperationInfo finalBusinessObjectOperationInfo = businessObjectOperationInfo;
        Assert.isTrue(!businessObjectBO.getOperationMap().containsKey(operationCode),
                () -> "在业务对象[" + businessObjectBO.getCode() + "]存在重复的业务操作编码:"
                        + finalBusinessObjectOperationInfo.getOperationCode());

        // 转换业务操作
        var operationBO = new BusinessObjectConvertBO.Operation();
        operationBO.setOpenApiCode(businessObjectBO.getOpenApiCode());
        operationBO.setBusinessObjectCode(businessObjectCode);
        operationBO.setAppCode(businessObjectBO.getAppCode());
        operationBO.setOperationCode(operationCode);
        operationBO.setOperationDescription(businessObjectOperationInfo.getOperationDescription());
        operationBO.setDetail(businessObjectOperationInfo.getDetail());
        ObjUtil.ifNotNull(businessObjectOperationInfo.getOperationType(), t -> operationBO.setOperationType(t.name()));
        ObjUtil.ifNotNull(businessObjectOperationInfo.getOperationScopeEnum(), t -> operationBO.setScope(t.name()));
        ObjUtil.ifNotNull(businessObjectOperationInfo.getOperationStateEnum(), t -> operationBO.setState(t.name()));
        operationBO.setPermissionEnabled(businessObjectOperationInfo.isDataPermissionEnabled());
        operationBO.setFieldPermissionAutoFilter(businessObjectOperationInfo.isFieldPermissionAutoFilter());

        operationBO.setApiName(apiOperationDO.getSummary());
        operationBO.setApiDescription(apiOperationDO.getDescription());
        operationBO.setApiUrl(apiOperationDO.getUrl());
        operationBO.setApiMethod(apiOperationDO.getMethod());
        operationBO.setEnabled(businessObjectOperationInfo.isEnabled());
        operationBO.setParamList(new ArrayList<>());
        operationBO.setPermissionRef(businessObjectOperationInfo.getPermissionRef());
        operationContext.businessOperationBO = operationBO;

        // 获取参数
        this.fillBusinessOperationParams(operationContext, operationBO);

        businessObjectBO.getOperationMap().put(operationBO.getOperationCode(), operationBO);
        return operationBO;
    }

    private void fillBusinessOperationParams(@NotNull BusinessOperationContext operationContext, BusinessObjectConvertBO.Operation businessOperationBO) {
        var apiOperation = operationContext.openApiOperation;
        var apiOperationDO = operationContext.openApiOperationDO;
        // 入参
        if (CollUtil.isNotEmpty(apiOperation.getParameters())) {
            for (Parameter apiParam : apiOperation.getParameters()) {
                BusinessObjectConvertBO.OperationParam paramBO = new BusinessObjectConvertBO.OperationParam();
                paramBO.setOpenApiCode(operationContext.businessObjectBO.getOpenApiCode());
                paramBO.setBusinessObjectCode(operationContext.businessObjectBO.getCode());
                paramBO.setOpenApiOperationCode(apiOperationDO.getOperationCode());
                paramBO.setBusinessOperationCode(businessOperationBO.getOperationCode());
                paramBO.setFieldName(apiParam.getName());
                paramBO.setFieldDescription(apiParam.getDescription());
                paramBO.setRequest(true);

                Assert.hasText(apiParam.getIn(), () -> "OpenApi的操作[" + apiOperationDO.getId() + "]的参数[" + apiParam.getName() + "]的类型为空");
                paramBO.setFieldType(this.convert2FieldType(apiParam.getSchema()));
                paramBO.setFieldJavaType(this.convert2JavaType(paramBO.getFieldType()));

                paramBO.setRef(false);
                paramBO.setRefValue(null);
                paramBO.setDepth(1);
                paramBO.setParamPath(paramBO.getFieldName() == null ? "" : paramBO.getFieldName());
                paramBO.setApiResult(false);
                paramBO.setRequired(Boolean.TRUE.equals(apiParam.getRequired()));
                paramBO.setDeprecated(Boolean.TRUE.equals(apiParam.getDeprecated()));
                paramBO.setParamIn(apiParam.getIn());
                paramBO.setSingle(true);
                paramBO.setCollectionType(null);
                paramBO.setEnabled(true);

                businessOperationBO.getParamList().add(paramBO);
            }
        }
        if (apiOperation.getRequestBody() != null) {
            var body = this.obtainRequestBody(apiOperation);
            Assert.notNull(body, () -> "OpenApi的操作[" + apiOperationDO.getId() + "]请求体为空");
            var paramBO = this.convertBody(operationContext, body, BusinessObjectParamInEnum.REQUEST_BODY, true, false, false, null, null, null);
            businessOperationBO.getParamList().addAll(paramBO);
        }
        // 返参
        if (apiOperation.getResponses() != null) {
            var body = this.obtainResponseBody(apiOperation);
            if (body != null) {
                var paramBO = this.convertBody(operationContext, body, BusinessObjectParamInEnum.RESPONSE_BODY, false, false, false, null, null, null);
                businessOperationBO.getParamList().addAll(paramBO);
            }
        }
    }

    private List<BusinessObjectConvertBO.OperationParam> convertBody(@NotNull BusinessOperationContext operationContext, Schema<?> schema,
                                                                     BusinessObjectParamInEnum paramIn, boolean isRequestParam,
                                                                     boolean isApiResult, boolean isPagingVO, String parentRef, String parentPath, Integer depth) {
        var operation = operationContext.openApiOperation;
        if (schema == null) {
            log.debug("{}出现schema为空", operation.getOperationId());
            return Collections.emptyList();
        }

        if (BusinessObjectsConstant.OPENAPI_SCHEMA_TYPE_OBJECT.equals(schema.getType())) {
            // 对象
            var properties = schema.getProperties();
            if (CollUtil.isNotEmpty(properties)) {
                boolean finalIsApiResult = isApiResult || isApiResult(schema);
                boolean finalIsPagingVO = isPagingVO || isPagingVO(schema);
                return properties.entrySet().stream().map(entry -> this.convertSchema2Param(operationContext, entry.getValue(), entry.getKey(), paramIn, isRequestParam, finalIsApiResult, finalIsPagingVO,
                                parentRef, parentPath, depth))
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList());
            }
        }
        if (BusinessObjectsConstant.OPENAPI_SCHEMA_TYPE_ARRAY.equals(schema.getType())) {
            // 集合
            var param = this.convertSchema2Param(operationContext, schema, schema.getName(), paramIn, isRequestParam, isApiResult, isPagingVO, parentRef, parentPath, depth);
            return param == null ? Collections.emptyList() : List.of(param);
        }

        var param = this.convertSchema2Param(operationContext, schema, schema.getName(), paramIn, isRequestParam, isApiResult, isPagingVO, parentRef, parentPath, depth);
        return param == null ? Collections.emptyList() : List.of(param);
    }

    private BusinessObjectConvertBO.OperationParam convertSchema2Param(@NotNull BusinessOperationContext operationContext, Schema<?> schema,
                                                                       String schemaName, BusinessObjectParamInEnum paramIn,
                                                                       boolean isRequestParam, boolean isApiResult, boolean isPagingVO,
                                                                       String parentRef, String parentPath, Integer depth) {
        if (schema == null) {
            return null;
        }
        var isArray = BusinessObjectsConstant.OPENAPI_SCHEMA_TYPE_ARRAY.equals(schema.getType());
        BusinessObjectConvertBO.OperationParam param = new BusinessObjectConvertBO.OperationParam();
        param.setOpenApiCode(operationContext.businessObjectBO.getOpenApiCode());
        param.setOpenApiOperationCode(operationContext.openApiOperationDO.getOperationCode());
        param.setBusinessObjectCode(operationContext.businessObjectBO.getCode());
        param.setBusinessOperationCode(operationContext.businessOperationBO.getOperationCode());
        param.setFieldName(schemaName);
        param.setFieldDescription(schema.getDescription());
        param.setRequest(isRequestParam);
        param.setFieldType(convert2FieldType(schema));
        param.setFieldJavaType(this.convert2JavaType(param.getFieldType()));
        param.setRef(StringUtils.hasText(schema.get$ref()));
        param.setRefValue(schema.get$ref());
        param.setDepth(depth == null ? 1 : depth + 1);
        var pathName = CharSequenceUtil.blankToDefault(param.getFieldName(), "");
        param.setParamPath(parentPath == null ? pathName : parentPath + ":" + pathName);
        param.setApiResult(isApiResult);
        param.setPagingResult(isPagingVO);
        param.setRequired(Boolean.FALSE.equals(schema.getNullable()));
        param.setDeprecated(Boolean.TRUE.equals(schema.getDeprecated()));
        param.setParamIn(paramIn.toString());
        param.setSingle(!isArray);

        if (StringUtils.hasText(schema.get$ref())) {
            if (operationContext.resolvedComponents.contains(schema.get$ref())) {
                return null;
            }
            operationContext.resolvedComponents.add(schema.get$ref());
            if (schema.get$ref().equals(parentRef)) {
                // 避免出现死循环
                return null;
            }

            var openApiDO = operationContext.rootContext.openApiDO;
            var refSchema = operationContext.rootContext.componentSchemaCache.computeIfAbsent(schema.get$ref(), ref -> {
                var resourceId = openApiSchemaRepoProc.getResourceId(openApiDO.getId(), ref);
                Assert.notNull(resourceId, () -> "OpenApi[" + openApiDO.getId() + "]存在ComponentSchema的资源ID为空，ref=" + schema.get$ref());

                var resource = resourceRepoProc.getResourceBytes(resourceId);
                Assert.isTrue(resource != null && resource.length > 0, () -> "OpenApi[" + openApiDO.getId() + "]存在ComponentSchema的资源为空，ref=" + schema.get$ref());
                return JSONUtil.jsonBytes2Obj(resource, Schema.class);
            });
            if (refSchema != null && CharSequenceUtil.isBlank(param.getFieldDescription())) {
                param.setFieldDescription(refSchema.getDescription());
            }
            param.setItemList(this.convertBody(operationContext, refSchema, paramIn, isRequestParam, false, false, schema.get$ref(), param.getParamPath(), param.getDepth()));
        } else {
            param.setItemList(this.convertBody(operationContext, schema.getItems(), paramIn, isRequestParam, false, false, parentRef, param.getParamPath(), param.getDepth()));
        }

        param.setCollectionType(null);
        param.setEnabled(true);

        return param;
    }

    private boolean isApiResult(Schema<?> schema) {
        if (CollUtil.isEmpty(schema.getProperties())) {
            return false;
        }

        var properties = schema.getProperties();
        return properties.containsKey("code") && properties.containsKey("msg") && properties.containsKey("success")
                && properties.containsKey("errorNo") && properties.containsKey("data") && properties.containsKey("time");
    }

    public boolean isPagingVO(Schema<?> schema) {
        if (CollUtil.isEmpty(schema.getProperties())) {
            return false;
        }

        var properties = schema.getProperties();
        return properties.containsKey("total") && properties.containsKey("records");
    }

    private Schema<?> obtainRequestBody(Operation operation) {
        Assert.notNull(operation, "操作为空");
        var requestBody = operation.getRequestBody();
        Assert.notNull(requestBody, "请求头为空");
        var content = requestBody.getContent().get(MediaType.APPLICATION_JSON_VALUE);

        return content == null ? null : content.getSchema();
    }

    private Schema<?> obtainResponseBody(Operation operation) {
        Assert.notNull(operation, "操作为空");
        var responses = operation.getResponses();
        if (responses.isEmpty()) {
            return null;
        }

        // 获取成功时的响应
        var responseOk = responses.get(HttpStatus.OK.value() + "");
        if (responseOk == null) {
            return null;
        }
        var responseContent = responseOk.getContent();
        if (responseContent == null) {
            return null;
        }
        var mediaType = responseContent.get(MediaType.APPLICATION_JSON_VALUE);
        return mediaType == null ? null : mediaType.getSchema();
    }

    private String convert2FieldType(Schema<?> schema) {
        if (schema == null) {
            return "";
        }

        if (CharSequenceUtil.isNotBlank(schema.get$ref())) {
            return BusinessObjectsConstant.OPENAPI_FIELD_TYPE_REF;
        }

        var type = schema.getType();
        if (CharSequenceUtil.isBlank(type)) {
            return BusinessObjectsConstant.OPENAPI_FIELD_TYPE_STRING;
        }

        if (CharSequenceUtil.isNotBlank(schema.getFormat())) {
            type = type + ":" + schema.getFormat();
        }
        return type;
    }

    private String convert2JavaType(String fieldType) {
        switch (fieldType) {
            case BusinessObjectsConstant.OPENAPI_FIELD_TYPE_STRING:
                return String.class.getName();
            case BusinessObjectsConstant.OPENAPI_FIELD_TYPE_LONG:
                return Long.class.getName();
            case BusinessObjectsConstant.OPENAPI_FIELD_TYPE_INTEGER:
                return Integer.class.getName();
            case BusinessObjectsConstant.OPENAPI_FIELD_TYPE_LOCAL_DATETIME:
                return LocalDateTime.class.getName();
            case BusinessObjectsConstant.OPENAPI_FIELD_TYPE_LOCAL_DATE:
                return LocalDate.class.getName();
            case BusinessObjectsConstant.OPENAPI_FIELD_TYPE_BOOLEAN:
                return Boolean.class.getName();
        }
        return null;
    }

    private void checkForGenerate(SysOpenApiInfoDO openApiDO) {
        var state = OpenApiStatusEnum.parse(openApiDO.getState());
        if (state != OpenApiStatusEnum.FINISHED) {
            throw new BusinessException("请等待OpenApi更新完成后再操作");
        }

        var businessObjectState = OpenApiStatusEnum.parse(openApiDO.getBusinessObjectState());
        if (businessObjectState == OpenApiStatusEnum.UPDATING) {
            throw new BusinessException("正在更新中，请稍后后再操作");
        }
    }

    static class RootContext {
        private final SysOpenApiInfoDO openApiDO;
        private final Map<String, Schema<?>> componentSchemaCache;
        private final Map<String, BusinessObjectConvertBO> businessObjects;

        public RootContext(@NotNull SysOpenApiInfoDO openApiDO) {
            Assert.notNull(openApiDO, "OpenApiDO信息为空");

            this.openApiDO = openApiDO;
            this.componentSchemaCache = new HashMap<>();
            this.businessObjects = new HashMap<>();
        }
    }

    static class BusinessOperationContext {
        private final RootContext rootContext;
        private final SysOpenApiOperationDO openApiOperationDO;
        private final Operation openApiOperation;
        private final Map<String, Object> openApiOperationExtensions;
        private final Set<String> resolvedComponents;

        private BusinessObjectConvertBO businessObjectBO;
        private BusinessObjectConvertBO.Operation businessOperationBO;

        public BusinessOperationContext(@NotNull RootContext rootContext, @NotNull SysOpenApiOperationDO openApiOperationDO, @NotNull Operation openApiOperation) {
            Assert.notNull(openApiOperation, () -> "OpenApi的操作[" + openApiOperationDO.getId() + "]的资源为空");
            this.openApiOperationExtensions = openApiOperation.getExtensions();
            Assert.notNull(openApiOperationExtensions, () -> "OpenApi的操作[" + openApiOperationDO.getId() + "]的扩展信息为空");

            this.rootContext = rootContext;
            this.openApiOperationDO = openApiOperationDO;
            this.openApiOperation = openApiOperation;
            this.resolvedComponents = new HashSet<>();
        }
    }
}
