package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.model.dto.SysBusinessOperationDTO;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.common.BusinessObjectNodeType;
import com.elitescloud.cloudt.system.constant.SysCacheConstant;
import com.elitescloud.cloudt.system.model.bo.BusinessRefBO;
import com.elitescloud.cloudt.system.model.vo.resp.businessobject.BusinessObjectTreeNodeRespVO;
import com.elitescloud.cloudt.system.service.BusinessObjectQueryService;
import com.elitescloud.cloudt.system.service.model.bo.AppBO;
import com.elitescloud.cloudt.system.service.repo.AppRepoProc;
import com.elitescloud.cloudt.system.service.repo.BusinessObjectRefRepoProc;
import com.elitescloud.cloudt.system.service.repo.BusinessObjectRepoProc;
import com.elitescloud.cloudt.system.service.repo.BusinessOperationRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/19
 */
@Slf4j
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class BusinessObjectQueryServiceImpl extends BaseServiceImpl implements BusinessObjectQueryService {

    @Autowired
    private BusinessObjectRepoProc businessObjectRepoProc;
    @Autowired
    private BusinessOperationRepoProc businessOperationRepoProc;
    @Autowired
    private BusinessObjectRefRepoProc refRepoProc;
    @Autowired
    private AppRepoProc appRepoProc;

    @Autowired
    private TenantClientProvider tenantClientProvider;

    @Override
    public ApiResult<List<BusinessObjectTreeNodeRespVO>> businessObjectTree(Boolean tree, Boolean withApp, Boolean withOperation,
                                                                            Boolean withRef, Boolean permissionOnly) {
        if (tree == null) {
            tree = true;
        }
        if (withApp == null) {
            withApp = true;
        }
        if (withOperation == null) {
            withOperation = true;
        }
        if (withRef == null) {
            withRef = false;
        }
        if (permissionOnly == null) {
            permissionOnly = false;
        }

        var businessObjectList = this.queryBusinessObjectTree(tree, withApp, withOperation, withRef, permissionOnly);
        return ApiResult.ok(businessObjectList);
    }

    @Override
    public ApiResult<List<SysBusinessOperationDTO>> allOperationDTO() {
        var dtoList = businessOperationRepoProc.allDtoList(true);

        // 加入缓存
        this.cacheBusinessObjects(dtoList);
        return ApiResult.ok(dtoList);
    }

    private void cacheBusinessObjects(List<SysBusinessOperationDTO> dtoList) {
        // 查询启用状态的操作
        var businessMap = dtoList.stream()
                .collect(Collectors.toMap(SysBusinessOperationDTO::getOperationCode, Function.identity(), (t1, t2) -> t1));
        ;

        // 先清空现有的
        redisUtils.del(SysCacheConstant.BUSINESS_OPERATION_ALL_HASH);

        // 存入新的
        if (!businessMap.isEmpty()) {
            redisUtils.hmset(SysCacheConstant.BUSINESS_OPERATION_ALL_HASH, businessMap);
        }
    }

    private List<BusinessObjectTreeNodeRespVO> queryBusinessObjectTree(boolean tree, boolean withApp, boolean withOperation,
                                                                       boolean withRef, boolean permissionOnly) {
        // 租户应用信息
        var tenantApps = this.tenantApps();
        if (tenantApps.isEmpty()) {
            return Collections.emptyList();
        }

        // 业务对象信息
        var businessObjectList = this.queryBusinessObjectTreeNode(tenantApps, withApp);
        if (businessObjectList.isEmpty()) {
            return Collections.emptyList();
        }
        List<BusinessObjectTreeNodeRespVO> respVoList = new ArrayList<>(128);

        // 应用信息
        if (withApp) {
            // 将业务对象挂到app下面
            var businessObjectAppMap = businessObjectList.stream().collect(Collectors.groupingBy(BusinessObjectTreeNodeRespVO::getParentCode));
            var appNodeList = this.convertApp2BusinessObjectTreeNode(tenantApps);
            for (BusinessObjectTreeNodeRespVO appNode : appNodeList) {
                var businessObjects = businessObjectAppMap.get(appNode.getCode());
                if (CollUtil.isEmpty(businessObjects)) {
                    continue;
                }
                appNode.setChildren(businessObjects);
                respVoList.add(appNode);
            }
        } else {
            respVoList.addAll(businessObjectList);
        }

        // 业务操作信息
        if (withOperation) {
            var operationList = this.queryBusinessOperationTreeNode(tenantApps, permissionOnly);
            var operationsMap = operationList.stream()
                    .collect(Collectors.groupingBy(BusinessObjectTreeNodeRespVO::getParentId));
            var businessObjectMap = businessObjectList.stream()
                    .collect(Collectors.toMap(BusinessObjectTreeNodeRespVO::getId, Function.identity(), (t1, t2) -> t1));
            for (Map.Entry<Long, List<BusinessObjectTreeNodeRespVO>> entry : operationsMap.entrySet()) {
                var businessObject = businessObjectMap.get(entry.getKey());
                if (businessObject != null) {
                    businessObject.setChildren(entry.getValue());
                }
            }

            // 关联的
            if (withRef) {
                this.queryRefForTree(businessObjectList, operationList);
            }
        } else if (permissionOnly) {
            var permissionEnabledMap = businessOperationRepoProc.countBusinessObjectOfPermissionEnabled();
            for (BusinessObjectTreeNodeRespVO businessObjectTreeNodeRespVO : businessObjectList) {
                businessObjectTreeNodeRespVO.setHasChildren(permissionEnabledMap.getOrDefault(businessObjectTreeNodeRespVO.getId(), 0L) > 0);
            }
        }

        if (permissionOnly) {
            respVoList = this.filteredHasChildren(respVoList);
        }
        if (respVoList.isEmpty() || tree) {
            return respVoList;
        }

        return CollectionUtil.expandTree(respVoList, BusinessObjectTreeNodeRespVO::getChildren);
    }

    private void queryRefForTree(List<BusinessObjectTreeNodeRespVO> businessObjectList, List<BusinessObjectTreeNodeRespVO> operationList) {
        var refMap = refRepoProc.listSimpleBo().stream().collect(Collectors.groupingBy(BusinessRefBO::getBusinessObjectCode));
        if (!refMap.isEmpty()) {
            var operationMap = operationList.stream().collect(Collectors.toMap(BusinessObjectTreeNodeRespVO::getCode, Function.identity(), (t1, t2) -> t1));

            for (BusinessObjectTreeNodeRespVO businessObject : businessObjectList) {
                var refList = refMap.getOrDefault(businessObject.getCode(), Collections.emptyList());
                if (!refList.isEmpty()) {
                    var refVoList = refList.stream()
                            .filter(t -> operationMap.containsKey(t.getRef()))
                            .map(t -> {
                                var operation = operationMap.get(t.getRef());
                                BusinessObjectTreeNodeRespVO respVO = new BusinessObjectTreeNodeRespVO();
                                respVO.setNodeType(BusinessObjectNodeType.REF_OPERATION);
                                respVO.setRefNode(true);
                                respVO.setIcon(null);
                                respVO.setId(t.getId());
                                respVO.setCode(t.getRef());
                                respVO.setName(operation.getName());
                                respVO.setSortNo(t.getSortNo());
                                respVO.setAppCode(operation.getAppCode());
                                respVO.setApiMethod(operation.getApiMethod());
                                respVO.setApiUrl(operation.getApiUrl());
                                respVO.setParentId(businessObject.getId());
                                respVO.setParentCode(businessObject.getCode());
                                respVO.setChildren(Collections.emptyList());

                                return respVO;
                            }).collect(Collectors.toList());
                    if (CollUtil.isEmpty(businessObject.getChildren())) {
                        businessObject.setChildren(refVoList);
                        return;
                    }
                    businessObject.getChildren().addAll(refVoList);
                }
            }
        }
    }

    private List<BusinessObjectTreeNodeRespVO> filteredHasChildren(List<BusinessObjectTreeNodeRespVO> respVoList) {
        if (respVoList.isEmpty()) {
            return respVoList;
        }

        List<BusinessObjectTreeNodeRespVO> dataList = new ArrayList<>();
        for (BusinessObjectTreeNodeRespVO respVO : respVoList) {
            if (respVO.getNodeType() == BusinessObjectNodeType.APP) {
                // 应用
                var businessObjectList = this.filteredHasChildren(respVO.getChildren());
                respVO.setChildren(businessObjectList);
                if (businessObjectList.isEmpty()) {
                    continue;
                }
            }
            if (respVO.getNodeType() == BusinessObjectNodeType.BUSINESS_OBJECT) {
                // 业务对象
                var businessOperationList = respVO.getChildren();
                if (CollUtil.isEmpty(businessOperationList)) {
                    if (Boolean.FALSE.equals(respVO.getHasChildren())) {
                        continue;
                    }
                } else {
                    businessOperationList = this.filteredHasChildren(businessOperationList);
                    respVO.setChildren(businessOperationList);
                    if (businessOperationList.isEmpty()) {
                        continue;
                    }
                }
            }

            dataList.add(respVO);
        }
        return dataList;
    }

    private List<BusinessObjectTreeNodeRespVO> convertApp2BusinessObjectTreeNode(Map<String, AppBO> tenantApps) {
        if (tenantApps.isEmpty()) {
            return Collections.emptyList();
        }
        List<BusinessObjectTreeNodeRespVO> businessObjectTreeNodeRespVOList = new ArrayList<>();

        int appOrder = 1;
        for (var entry : tenantApps.entrySet()) {
            BusinessObjectTreeNodeRespVO respVO = new BusinessObjectTreeNodeRespVO();
            respVO.setNodeType(BusinessObjectNodeType.APP);
            respVO.setRefNode(false);
            respVO.setIcon(entry.getValue().getIcon());
            respVO.setAppCode(entry.getValue().getAppCode());
            respVO.setId(entry.getValue().getId());
            respVO.setCode(entry.getValue().getAppCode());
            respVO.setName(entry.getValue().getAppName());
            respVO.setSortNo(appOrder++);
            respVO.setChildren(Collections.emptyList());

            businessObjectTreeNodeRespVOList.add(respVO);
        }

        return businessObjectTreeNodeRespVOList;
    }

    private List<BusinessObjectTreeNodeRespVO> queryBusinessObjectTreeNode(Map<String, AppBO> tenantApps, boolean withApp) {
        AtomicInteger businessObjectOrder = new AtomicInteger(1);
        return businessObjectRepoProc.all().stream()
                .filter(t -> {
                    if (!Boolean.TRUE.equals(t.getEnabled())) {
                        return false;
                    }
                    return tenantApps.containsKey(t.getAppCode());
                })
                .map(t -> {
                    BusinessObjectTreeNodeRespVO respVO = new BusinessObjectTreeNodeRespVO();
                    respVO.setNodeType(BusinessObjectNodeType.BUSINESS_OBJECT);
                    respVO.setRefNode(false);
                    respVO.setId(t.getId());
                    respVO.setCode(t.getCode());
                    respVO.setName(CharSequenceUtil.blankToDefault(t.getCustomName(), t.getName()));
                    respVO.setSortNo(businessObjectOrder.getAndAdd(1));
                    respVO.setAppCode(t.getAppCode());
                    if (withApp) {
                        respVO.setParentId(tenantApps.get(t.getAppCode()).getId());
                        respVO.setParentCode(t.getAppCode());
                    }
                    respVO.setChildren(Collections.emptyList());

                    return respVO;
                })
                .collect(Collectors.toList());
    }

    private List<BusinessObjectTreeNodeRespVO> queryBusinessOperationTreeNode(Map<String, AppBO> tenantApps, boolean permissionOnly) {
        return businessOperationRepoProc.listSimpleByType(permissionOnly).stream()
                .filter(t -> {
                    if (!Boolean.TRUE.equals(t.getEnabled())) {
                        return false;
                    }
                    return tenantApps.containsKey(t.getAppCode());
                }).map(t -> {
                    BusinessObjectTreeNodeRespVO respVO = new BusinessObjectTreeNodeRespVO();
                    respVO.setNodeType(BusinessObjectNodeType.BUSINESS_OPERATION);
                    respVO.setRefNode(false);
                    respVO.setId(t.getId());
                    respVO.setCode(t.getOperationCode());
                    respVO.setName(t.getOperationDescription());
                    respVO.setSortNo(t.getDisplayOrder());
                    respVO.setAppCode(t.getAppCode());
                    respVO.setApiMethod(t.getApiMethod());
                    respVO.setApiUrl(t.getApiUrl());
                    respVO.setParentId(t.getBusinessObjectId());
                    respVO.setParentCode(t.getBusinessObjectCode());
                    respVO.setChildren(Collections.emptyList());
                    if (StringUtils.hasText(t.getPermissionRef())) {
                        respVO.setRefPermission(t.getPermissionRef());
                    }

                    return respVO;
                }).collect(Collectors.toList());
    }

    private Map<String, AppBO> tenantApps() {
        var tenant = tenantClientProvider.getCurrentTenant();
        if (tenant != null && CollUtil.isEmpty(tenant.getAppCodes())) {
            return Collections.emptyMap();
        }

        var appList = appRepoProc.allBO(true);
        var tenantAppCodes = tenant == null ? null : tenant.getAppCodes();
        Map<String, AppBO> appsMap = new LinkedHashMap<>(appList.size());
        for (AppBO param : appList) {
            if (tenantAppCodes == null || tenantAppCodes.contains(param.getAppCode())) {
                appsMap.put(param.getAppCode(), param);
            }
        }
        return appsMap;
    }
}
