package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.DataSelectorConverter;
import com.elitescloud.cloudt.system.model.vo.query.sys.DataSelectorPagedQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.DataSelectorDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.DataSelectorGroupDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.DataSelectorPagedRespVO;
import com.elitescloud.cloudt.system.model.vo.save.sys.DataSelectorGroupSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.sys.DataSelectorSaveVO;
import com.elitescloud.cloudt.system.service.DataSelectorMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysDataSelectorDO;
import com.elitescloud.cloudt.system.service.model.entity.SysDataSelectorGroupDO;
import com.elitescloud.cloudt.system.service.repo.DataSelectorGroupRepoProc;
import com.elitescloud.cloudt.system.service.repo.DataSelectorRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/16
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class DataSelectorMngServiceImpl implements DataSelectorMngService {

    @Autowired
    private DataSelectorRepoProc repoProc;
    @Autowired
    private DataSelectorGroupRepoProc groupRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveGroup(DataSelectorGroupSaveVO saveVO) {
        SysDataSelectorGroupDO groupDO = null;
        try {
            groupDO = this.checkAndConvert(saveVO);
        } catch (Exception e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        groupRepoProc.save(groupDO);

        return ApiResult.ok(groupDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> deleteGroup(String groupCode) {
        Assert.notBlank(groupCode, "分组编码为空");

        var exists = repoProc.existsGroup(groupCode);
        if (exists) {
            return ApiResult.fail("请先删除分组下的数据");
        }

        groupRepoProc.deleteByGroupCode(groupCode);
        return ApiResult.ok(groupCode);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> updateGroupEnabled(String groupCode, Boolean enabled) {
        Assert.notBlank(groupCode, "分组编码为空");
        Assert.notNull(enabled, "启用状态为空");

        groupRepoProc.updateEnabled(groupCode, enabled);
        return ApiResult.ok(groupCode);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> updateGroupName(String groupCode, String name) {
        Assert.notBlank(groupCode, "分组编码为空");

        groupRepoProc.updateGroupName(groupCode, name);
        return ApiResult.ok(groupCode);
    }

    @Override
    public ApiResult<DataSelectorGroupDetailRespVO> getGroupDetail(String groupCode) {
        Assert.notNull(groupCode, "分组编码为空");

        var groupDO = groupRepoProc.getByGroupCode(groupCode);
        if (groupDO == null) {
            return ApiResult.fail("分组信息为空");
        }
        var respVO = DataSelectorConverter.INSTANCE.do2RespVO(groupDO);

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> listGroup() {
        var dataList = groupRepoProc.listGroup();
        return ApiResult.ok(dataList);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(DataSelectorSaveVO saveVO) {
        SysDataSelectorDO selectorDO = null;
        try {
            selectorDO = this.checkAndConvert(saveVO);
        } catch (Exception e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        repoProc.save(selectorDO);
        return ApiResult.ok(selectorDO.getId());
    }

    @Override
    public ApiResult<DataSelectorDetailRespVO> getDetail(Long id) {
        Assert.notNull(id, "ID为空");

        var selectorDO = repoProc.get(id);
        if (selectorDO == null) {
            return ApiResult.fail("数据不存在");
        }

        var respVO = DataSelectorConverter.INSTANCE.do2DetailRespVO(selectorDO);

        if (StringUtils.hasText(respVO.getGroupCode())) {
            respVO.setGroupName(groupRepoProc.getNameByCode(respVO.getGroupCode()));
        }

        respVO.setApiHeaders(JSONUtil.json2Obj(selectorDO.getApiHeadersJson(), new TypeReference<HashMap<String, String>>() {
        }));
        respVO.setApiReqBody(JSONUtil.json2Obj(selectorDO.getApiReqBodyJson(), new TypeReference<HashMap<String, Object>>() {
        }));
        respVO.setApiReqParams(JSONUtil.json2Obj(selectorDO.getApiReqParamsJson(), new TypeReference<HashMap<String, String>>() {
        }));
        respVO.setDataFields(StringUtils.hasText(selectorDO.getDataFieldsStr()) ?
                Arrays.stream(selectorDO.getDataFieldsStr().split(",")).collect(Collectors.toList()) : Collections.emptyList());
        respVO.setExtAttributes(JSONUtil.json2Obj(selectorDO.getExtAttributesJson(), new TypeReference<HashMap<String, Object>>() {
        }));

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<PagingVO<DataSelectorPagedRespVO>> pageMng(DataSelectorPagedQueryVO queryVO) {
        var pageData = repoProc.pageMng(queryVO);
        return ApiResult.ok(pageData);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id, Boolean enabled) {
        repoProc.updateEnabled(id, enabled == null || enabled);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> delete(List<Long> ids) {
        Assert.notEmpty(ids, "ID为空");

        repoProc.delete(ids);
        return ApiResult.ok(true);
    }

    private SysDataSelectorDO checkAndConvert(DataSelectorSaveVO saveVO) {
        SysDataSelectorDO selectorDO = null;
        if (StringUtils.hasText(saveVO.getGroupCode())) {
            Assert.isTrue(groupRepoProc.existsCode(saveVO.getGroupCode()), "分组不存在");
        } else {
            saveVO.setGroupCode("");
        }

        Assert.notBlank(saveVO.getSelectorCode(), "编码为空");
        if (saveVO.getId() == null) {
            selectorDO = new SysDataSelectorDO();
            Assert.isTrue(!repoProc.existsCode(saveVO.getSelectorCode()), "编码已存在");
        } else {
            selectorDO = repoProc.get(saveVO.getId());
            Assert.notNull(selectorDO, "数据不存在");
            Assert.isTrue(selectorDO.getSelectorCode().equals(saveVO.getSelectorCode()), "编码不可修改");
        }
        ObjUtil.ifNull(saveVO.getEnabled(), true, saveVO::setEnabled);

        Assert.notNull(saveVO.getComponent(), "未明确是否使用的是组件");
        if (saveVO.getComponent()) {
            Assert.notBlank(saveVO.getComponentCode(), "组件编码为空");
        } else {
            Assert.notBlank(saveVO.getDataApi(), "数据接口API为空");
            Assert.notBlank(saveVO.getDataApiMethod(), "数据接口API请求方式为空");
        }
        if (CollUtil.isNotEmpty(saveVO.getDataFields())) {
            saveVO.setDataFields(saveVO.getDataFields().stream().filter(StringUtils::hasText).distinct().collect(Collectors.toList()));
        }
        if (saveVO.getValueNumLimit() == null) {
            saveVO.setValueNumLimit(-1);
        }

        DataSelectorConverter.INSTANCE.copy(saveVO, selectorDO);
        selectorDO.setApiHeadersJson(JSONUtil.toJsonString(saveVO.getApiHeaders()));
        selectorDO.setApiReqBodyJson(JSONUtil.toJsonString(saveVO.getApiReqBody()));
        selectorDO.setApiReqParamsJson(JSONUtil.toJsonString(saveVO.getApiReqParams()));
        selectorDO.setExtAttributesJson(JSONUtil.toJsonString(saveVO.getExtAttributes()));
        selectorDO.setDataFieldsStr(CollUtil.isEmpty(saveVO.getDataFields()) ? null : String.join(",", saveVO.getDataFields()));
        return selectorDO;
    }

    private SysDataSelectorGroupDO checkAndConvert(DataSelectorGroupSaveVO saveVO) {
        SysDataSelectorGroupDO groupDO = null;
        Assert.notBlank(saveVO.getGroupCode(), "分组编码为空");
        if (saveVO.getId() == null) {
            groupDO = new SysDataSelectorGroupDO();
            groupDO.setGroupCode(saveVO.getGroupCode());
            Assert.isTrue(!groupRepoProc.existsCode(saveVO.getGroupCode()), "分组编码已存在");
        } else {
            groupDO = groupRepoProc.get(saveVO.getId());
            Assert.notNull(groupDO, "分组不存在");
            Assert.isTrue(saveVO.getGroupCode().equals(groupDO.getGroupCode()), "分组编码不可修改");
        }

        ObjUtil.ifNull(saveVO.getEnabled(), true, saveVO::setEnabled);
        ObjUtil.ifNull(saveVO.getSortNo(), 0, saveVO::setSortNo);

        DataSelectorConverter.INSTANCE.copyGroup(saveVO, groupDO);

        return groupDO;
    }
}
