package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.constant.Gender;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.SysConstant;
import com.elitescloud.cloudt.system.convert.AreaConvert;
import com.elitescloud.cloudt.system.convert.EmployeeConvert;
import com.elitescloud.cloudt.system.convert.UserConvert;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.req.UserQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeePageRespDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeeUnderlingDTO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeePagedRespVO;
import com.elitescloud.cloudt.system.service.UserQueryService;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/11/2
 */
abstract class EmployeeBaseServiceImpl extends BaseServiceImpl {
    protected static final EmployeeConvert CONVERT = EmployeeConvert.INSTANCE;
    protected static final UserConvert CONVERT_USER = UserConvert.INSTANCE;
    protected static final AreaConvert CONVERT_AREA = AreaConvert.INSTANCE;

    @Autowired
    protected EmployeeRepoProc employeeRepoProc;
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    protected UserQueryService userQueryService;

    protected void fillUserInfo(List<EmployeePagedRespVO> dataList, Map<Long, SysUserBasicDTO> userMap) {
        if (CollUtil.isEmpty(dataList)) {
            // 数据为空
            return;
        }

        // 查询用户信息
        if (CollectionUtils.isEmpty(userMap)) {
            var userIds = dataList.stream()
                    .filter(t -> t.getUserId() != null && t.getUserId() != SysConstant.NO_USER_ID)
                    .map(EmployeePagedRespVO::getUserId)
                    .collect(Collectors.toSet());
            UserQueryDTO queryDTO = new UserQueryDTO();
            queryDTO.setUserIds(userIds);
            var users = userRepoCall(repo -> repo.queryBasicDto(queryDTO));
            userMap = users
                    .stream()
                    .collect(Collectors.toMap(SysUserBasicDTO::getId, t -> t, (t1, t2) -> t1));
        }

        // 设置用户信息
        for (EmployeePagedRespVO record : dataList) {
            var user = userMap.get(record.getUserId());
            if (user == null) {
                record.setFullName(this.convertFullName(record.getLastName(), record.getFirstName()));
                record.setGenderName(new Gender(record.getGender()).getDescription());
                continue;
            }

            record.setUsername(user.getUsername());
            record.setFullName(user.getFullName());
            if (!StringUtils.hasText(record.getGender())) {
                record.setGender(user.getGender());
            }
            record.setGenderName(new Gender(record.getGender()).getDescription());
            record.setMobile(user.getMobile());
            if (!StringUtils.hasText(record.getEmail())) {
                record.setEmail(user.getEmail());
            }
            record.setEnabled(Boolean.TRUE.equals(record.getEnabled()) && Boolean.TRUE.equals(user.getEnabled()));
        }
    }

    /**
     * 获取全名
     *
     * @return 全名
     */
    protected String convertFullName(String lastName, String firstName) {
        if (!StringUtils.hasText(lastName)) {
            return firstName;
        }

        return lastName + CharSequenceUtil.blankToDefault(firstName, "");
    }

    protected void fillUserInfoDTO(PagingVO<EmployeePageRespDTO> pagingVO, Map<Long, SysUserBasicDTO> userMap) {
        if (pagingVO.isEmpty()) {
            // 数据为空
            return;
        }

        // 查询用户信息
        if (CollectionUtils.isEmpty(userMap)) {
            var userIds = pagingVO.getRecords().stream()
                    .map(EmployeePageRespDTO::getUserId)
                    .filter(t -> t != null && t != SysConstant.NO_USER_ID)
                    .collect(Collectors.toSet());
            UserQueryDTO queryDTO = new UserQueryDTO();
            queryDTO.setUserIds(userIds);
            var users = userRepoCall(repo -> repo.queryBasicDto(queryDTO));
            userMap = users
                    .stream()
                    .collect(Collectors.toMap(SysUserBasicDTO::getId, t -> t, (t1, t2) -> t1));
        }

        // 设置用户信息
        for (EmployeePageRespDTO record : pagingVO.getRecords()) {
            var user = userMap.get(record.getUserId());
            if (user == null) {
                continue;
            }

            record.setUsername(user.getUsername());
            record.setFullName(user.getFullName());
            if (!StringUtils.hasText(record.getGender())) {
                record.setGender(user.getGender());
            }
            record.setGenderName(new Gender(record.getGender()).getDescription());
            if (!StringUtils.hasText(record.getPhone())) {
                record.setPhone(user.getMobile());
            }
            if (!StringUtils.hasText(record.getEmail())) {
                record.setEmail(user.getEmail());
            }
            record.setEnabled(Boolean.TRUE.equals(record.getEnabled()) && Boolean.TRUE.equals(user.getEnabled()));
        }
    }

    protected SysEmployeeBasicDTO fillTenantUserInfo(SysEmployeeBasicDTO employee) {
        if (employee == null) {
            return null;
        }
        if (employee.getUserId() == null || employee.getUserId() == SysConstant.NO_USER_ID) {
            return null;
        }

        // 状态
        if (Boolean.TRUE.equals(employee.getEnabled())) {
            var enabled = this.userRepoCall(repo -> repo.getEnabled(employee.getUserId()));
            employee.setEnabled(Boolean.TRUE.equals(enabled));
        }
        return employee;
    }

    protected List<EmployeeUnderlingDTO> fillTenantUserInfo(List<EmployeeUnderlingDTO> underlingList) {
        if (CollUtil.isEmpty(underlingList)) {
            return underlingList;
        }

        var userIds = underlingList.stream().map(EmployeeUnderlingDTO::getUserId).collect(Collectors.toSet());
        var tenantId = super.currentTenantId();
        var tenantUserMap = this.userRepoCall(repo -> repo.getEnabled(tenantId, userIds));
        for (EmployeeUnderlingDTO dto : underlingList) {
            dto.setEnabled(Boolean.TRUE.equals(dto.getEnabled()) && Boolean.TRUE.equals(tenantUserMap.get(dto.getUserId())));
        }
        return underlingList;
    }

    /**
     * 用户信息在默认schema下
     *
     * @param function
     * @param <T>
     * @return
     */
    protected <T> T userRepoCall(Function<UserRepoProc, T> function) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> function.apply(userRepoProc));
    }
}
