package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.AreaVO;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.constant.EmployeeType;
import com.elitescloud.cloudt.system.constant.SysConstant;
import com.elitescloud.cloudt.system.constant.UserType;
import com.elitescloud.cloudt.system.convert.EmployeeConvert;
import com.elitescloud.cloudt.system.dto.*;
import com.elitescloud.cloudt.system.dto.req.EmployeePageQueryDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeQueryDTO;
import com.elitescloud.cloudt.system.dto.req.UserQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeeOrgRespDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeePageRespDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeeUnderlingDTO;
import com.elitescloud.cloudt.system.model.bo.EmployeeOrgBO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonEmployeePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmployeeListQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeeListRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeePagedRespVO;
import com.elitescloud.cloudt.system.service.AreaQueryService;
import com.elitescloud.cloudt.system.service.EmployeeQueryService;
import com.elitescloud.cloudt.system.service.manager.PermissionQueryManager;
import com.elitescloud.cloudt.system.service.model.entity.SysEmployeeDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.service.repo.*;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/11/2
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class EmployeeQueryServiceImpl extends EmployeeBaseServiceImpl implements EmployeeQueryService {
    private static final EmployeeConvert CONVERT = EmployeeConvert.INSTANCE;

    @Autowired
    private EmployeeRepoProc repoProc;
    @Autowired
    private EmployeeOrgRepoProc employeeOrgRepoProc;
    @Autowired
    private OrgRepoProc orgRepoProc;
    @Autowired
    private UserRoleRepoProc userRoleRepoProc;
    @Autowired
    private AreaQueryService areaQueryService;
    @Autowired
    private PermissionQueryManager permissionQueryManager;

    @Override
    public ApiResult<PagingVO<EmployeePagedRespVO>> pageQuery(CommonEmployeePageQueryVO queryVO) {
        var userFilterMap = queryUser(queryVO);
        if (userFilterMap != null && userFilterMap.isEmpty()) {
            // 根据账号未查询到
            return ApiResult.ok(PagingVO.empty());
        }
        // 查询员工
        Set<Long> userIds = userFilterMap == null ? null : userFilterMap.keySet();
        Set<Long> userIdsOfDisabled = userIds == null && queryVO.getEnabled() != null ? this.queryDisabledUserIds() : null;
        var pageData = repoProc.pageQuery(queryVO, userIds, userIdsOfDisabled);
        if (pageData.isEmpty()) {
            return ApiResult.ok(PagingVO.<EmployeePagedRespVO>builder().total(pageData.getTotal()).build());
        }
        // 员工类型
        var employeeTypeMap = udcMap(new EmployeeType());
        var ids = pageData.stream().map(EmployeePagedRespVO::getId).collect(Collectors.toList());
        userIds = pageData.stream().map(EmployeePagedRespVO::getUserId).filter(t -> t != null && t != SysConstant.NO_USER_ID).collect(Collectors.toSet());
        // 员工所属组织
        Map<Long, List<IdCodeNameParam>> orgMap = Boolean.TRUE.equals(queryVO.getWithOrg()) ? employeeOrgRepoProc.getEmployeeOrgs(ids, true) : Collections.emptyMap();
        // 角色
        Map<Long, List<IdCodeNameParam>> roleMap = Boolean.TRUE.equals(queryVO.getWithRole()) ? permissionQueryManager.queryUserRoles(userIds) : Collections.emptyMap();
        pageData.each(t -> {
            // 员工类型
            if (StringUtils.hasText(t.getType())) {
                t.setTypeName(employeeTypeMap.get(t.getType()));
            }
            t.setOrgNameList(orgMap.getOrDefault(t.getId(), Collections.emptyList()).stream().map(IdCodeNameParam::getName).collect(Collectors.toList()));
            t.setRoleNameList(roleMap.getOrDefault(t.getUserId(), Collections.emptyList()).stream().map(IdCodeNameParam::getName).collect(Collectors.toList()));
            t.setWithoutLoginAccount(t.getUserId() == null || t.getUserId() == SysConstant.NO_USER_ID);
        });

        // 填充用户信息
        fillUserInfo(pageData.getRecords(), userFilterMap);
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<PagingVO<EmployeePageRespDTO>> queryByPage(EmployeePageQueryDTO queryDTO) {
//        var userFilterMap = queryUser(queryDTO);
//        if (userFilterMap != null && userFilterMap.isEmpty()) {
//            // 根据账号未查询到
//            return ApiResult.ok(PagingVO.empty());
//        }
        // 查询员工
//        Set<Long> userIds = userFilterMap == null ? null : userFilterMap.keySet();
        Set<Long> userIds = null;
        Map<Long, SysUserBasicDTO> userFilterMap = null;
        Set<Long> userIdsOfDisabled = userIds == null && queryDTO.getEnabled() != null ? this.queryDisabledUserIds() : null;
        var pageData = repoProc.pageQuery(queryDTO, userIds, userIdsOfDisabled);
        if (pageData.isEmpty()) {
            return ApiResult.ok(PagingVO.<EmployeePageRespDTO>builder().total(pageData.getTotal()).build());
        }

        // 查询员工的组织
        Map<Long, List<IdCodeNameParam>> employeeOrgMap;
        Map<Long, IdCodeNameParam> parentOrgMap;
        Map<Long, List<EmployeeOrgRespDTO>> groupedByEmployeeId;
        boolean withOrg = queryDTO.getWithoutOrg() == null || !queryDTO.getWithoutOrg() || StringUtils.hasText(queryDTO.getWithParentOrgType());
        if (withOrg) {
            var ids = pageData.stream().map(EmployeePageRespDTO::getId).collect(Collectors.toList());
            employeeOrgMap = employeeOrgRepoProc.getEmployeeOrgs(ids, true);
            // 查询组织的上级组织
            Set<Long> orgIds = CharSequenceUtil.isBlank(queryDTO.getWithParentOrgType()) ? Collections.emptySet() :
                    employeeOrgMap.values().stream().flatMap(Collection::stream).map(IdCodeNameParam::getId).collect(Collectors.toSet());
            parentOrgMap = orgIds.isEmpty() ? Collections.emptyMap() : orgRepoProc.queryParentNameForType(orgIds, queryDTO.getWithParentOrgType(), true);

            // 查询上级
            if (Boolean.TRUE.equals(queryDTO.getWithLeader())) {
                List<EmployeeOrgBO> empLeadersMap = employeeOrgRepoProc.getLeaders(ids);
                List<EmployeeOrgRespDTO> employeeOrgRespDTOList = empLeadersMap.stream().map(employeeOrgBO -> {
                    EmployeeOrgRespDTO employeeOrgRespDTO = new EmployeeOrgRespDTO();
                    employeeOrgRespDTO.setOwnerId(employeeOrgBO.getOwnerId());
                    employeeOrgRespDTO.setOrgId(employeeOrgBO.getOrgId());
                    employeeOrgRespDTO.setOrgName(employeeOrgBO.getOrgName());
                    employeeOrgRespDTO.setEmployeeId(employeeOrgBO.getEmployeeId());
                    employeeOrgRespDTO.setEmployeeCode(employeeOrgBO.getEmployeeCode());
                    employeeOrgRespDTO.setEmployeeName(employeeOrgBO.getLastName());
                    employeeOrgRespDTO.setEmployeeType(employeeOrgBO.getEmployeeType());
                    return employeeOrgRespDTO;
                }).collect(Collectors.toList());
                groupedByEmployeeId = employeeOrgRespDTOList.stream()
                        .collect(Collectors.groupingBy(EmployeeOrgRespDTO::getOwnerId));
            } else {
                groupedByEmployeeId = Collections.emptyMap();
            }
        } else {
            employeeOrgMap = Collections.emptyMap();
            parentOrgMap = Collections.emptyMap();
            groupedByEmployeeId = Collections.emptyMap();
        }
        var employeeTypeMap = super.udcMap(new EmployeeType());
        pageData.each(t -> {
            // 员工类型
            if (StringUtils.hasText(t.getType())) {
                t.setTypeName(employeeTypeMap.get(t.getType()));
            }
            // 组织
            if (withOrg) {
                t.setOrgList(employeeOrgMap.getOrDefault(t.getId(), Collections.emptyList()));
                t.setOrgNames(employeeOrgMap.getOrDefault(t.getId(), Collections.emptyList()).stream().map(IdCodeNameParam::getName).collect(Collectors.toList()));
                List<IdCodeNameParam> parentOrgList = new ArrayList<>();
                for (IdCodeNameParam idCodeNameParam : t.getOrgList()) {
                    parentOrgList.add(parentOrgMap.get(idCodeNameParam.getId()));
                }
                t.setParentOrgList(parentOrgList);
                t.setLeadEmployeeOrg(groupedByEmployeeId.get(t.getId()));
            }
            t.setWithoutLoginAccount(t.getUserId() == null || t.getUserId() == SysConstant.NO_USER_ID);
        });

        // 填充用户信息
        fillUserInfoDTO(pageData, userFilterMap);
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<List<String>> getSalesmanId(Long id) {
        if (id == null) {
            return ApiResult.fail("员工ID为空");
        }

        var userId = repoProc.getUserId(id);
        if (userId == null) {
            return ApiResult.ok(Collections.emptyList());
        }
        return this.getSalesmanIdByUserId(userId);
    }

    @Override
    public ApiResult<List<String>> getSalesmanIdByUserId(Long userId) {
        if (userId == null) {
            return ApiResult.fail("账号ID为空");
        }

        var userTypes = userQueryService.getUserTypes(userId).computeData();
        if (userTypes.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        var salesmanIds = userTypes.stream().filter(t -> UserType.SALESMAN.getValue().equals(t.getUserType()))
                .map(SysUserTypeDTO::getIdentityId)
                .collect(Collectors.toSet());
        return ApiResult.ok(new ArrayList<>(salesmanIds));
    }

    @Override
    public ApiResult<List<EmployeeListRespVO>> listQuery(EmployeeListQueryVO queryVO) {
        var employeeList = repoProc.listQuery(queryVO);
        for (EmployeeListRespVO employee : employeeList) {
            employee.setWithoutLoginAccount(employee.getUserId() == null || employee.getUserId() == SysConstant.NO_USER_ID);
        }
        return ApiResult.ok(employeeList);
    }

    @Override
    public ApiResult<SysEmployeeBasicDTO> getById(Long id) {
        Assert.notNull(id, "用户ID为空");

        return repoProc.getOptional(id)
                .map(t -> {
                    var dto = CONVERT.do2BasicDto(t);
                    this.fillTenantUserInfo(dto);
                    return dto;
                })
                .map(ApiResult::ok)
                .orElse(ApiResult.ok(null));
    }

    @Override
    public ApiResult<SysEmployeeBasicDTO> getByCode(String code) {
        Assert.hasText(code, "员工编号为空");

        return repoProc.getByCode(code)
                .map(t -> {
                    var dto = CONVERT.do2BasicDto(t);
                    this.fillTenantUserInfo(dto);
                    return dto;
                })
                .map(ApiResult::ok)
                .orElse(ApiResult.ok(null));
    }

    @Override
    public ApiResult<SysEmployeeBasicDTO> getByUsername(String username) {
        Assert.hasText(username, "账号为空");

        var userId = super.userRepoCall(repo -> repo.getIdByUsername(username));
        if (userId == null) {
            return ApiResult.ok();
        }
        return this.getByUserId(userId);
    }

    @Override
    public ApiResult<SysEmployeeBasicDTO> getByUserId(Long userId) {
        Assert.notNull(userId, "用户ID为空");

        return repoProc.getByUserId(userId)
                .map(t -> {
                    var dto = CONVERT.do2BasicDto(t);
                    this.fillTenantUserInfo(dto);
                    return dto;
                })
                .map(ApiResult::ok)
                .orElse(ApiResult.ok(null));
    }

    @Override
    public ApiResult<SysEmployeeDetailDTO> getDetailById(Long id) {
        Assert.notNull(id, "员工ID为空");
        var employee = repoProc.get(id);
        if (employee == null) {
            return ApiResult.ok();
        }

        // 转为dto
        var employeeList = this.convertToEmployeeDetail(List.of(employee));
        return ApiResult.ok(employeeList.get(0));
    }

    @Override
    public ApiResult<SysEmployeeDetailDTO> getDetailByCode(String code) {
        Assert.hasText(code, "员工编号为空");
        var employee = repoProc.getByCode(code).orElse(null);
        if (employee == null) {
            return ApiResult.ok();
        }

        // 转为dto
        var employeeList = this.convertToEmployeeDetail(List.of(employee));
        return ApiResult.ok(employeeList.get(0));
    }

    @Override
    public ApiResult<SysEmployeeDetailDTO> getDetailByUserId(Long userId) {
        Assert.notNull(userId, "用户ID为空");
        var employee = repoProc.getByUserId(userId).orElse(null);
        if (employee == null) {
            return ApiResult.ok();
        }

        // 转为dto
        var employeeList = this.convertToEmployeeDetail(List.of(employee));
        return ApiResult.ok(employeeList.get(0));
    }

    @Override
    public ApiResult<SysEmployeeDetailDTO> getDetailByUsername(String username) {
        Assert.hasText(username, "账号为空");
        var userId = super.userRepoCall(repo -> repo.getIdByUsername(username));
        if (userId == null) {
            return ApiResult.ok(null);
        }

        return this.getDetailByUserId(userId);
    }

    @Override
    public ApiResult<List<SysEmployeeDetailDTO>> getDetailListByCode(Set<String> codes) {
        Assert.notEmpty(codes, "员工编号为空");
        var employeeList = repoProc.getByCode(codes);

        // 转为dto
        var employeeDtoList = this.convertToEmployeeDetail(employeeList);
        return ApiResult.ok(employeeDtoList);
    }

    @Override
    public ApiResult<List<EmployeeUnderlingDTO>> getUnderlingByCode(String code, Boolean cascade, Boolean enabled) {
        Assert.hasText(code, "员工编号为空");
        cascade = cascade != null && cascade;

        // 查询员工ID
        var id = repoProc.getIdByCode(code);
        if (id == null) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 查询下属
        var underlingList = employeeOrgRepoProc.getUnderlings(Set.of(id), enabled);
        if (underlingList.isEmpty()) {
            // 没有下属
            return ApiResult.ok(Collections.emptyList());
        }
        var typeMap = super.udcMap(new EmployeeType());
        var dtoList = this.queryUnderling(underlingList, cascade, enabled, typeMap);
        return ApiResult.ok(dtoList);
    }

    @Override
    public ApiResult<List<SysUnderlingDTO>> getUnderlingId(Long id) {
        if (id == null) {
            return ApiResult.fail("员工ID为空");
        }
        var underlingList = employeeOrgRepoProc.getUnderlings(Set.of(id), true);
        if (underlingList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        var underlingDtoList = underlingList.stream().map(t -> {
            SysUnderlingDTO underlingDTO = new SysUnderlingDTO();
            underlingDTO.setUserId(t.getUserId());
            underlingDTO.setEmployeeId(t.getEmployeeId());
            underlingDTO.setDirectly(true);

            return underlingDTO;
        }).collect(Collectors.toList());
        return ApiResult.ok(underlingDtoList);
    }

    @Override
    public ApiResult<List<SysEmployeeBasicDTO>> queryList(EmployeeQueryDTO queryDTO) {
        // 先查询用户ID
        Set<Long> userIds = queryDTO.getUserIds();
        if (!CollectionUtils.isEmpty(queryDTO.getUsernames())) {
            var tempUserIds = super.userRepoCall(repo -> repo.getIdByUsername(queryDTO.getUsernames()));
            if (tempUserIds.isEmpty()) {
                return ApiResult.ok(Collections.emptyList());
            }
            // 合并用户ID
            if (CollectionUtils.isEmpty(userIds)) {
                userIds = new HashSet<>(tempUserIds);
            } else {
                userIds = userIds.stream().filter(tempUserIds::contains).collect(Collectors.toSet());
                if (userIds.isEmpty()) {
                    return ApiResult.ok(Collections.emptyList());
                }
            }
        }
        queryDTO.setUserIds(userIds);

        // 查询员工
        var recordList = repoProc.queryList(queryDTO).stream()
                .map(CONVERT::do2BasicDto)
                .collect(Collectors.toList());
        return ApiResult.ok(recordList);
    }

    @Override
    public ApiResult<List<Long>> getUserIdByLeaderUserId(Long leaderUserId) {
        var userIds = employeeOrgRepoProc.getUserIdByLeaderUserId(leaderUserId);
        return ApiResult.ok(userIds);
    }

    @Override
    public ApiResult<Map<Long, List<IdCodeNameParam>>> getEmployeeOrgs(List<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.ok(Collections.emptyMap());
        }

        var data = employeeOrgRepoProc.getEmployeeOrgs(ids, true);
        return ApiResult.ok(data);
    }

    @Override
    public ApiResult<Boolean> existsCode(String code) {
        Assert.hasText(code, "员工编号为空");

        var result = repoProc.existsCode(code);
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> queryCodeName(Set<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.fail("ID为空");
        }

        var employeeNames = repoProc.queryEmployeeName(ids);
        return ApiResult.ok(employeeNames);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> queryCodeNameByCode(Set<String> codes) {
        if (CollUtil.isEmpty(codes)) {
            return ApiResult.fail("Code为空");
        }

        var employeeNames = repoProc.queryEmployeeNameByCodes(codes);
        return ApiResult.ok(employeeNames);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> queryCodeNameByUserId(Set<Long> userIds) {
        if (CollUtil.isEmpty(userIds)) {
            return ApiResult.fail("userId为空");
        }

        var employeeNames = repoProc.queryEmployeeNameByUserIds(userIds);
        return ApiResult.ok(employeeNames);
    }

    private Map<Long, SysUserBasicDTO> queryUser(CommonEmployeePageQueryVO queryVO) {
        UserQueryDTO queryDTO = new UserQueryDTO();
        boolean flag = false;

        if (StringUtils.hasText(queryVO.getUsername())) {
            queryDTO.setUsernameLike(Set.of(queryVO.getUsername()));
            flag = true;
        }
        if (queryVO.getEnabled() != null) {
            queryDTO.setEnabled(queryVO.getEnabled());
        }

        if (!flag) {
            // 不根据用户查询
            return null;
        }

        var users = userRepoCall(repo -> repo.queryBasicDto(queryDTO));
        if (users.isEmpty()) {
            return Collections.emptyMap();
        }
        return users
                .stream()
                .collect(Collectors.toMap(SysUserBasicDTO::getId, t -> t, (t1, t2) -> t1));
    }

    private Set<Long> queryDisabledUserIds() {
        return new HashSet<>(this.userRepoCall(UserRepoProc::queryIdsOfDisabled));
    }

    private Map<Long, SysUserBasicDTO> queryUser(EmployeePageQueryDTO queryDTO) {
        UserQueryDTO param = new UserQueryDTO();
        boolean flag = false;

        if (StringUtils.hasText(queryDTO.getUsername())) {
            param.setUsernameLike(Set.of(queryDTO.getUsername()));
            flag = true;
        }
        if (CollUtil.isNotEmpty(queryDTO.getUserIds())) {
            param.setUserIds(queryDTO.getUserIds());
            flag = true;
        }
        if (CollUtil.isNotEmpty(queryDTO.getUsernames())) {
            param.setUsernames(queryDTO.getUsernames());
            flag = true;
        }
        if (queryDTO.getEnabled() != null) {
            param.setEnabled(queryDTO.getEnabled());
        }

        if (!flag) {
            // 不根据用户查询
            return null;
        }

        var users = userRepoCall(repo -> repo.queryBasicDto(param));
        if (users.isEmpty()) {
            return Collections.emptyMap();
        }
        return users
                .stream()
                .collect(Collectors.toMap(SysUserBasicDTO::getId, t -> t, (t1, t2) -> t1));
    }

    private List<EmployeeUnderlingDTO> queryUnderling(List<EmployeeOrgBO> underlingEmpOrgList, boolean cascade, Boolean enabled, Map<String, String> typeNameMap) {
        if (underlingEmpOrgList.isEmpty()) {
            // 没有下级
            return Collections.emptyList();
        }
        // 下级员工ID
        Set<Long> underlingEmpIds = underlingEmpOrgList.stream().map(EmployeeOrgBO::getEmployeeId).collect(Collectors.toSet());

        // 查询下级的下级
        Map<Long, List<EmployeeUnderlingDTO>> underlingUnderlingMap = null;
        if (cascade) {
            var underlingUnderlingList = employeeOrgRepoProc.getUnderlings(underlingEmpIds, enabled);
            underlingUnderlingMap = this.queryUnderling(underlingUnderlingList, cascade, enabled, typeNameMap).stream()
                    .collect(Collectors.groupingBy(EmployeeUnderlingDTO::getLeaderEmployeeId));
        } else {
            underlingUnderlingMap = Collections.emptyMap();
        }

        // 查询下级的组织名称
        var underlingEmpOrgNameMap = employeeOrgRepoProc.getEmployeeOrgs(underlingEmpIds, enabled);

        // 下级详细信息
        var employeeMap = repoProc.get(underlingEmpIds).stream().collect(Collectors.toMap(SysEmployeeDO::getId, t -> t, (t1, t2) -> t1));
        Map<Long, List<EmployeeUnderlingDTO>> finalUnderlingUnderlingMap = underlingUnderlingMap;

        var underlingList = underlingEmpOrgList.stream()
                .filter(t -> employeeMap.containsKey(t.getEmployeeId()))
                .map(t -> {
                    var dto = CONVERT.do2UnderlingDto(employeeMap.get(t.getEmployeeId()));
                    dto.setTypeName(typeNameMap.get(dto.getType()));

                    dto.setOrgNames(underlingEmpOrgNameMap.getOrDefault(dto.getId(), Collections.emptyList()).stream().map(IdCodeNameParam::getName).collect(Collectors.toList()));
                    dto.setLeaderEmployeeId(t.getOwnerId());
                    dto.setUnderlingList(finalUnderlingUnderlingMap.getOrDefault(dto.getId(), Collections.emptyList()));
                    return dto;
                })
                .collect(Collectors.toList());
        return super.fillTenantUserInfo(underlingList);
    }

    private List<SysEmployeeDetailDTO> convertToEmployeeDetail(List<SysEmployeeDO> doList) {
        if (doList.isEmpty()) {
            return Collections.emptyList();
        }
        // 员工类型
        var empTypeNameMap = super.udcMap(new EmployeeType());
        // 用户信息
        var userIds = doList.stream().map(SysEmployeeDO::getUserId).filter(t -> t != null && t != SysConstant.NO_USER_ID).collect(Collectors.toList());
        var ids = doList.stream().map(SysEmployeeDO::getId).collect(Collectors.toList());
        var userInfos = userRepoCall(re -> re.get(userIds));
        var userInfoMap = userInfos.stream().collect(Collectors.toMap(SysUserDO::getId, t -> t, (t1, t2) -> t1));
        // 地区信息
        var areaCodes = userInfos.stream().flatMap(t -> Stream.of(t.getCountryCode(), t.getProvinceCode(), t.getCityCode(), t.getCountyCode()).filter(StringUtils::hasText)).collect(Collectors.toSet());
        Map<String, String> areaNameMap = null;
        if (areaCodes.isEmpty()) {
            areaNameMap = Collections.emptyMap();
        } else {
            areaNameMap = ObjectUtil.defaultIfNull(areaQueryService.queryNamesByAreaCode(areaCodes).getData(), Collections.emptyMap());
        }
        Map<String, String> finalAreaNameMap = areaNameMap;

        // 员工组织
        var orgMap = employeeOrgRepoProc.getLeaders(ids).stream()
                .map(t -> {
                    SysEmployeeDetailDTO.EmployeeOrg temp = new SysEmployeeDetailDTO.EmployeeOrg();
                    temp.setEmployeeId(t.getOwnerId());
                    temp.setOrgId(t.getOrgId());
                    temp.setOrgCode(t.getOrgCode());
                    temp.setOrgName(t.getOrgName());
                    temp.setOrgType(t.getOrgType());
                    temp.setLeaderEmployeeCode(t.getEmployeeCode());
                    temp.setLeaderEmployeeId(t.getEmployeeId());
                    temp.setLeaderUserId(t.getUserId());
                    temp.setLeaderUserFullName(t.getFullName());
                    temp.setOrgList(orgRepoProc.getBasicDtoOfParents(t.getOrgId()));
                    if (t.getPositionId() != null) {
                        temp.setPosition(new IdCodeNameParam(t.getPositionId(), t.getPositionCode(), t.getPositionName()));
                    }

                    return temp;
                }).collect(Collectors.groupingBy(SysEmployeeDetailDTO.EmployeeOrg::getEmployeeId));
        // 角色信息
        var userRoleMap = userRoleRepoProc.getRoleOfUser(userIds, null, null);

        return doList.stream()
//                .filter(t -> userInfoMap.containsKey(t.getUserId()))
                .map(t -> {
                    var dto = CONVERT.do2DetailDto(t);
                    dto.setWithoutLoginAccount(t.getUserId() == null || t.getUserId() == SysConstant.NO_USER_ID);

                    // 员工类型
                    dto.setTypeName(empTypeNameMap.get(dto.getType()));

                    // 账号信息
                    if (!dto.getWithoutLoginAccount() && t.getUserId() != null && userInfoMap.containsKey(t.getUserId())) {
                        var userInfo = userInfoMap.get(t.getUserId());
                        dto.setFullName(userInfo.getFullName());
                        dto.setGender(userInfo.getGender());
                        dto.setGenderName(userInfo.getGenderName());
                        dto.setEmail(CharSequenceUtil.blankToDefault(dto.getEmail(), userInfo.getEmail()));
                        dto.setPhone(CharSequenceUtil.blankToDefault(dto.getPhone(), userInfo.getMobile()));
                        dto.setBirthDate(userInfo.getBirthDate());
                        dto.setIdCard(userInfo.getIdCard());
                        dto.setEnabled(Boolean.TRUE.equals(dto.getEnabled()) && Boolean.TRUE.equals(userInfo.getEnabled()));
                        // 所在地
                        dto.setAreaVO(this.convertAreaVO(userInfo, finalAreaNameMap));
                        dto.setAddress(userInfo.getAddress());
                    }

                    // 组织
                    dto.setOrgList(orgMap.getOrDefault(t.getId(), Collections.emptyList()));
                    // 角色
                    dto.setRoleList(userRoleMap.getOrDefault(dto.getUserId(), Collections.emptyList()));

                    return dto;
                }).collect(Collectors.toList());
    }

    private AreaVO convertAreaVO(SysUserDO userDo, Map<String, String> nameMap) {
        AreaVO areaVO = new AreaVO();
        areaVO.setCountryCode(userDo.getCountryCode());
        areaVO.setCountryName(nameMap.get(userDo.getCountryCode()));
        areaVO.setProvinceCode(userDo.getProvinceCode());
        areaVO.setProvinceName(nameMap.get(userDo.getProvinceCode()));
        areaVO.setCityCode(userDo.getCityCode());
        areaVO.setCityName(nameMap.get(userDo.getCityCode()));
        areaVO.setCountyCode(userDo.getCountyCode());
        areaVO.setCountyName(nameMap.get(userDo.getCountyCode()));

        return areaVO;
    }
}
