package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCache;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.common.param.SysSendVerifyCodeVO;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.core.support.verifycode.common.VerifyCodeManager;
import com.elitescloud.boot.datasecurity.common.DataSecurityUtil;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleConditionEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleRelationEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.constant.DataPermissionType;
import com.elitescloud.cloudt.system.constant.EmployeeType;
import com.elitescloud.cloudt.system.constant.OrgType;
import com.elitescloud.cloudt.system.convert.EmployeeConvert;
import com.elitescloud.cloudt.system.convert.PermissionConverter;
import com.elitescloud.cloudt.system.dto.*;
import com.elitescloud.cloudt.system.model.bo.BusinessObjectBO;
import com.elitescloud.cloudt.system.model.bo.BusinessOperationBO;
import com.elitescloud.cloudt.system.model.bo.EmployeeOrgBO;
import com.elitescloud.cloudt.system.model.bo.MenuSimpleBO;
import com.elitescloud.cloudt.system.model.vo.resp.index.*;
import com.elitescloud.cloudt.system.model.vo.save.index.ModifyPasswordSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.index.PasswordUpdateSaveVO;
import com.elitescloud.cloudt.system.service.FrontTableCfgService;
import com.elitescloud.cloudt.system.service.IndexUserService;
import com.elitescloud.cloudt.system.service.manager.EmployeeOrgManager;
import com.elitescloud.cloudt.system.service.manager.OuQueryManager;
import com.elitescloud.cloudt.system.service.manager.PermissionQueryManager;
import com.elitescloud.cloudt.system.service.manager.UserMngManager;
import com.elitescloud.cloudt.system.service.repo.*;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/17
 */
@Slf4j
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class IndexUserServiceImpl extends BaseServiceImpl implements IndexUserService {

    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private EmployeeRepoProc employeeRepoProc;
    @Autowired
    private EmployeeOrgManager employeeOrgManager;
    @Autowired
    private EmployeeOrgRepoProc employeeOrgRepoProc;
    @Autowired
    private UserMngManager userMngManager;
    @Autowired
    private OuQueryManager ouQueryManager;
    @Autowired
    private OrgRepoProc orgRepoProc;
    @Autowired
    private PermissionQueryManager permissionQueryManager;
    @Autowired
    private AuthorizationProperties authorizationProperties;
    @Autowired
    private AuthenticationCache authenticationCache;

    @Autowired
    private VerifyCodeManager verifyCodeManager;

    @Autowired
    private FrontTableCfgService frontTableCfgService;
    @Autowired
    private AppRepoProc appRepoProc;
    @Autowired
    private MenuRepoProc menuRepoProc;
    @Autowired
    private BusinessObjectRepoProc businessObjectRepoProc;
    @Autowired
    private BusinessOperationRepoProc businessOperationRepoProc;

    @Override
    public ApiResult<CurrentEmployeeRespVO> getEmployeeInfo() {
        GeneralUserDetails currentUser = super.currentUser(true);
        return employeeRepoProc.getByUserId(currentUser.getUserId())
                .map(t -> {
                    var respVO = EmployeeConvert.INSTANCE.do2CurrentRespVO(t);
                    if (CharSequenceUtil.isNotBlank(respVO.getType())) {
                        respVO.setTypeName(super.udcValue(new EmployeeType(respVO.getType())));
                    }
                    respVO.setUsername(currentUser.getUsername());
                    respVO.setFullName(currentUser.getUser().getPrettyName());

                    // 员工的公司信息
                    if (currentUser.getOrgId() != null) {
                        CurrentEmployeeRespVO.EmployeeOrg org = new CurrentEmployeeRespVO.EmployeeOrg();
                        respVO.setOrgInfo(org);

                        org.setOrgId(currentUser.getOrgId());
                        org.setOrgCode(currentUser.getUser().getOrg().getCode());
                        org.setOrgName(currentUser.getUser().getOrg().getName());
                        var company = orgRepoProc.queryParentNameForType(List.of(currentUser.getOrgId()), OrgType.COMPANY.getValue(), true).get(currentUser.getOrgId());
                        if (company != null) {
                            org.setCompanyOrgId(company.getId());
                            org.setCompanyOrgCode(company.getCode());
                            org.setCompanyOrgName(company.getName());
                        }

                        // 领导
                        var leaders = employeeOrgRepoProc.getLeaders(List.of(t.getId()));
                        if (!leaders.isEmpty()) {
                            for (EmployeeOrgBO leader : leaders) {
                                if (leader.getOrgId().equals(currentUser.getOrgId())) {
                                    org.setLeaderId(leader.getEmployeeId());
                                    org.setLeaderCode(leader.getEmployeeCode());
                                    org.setLeaderUserId(leader.getUserId());
                                    org.setLeaderUsername(leader.getUsername());
                                    org.setLeaderFullName(leader.getFullName());
                                }
                            }
                        }
                    }

                    return respVO;
                }).map(ApiResult::ok)
                .orElse(ApiResult.ok());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updatePassword(ModifyPasswordSaveVO saveVO) {
        if (!CharSequenceUtil.equals(saveVO.getPdRepeat(), saveVO.getPdNew())) {
            return ApiResult.fail("两次输入密码不一致");
        }

        GeneralUserDetails currentUser = super.currentUser(true);

        // 修改密码
        userMngManager.updatePassword(currentUser.getUserId(), saveVO.getPdNew(), saveVO.getPd(), false);

        // 注销
        String token = SecurityContextUtil.currentToken();
        if (StringUtils.hasText(token)) {
            authenticationCache.removeUserDetail(token);
        }

        return ApiResult.ok(true);
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @TenantOrgTransaction(useTenantOrg = false)
    @Override
    public ApiResult<String> sendVerifyCodeForUpdatePwd(boolean retrieve, SysSendVerifyCodeVO verifyCodeParam) {
        if (retrieve) {
            boolean exists = false;
            if (CharSequenceUtil.isBlank(verifyCodeParam.getAccountType()) || VerifyCodeManager.ACCOUNT_TYPE_MOBILE.equals(verifyCodeParam.getAccountType())) {
                exists = userRepoProc.existsMobile(verifyCodeParam.getAccount());
                Assert.isTrue(exists, "未查询到用户信息，请输入正确的手机号");
            } else if (VerifyCodeManager.ACCOUNT_TYPE_EMAIL.equals(verifyCodeParam.getAccountType())) {
                exists = userRepoProc.existsEmail(verifyCodeParam.getAccount());
                Assert.isTrue(exists, "未查询到用户信息，请输入正确的邮箱");
            }
        } else {
            // 修改当前密码
            var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
            if (VerifyCodeManager.ACCOUNT_TYPE_MOBILE.equals(verifyCodeParam.getAccountType())) {
                Assert.isTrue(Objects.equals(currentUser.getUser().getMobile(), verifyCodeParam.getAccount()), "手机号输入错误");
            } else if (VerifyCodeManager.ACCOUNT_TYPE_EMAIL.equals(verifyCodeParam.getAccountType())) {
                Assert.isTrue(Objects.equals(currentUser.getUser().getEmail(), verifyCodeParam.getAccount()), "邮箱输入错误");
            }
        }
        if (StrUtil.isBlank(verifyCodeParam.getAccountType())) {
            verifyCodeParam.setAccountType(VerifyCodeManager.ACCOUNT_TYPE_MOBILE);
        }

        String type = retrieve ? VerifyCodeManager.BUSINESS_TYPE_RETRIEVE_PWD : VerifyCodeManager.BUSINESS_TYPE_UPDATE_PWD;
        var verifyCode = verifyCodeManager.send(type, verifyCodeParam);
        if (verifyCode == null) {
            return ApiResult.ok();
        }

        return ApiResult.ok("模拟发送验证码，验证码是" + verifyCode);
    }

    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @TenantOrgTransaction(useTenantOrg = false)
    @Override
    public ApiResult<Long> updatePwdByVerifyCode(boolean retrieve, PasswordUpdateSaveVO saveVO) {
        GeneralUserDetails currentUser = retrieve ? null : SecurityContextUtil.currentUserIfUnauthorizedThrow();

        // 先验证验证码
        if (StrUtil.isBlank(saveVO.getAccountType())) {
            saveVO.setAccountType(VerifyCodeManager.ACCOUNT_TYPE_MOBILE);
        }
        String type = retrieve ? VerifyCodeManager.BUSINESS_TYPE_RETRIEVE_PWD : VerifyCodeManager.BUSINESS_TYPE_UPDATE_PWD;
        var verifyMsg = verifyCodeManager.verify(type, saveVO.getAccount(), saveVO.getVerifyCode());
        if (verifyMsg != null) {
            return ApiResult.fail(verifyMsg);
        }

        // 修改密码
        List<Long> userIds = null;
        switch (saveVO.getAccountType()) {
            case VerifyCodeManager.ACCOUNT_TYPE_MOBILE:
                userIds = userRepoProc.getIdByMobile(saveVO.getAccount());
                break;
            case VerifyCodeManager.ACCOUNT_TYPE_EMAIL:
                userIds = userRepoProc.getIdByEmail(saveVO.getAccount());
                break;
            default:
                throw new IllegalArgumentException("暂不支持的账号类型");
        }
        if (CollectionUtils.isEmpty(userIds)) {
            return ApiResult.fail("未查询到用户信息，请确认账号输入正确");
        }

        if (currentUser != null) {
            // 登录用户只能修改自己的
            if (userIds.contains(currentUser.getUserId())) {
                userMngManager.updatePassword(currentUser.getUserId(), saveVO.getPassword(), null, false);
                return ApiResult.ok(currentUser.getUserId());
            }

            return ApiResult.fail("修改失败，您尚未绑定该账号");
        }

        if (userIds.size() > 1) {
            return ApiResult.fail("账号不唯一，无法确定唯一用户");
        }
        userMngManager.updatePassword(userIds.get(0), saveVO.getPassword(), null, false);

        return ApiResult.ok(userIds.get(0));
    }

    @Override
    public ApiResult<List<UserMenuRespVO>> getUserMenu(Boolean includeAction, Boolean tree) {
        var menuList = permissionQueryManager.queryUserMenusCacheable(ObjUtil.defaultIfNull(includeAction, false), ObjUtil.defaultIfNull(tree, true));
        return ApiResult.ok(menuList);
    }

    @Override
    public ApiResult<List<UserMenuRespVO>> getUserAction(String menuCode) {
        var actionList = StringUtils.hasText(menuCode) ? permissionQueryManager.queryUserActionByMenu(menuCode) : permissionQueryManager.queryAllUserAction();
        return ApiResult.ok(actionList);
    }

    @Override
    public ApiResult<List<UserFieldRespVO>> getUserField(String menuCode, String apiCode) {
        var fieldList = permissionQueryManager.queryUserField(menuCode, apiCode);
        return ApiResult.ok(fieldList);
    }

    @Override
    public ApiResult<String> getFrontTableCfg(String tableCode) {
        return frontTableCfgService.getUserCfg(tableCode);
    }

    @Override
    public ApiResult<SysUserDTO> switchTenant(Long tenantId) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        if (currentUser.getTenantId() != null && currentUser.getTenantId().longValue() == tenantId) {
            // 无需切换
            return ApiResult.fail("已在[" + currentUser.getTenant().getTenantName() + "]，无需切换");
        }

        // 获取指定租户
        var user = currentUser.getUser();
        if (CollectionUtils.isEmpty(user.getSysTenantDTOList())) {
            return ApiResult.fail("当前用户无租户");
        }
        var tenant = user.getSysTenantDTOList().stream().filter(t -> t.getId().longValue() == tenantId).findAny();
        if (tenant.isEmpty()) {
            return ApiResult.fail("切换失败，不在指定租户下");
        }

        // 切换租户
        switchCurrentTenant(currentUser, tenant.get());

        return ApiResult.ok(user);
    }

    @Override
    public ApiResult<SysUserDTO> switchOrg(Long orgId) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        if (currentUser.getOrgId() != null && currentUser.getOrgId().longValue() == orgId) {
            // 已在指定组织下
            return ApiResult.fail("已在[" + currentUser.getUser().getOrg().getName() + "]，无需切换");
        }

        // 获取指定组织
        var orgList = currentUser.getUser().getOrgList();
        if (CollectionUtils.isEmpty(orgList)) {
            return ApiResult.fail("当前用户无组织");
        }
        var org = orgList.stream().filter(t -> t.getId().longValue() == orgId).findAny();
        if (org.isEmpty()) {
            return ApiResult.fail("切换失败，不在指定组织下");
        }

        // 切换组织
        switchCurrentOrg(currentUser, org.get());
        return ApiResult.ok(currentUser.getUser());
    }

    @Override
    public ApiResult<UserDataPermissionRespVO> getDataPermission(Boolean all) {
        all = all != null && all;
        UserDataPermissionRespVO respVO = new UserDataPermissionRespVO();
        // 用户信息
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        respVO.setUsername(currentUser.getUsername());
        respVO.setFullName(currentUser.getUser().getPrettyName());
        respVO.setRoles(currentUser.getUser().getRoles());
        HttpServletRequest request = HttpServletUtil.currentRequest();
        if (request == null) {
            return ApiResult.ok(respVO);
        }

        // 数据权限查询
        var roleRule = all ? DataSecurityUtil.getAllDataPermission() : DataSecurityUtil.getDataPermission();

        // 数据格式转换
        respVO.setRowRuleList(this.convert2VO(roleRule == null ? null : roleRule.getSysDprRoleApiDataRuleListQueryDTO(), this::convert2VO));
        respVO.setApiFieldList(this.convert2VO(roleRule == null ? null : roleRule.getSysDpcRoleApiFieldsDTOList(), this::convert2VO));

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<AuthedUserFieldRespVO>> getFieldPermission() {
        var respVoList = DataSecurityUtil.getFieldPermission().stream()
                .map(PermissionConverter.INSTANCE::convertApiFieldRespVO)
                .collect(Collectors.toList());
        return ApiResult.ok(respVoList);
    }

    private <T extends BaseDataSecurityRuleDTO, R extends UserDataPermissionRespVO.Base> List<R> convert2VO(List<T> permissionList,
                                                                                                            Function<T, R> convert) {
        if (CollUtil.isEmpty(permissionList)) {
            return Collections.emptyList();
        }

        var roleMap = SecurityContextUtil.currentUserIfUnauthorizedThrow().getUser().getRoles()
                .stream().collect(Collectors.toMap(IdCodeNameParam::getCode, IdCodeNameParam::getName, (t1, t2) -> t1));
        var businessObjectCodes = permissionList.stream().map(BaseDataSecurityRuleDTO::getBusinessObjectCode).filter(StringUtils::hasText).collect(Collectors.toSet());
        var businessOperationCodes = permissionList.stream().flatMap(t -> Stream.of(t.getApiPermissionCode(), t.getPermissionRef()))
                .filter(StringUtils::hasText).collect(Collectors.toSet());
        var menuCodes = permissionList.stream().map(BaseDataSecurityRuleDTO::getMenusCode).filter(StringUtils::hasText).collect(Collectors.toSet());

        // 查询数据信息
        Map<String, String> appMap = new HashMap<>(32);
        Map<String, MenuSimpleBO> menuMap = new HashMap<>(menuCodes.size());
        Map<String, BusinessObjectBO> businessObjectMap = new HashMap<>(menuCodes.size());
        Map<String, BusinessOperationBO> businessOperationMap = new HashMap<>(menuCodes.size());
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            for (IdCodeNameParam allParam : appRepoProc.allParams(null)) {
                appMap.put(allParam.getCode(), allParam.getName());
            }

            if (!menuCodes.isEmpty()) {
                for (var menuSimpleBO : menuRepoProc.listSimpleByMenuCode(menuCodes)) {
                    menuMap.put(menuSimpleBO.getMenusCode(), menuSimpleBO);
                }
            }
            if (!businessObjectCodes.isEmpty()) {
                for (var businessObjectSimpleBO : businessObjectRepoProc.listSimple(businessObjectCodes)) {
                    businessObjectMap.put(businessObjectSimpleBO.getCode(), businessObjectSimpleBO);
                }
            }
            if (!businessOperationCodes.isEmpty()) {
                for (var businessOperationSimpleBO : businessOperationRepoProc.listSimple(businessOperationCodes, null)) {
                    businessOperationMap.put(businessOperationSimpleBO.getOperationCode(), businessOperationSimpleBO);
                }
            }

            return null;
        });

        return permissionList.stream().map(t -> {
            var permission = convert.apply(t);

            // 权限类型
            ObjUtil.ifNotNull(DataPermissionType.fromValue(permission.getPermissionType()),
                    p -> permission.setPermissionTypeName(p.getDescription()));
            ObjUtil.ifNotNull(businessOperationMap.get(permission.getPermissionRef()), p -> {
                permission.setPermissionRefName(CharSequenceUtil.blankToDefault(p.getCustomName(), p.getOperationDescription()));
            });
            permission.setRoleName(roleMap.get(permission.getRoleCode()));

            // 应用
            permission.setAppName(appMap.get(permission.getAppCode()));
            // 业务对象
            ObjUtil.ifNotNull(businessObjectMap.get(t.getBusinessObjectCode()), p -> permission.setBusinessObjectName(CharSequenceUtil.blankToDefault(p.getCustomName(), p.getName())));
            // 业务操作信息
            ObjUtil.ifNotNull(businessOperationMap.get(permission.getApiPermissionCode()), p -> {
                permission.setApiPermissionName(CharSequenceUtil.blankToDefault(p.getCustomName(), p.getOperationDescription()));
                permission.setApiPermissionPath(p.getApiUrl());
                permission.setApiPermissionRequestType(p.getApiMethod());

            });
            // 菜单
            ObjUtil.ifNotNull(menuMap.get(permission.getMenusCode()), p -> {
                permission.setMenusName(p.getMenusName());
                permission.setMenusRoute(p.getMenusRoute());
            });

            return permission;
        }).collect(Collectors.toList());
    }

    private UserDataPermissionRespVO.RowRule convert2VO(SysDprRoleApiDataRuleListQueryDTO rule) {
        var permission = PermissionConverter.INSTANCE.convertRowRule(rule);
        ObjUtil.ifNotBlank(DprRuleRelationEnum.getValue(permission.getDprRuleRelation()), permission::setDprRuleRelationName);
        ObjUtil.ifNotBlank(DprRuleConditionEnum.getValue(permission.getDprRuleCondition()), permission::setDprRuleConditionName);
        ObjUtil.ifNotBlank(DprRuleValueTypeEnum.getValue(permission.getDprRuleValueType()), permission::setDprRuleValueTypeName);

        return permission;
    }

    private UserDataPermissionRespVO.ApiField convert2VO(SysDpcRoleApiFieldsDTO rule) {
        return PermissionConverter.INSTANCE.convertApiField(rule);
    }

    private void switchCurrentTenant(GeneralUserDetails currentUser, SysTenantDTO tenantDTO) {
        var user = currentUser.getUser();
        currentUser.getUser().setSysTenantVO(tenantDTO);

        // 查询用户的组织
        var userOrg = employeeOrgManager.queryOrgByUser(currentUser.getUserId());
        user.setOrgList(userOrg.getOrgList());
        user.setOrg(userOrg.getOrg());
        user.setTenantOrg(userOrg.getTenantOrg());
        user.setTenantOrgAdminId(userOrg.getTenantOrgAdminId());

        // 查询角色
        var roleCodes = permissionQueryManager.queryRoleByUser(user);
        user.setRoleCodes(normalizeLoginUserRole(roleCodes));

        // 更新用户信息
        SecurityContextUtil.updateCurrentUser(currentUser);
    }

    private void switchCurrentOrg(GeneralUserDetails currentUser, SysOrgBasicDTO org) {
        var user = currentUser.getUser();
        user.setOrg(org);

        // 查询租户组织
        var tenantOrgAdminMap = employeeOrgManager.getTenantOrg(org.getId());
        if (!tenantOrgAdminMap.isEmpty()) {
            for (Map.Entry<SysOrgBasicDTO, Long> entry : tenantOrgAdminMap.entrySet()) {
                user.setTenantOrg(entry.getKey());
                user.setTenantOrgAdminId(entry.getValue());
            }
        }

        // 查询角色
        var roleCodes = permissionQueryManager.queryRoleByUser(user);
        user.setRoleCodes(normalizeLoginUserRole(roleCodes));

        // 查询公司
        if (org.getOuId() != null) {
            var ou = ouQueryManager.getOuBasicDTO(org.getOuId());
            if (ou != null && Boolean.TRUE.equals(ou.getEnabled())) {
                user.setOuId(ou.getId());
                user.setOuCode(ou.getOuCode());
                user.setOuName(ou.getOuName());
            }
        }

        // 更新用户信息
        SecurityContextUtil.updateCurrentUser(currentUser);
    }

    private Set<String> normalizeLoginUserRole(Set<String> roleCodes) {
        if (CollectionUtils.isEmpty(roleCodes)) {
            return Collections.emptySet();
        }
        if (!StringUtils.hasText(authorizationProperties.getRolePrefix())) {
            return roleCodes;
        }

        var prefix = authorizationProperties.getRolePrefix();
        return roleCodes.stream().map(t -> t.indexOf(prefix) < 0 ? prefix + t : t).collect(Collectors.toSet());
    }
}
