package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.boot.openfeign.common.DynamicClientHelper;
import com.elitescloud.boot.swagger.openapi.common.OpenApiRpcClient;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.service.OpenApiQueryService;
import com.elitescloud.cloudt.system.service.common.constant.OpenApiStatusEnum;
import com.elitescloud.cloudt.system.service.repo.OpenApiInfoRepoProc;
import com.elitescloud.cloudt.system.service.repo.OpenApiResourceRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.StreamUtils;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import java.nio.charset.StandardCharsets;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/12
 */
@Slf4j
@Service
public class OpenApiQueryServiceImpl implements OpenApiQueryService {

    @Autowired
    private OpenApiInfoRepoProc openApiInfoRepoProc;
    @Autowired
    private OpenApiResourceRepoProc resourceRepoProc;

    @Override
    public HttpEntity<StreamingResponseBody> downloadOpenApi(Long openApiId) {
        var openApiInfo = openApiInfoRepoProc.get(openApiId);
        if (openApiInfo == null) {
            return this.emptyBody("OpenApi不存在");
        }
        if (!OpenApiStatusEnum.FINISHED.name().equals(openApiInfo.getState())) {
            return this.emptyBody("OpenApi尚未构建成功");
        }

        byte[] resource = openApiInfo.getResourceId() == null ? null : resourceRepoProc.getResourceBytes(openApiInfo.getResourceId());
        if (resource == null || resource.length == 0) {
            return this.emptyBody("OpenApi未发现有效资源");
        }

        var contentDisposition = ContentDisposition.builder("attachment")
                .filename(openApiInfo.getAppCode() + "[OpenApi].json", StandardCharsets.UTF_8)
                .build();
        return ResponseEntity.ok()
                .contentType(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.CONTENT_DISPOSITION, contentDisposition.toString())
                .contentLength(resource.length)
                .body(outputStream -> {
                    try {
                        StreamUtils.copy(resource, outputStream);
                    } catch (Exception e) {
                        log.error("下载资源文件异常：", e);
                    }
                });
    }

    @Override
    public HttpEntity<StreamingResponseBody> downloadOpenApiByRpc(String appCode) {
        // 构建客户端实例
        var openApiClient = DynamicClientHelper.getClient(appCode, OpenApiRpcClient.class, OpenApiRpcClient.URI);
        // rpc调用获取openApi信息
        ApiResult<String> openApiResult = null;
        try {
            openApiResult = openApiClient.getOpenApiJsonPretty();
        } catch (Exception e) {
            log.error("调用服务{}获取OpenAPI异常：", appCode, e);
            return this.emptyBody("调用服务异常，请确认服务" + appCode + "在线且支持OpenApi下载");
        }

        if (openApiResult.isFailed() || openApiResult.getData() == null) {
            return this.emptyBody(openApiResult.getMsg());
        }
        String resource = openApiResult.getData();

        var contentDisposition = ContentDisposition.builder("attachment")
                .filename(appCode + "[OpenApi].json", StandardCharsets.UTF_8)
                .build();
        return ResponseEntity.ok()
                .contentType(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.CONTENT_DISPOSITION, contentDisposition.toString())
                .body(outputStream -> {
                    try {
                        StreamUtils.copy(resource.getBytes(StandardCharsets.UTF_8), outputStream);
                    } catch (Exception e) {
                        log.error("下载资源文件异常：", e);
                    }
                });
    }

    private HttpEntity<StreamingResponseBody> emptyBody(String msg) {
        var contentDisposition = ContentDisposition.builder("attachment")
                .filename(msg + ".txt", StandardCharsets.UTF_8)
                .build();
        return ResponseEntity.ok()
                .contentType(MediaType.TEXT_PLAIN)
                .header(HttpHeaders.CONTENT_DISPOSITION, contentDisposition.toString())
                .body(outputStream -> {
                    try {
                        StreamUtils.copy(msg.getBytes(StandardCharsets.UTF_8), outputStream);
                    } catch (Exception e) {
                        log.error("下载资源文件异常：", e);
                    }
                });
    }
}
