package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.SysRoleBusiness;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.RoleConvert;
import com.elitescloud.cloudt.system.model.vo.query.role.RolePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.GroupRoleRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RoleDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RolePageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.role.RoleCloneSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.role.RoleSaveVO;
import com.elitescloud.cloudt.system.service.RoleMngService;
import com.elitescloud.cloudt.system.service.common.constant.BelongType;
import com.elitescloud.cloudt.system.service.manager.PermissionMngManager;
import com.elitescloud.cloudt.system.service.manager.RoleMngManager;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleDO;
import com.elitescloud.cloudt.system.service.repo.RoleGroupRepoProc;
import com.elitescloud.cloudt.system.service.repo.RoleRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/13
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class RoleMngServiceImpl extends BaseServiceImpl implements RoleMngService {
    private static final RoleConvert CONVERT = RoleConvert.INSTANCE;

    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private RoleGroupRepoProc roleGroupRepoProc;
    @Autowired
    private RoleMngManager roleManager;
    @Autowired
    private PermissionMngManager permissionManager;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(RoleSaveVO saveVO, BelongType.Belonger belonger) {
        // 转换成BO对象
        var saveBO = CONVERT.saveVo2SaveBO(saveVO);
        if (belonger == null) {
            belonger = BelongType.getBelonger(false);
        }
        saveBO.setType(belonger.getBelongType().toString());
        saveBO.setTypeId(belonger.getBelongId());

        // 开始保存
        var roleDO = roleManager.upsert(saveBO);
        return ApiResult.ok(roleDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveClone(RoleCloneSaveVO cloneSaveVO) {
        Assert.notNull(cloneSaveVO.getOriginalId(), "请选择要复制的角色");
        var originalRole = roleRepoProc.get(cloneSaveVO.getOriginalId());
        if (originalRole == null) {
            return ApiResult.fail("角色不存在");
        }

        RoleSaveVO saveVO = CONVERT.cloneSaveVo2SaveVo(cloneSaveVO);
        var saveResult = this.save(saveVO, new BelongType.Belonger(new BelongType(originalRole.getType()), originalRole.getTypeId()));
        if (saveResult.isFailed()) {
            return saveResult;
        }

        // 复制权限信息
        permissionManager.cloneRolePermission(cloneSaveVO.getOriginalId(), saveResult.getData());

        return saveResult;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        Boolean enabled = roleRepoProc.getEnabled(id);
        enabled = enabled == null || !enabled;
        roleManager.updateEnabled(id, enabled);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateName(Long id, String name) {
        Assert.notNull(id, "ID为空");
        Assert.hasText(name, "角色名称为空");

        roleManager.updateName(id, name);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        roleManager.delete(id);

        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<RoleDetailRespVO> get(Long id) {
        return roleRepoProc.getOptional(id)
                .map(t -> {
                    var vo = CONVERT.do2DetailRespVO(t);
                    vo.setBusinessKeyName(super.udcValue(new SysRoleBusiness(vo.getBusinessKey())));

                    // 分组
                    if (t.getGroupId() == null || t.getGroupId() == SysRoleDO.DEFAULT_GROUP_ID) {
                        vo.setGroupName(SysRoleDO.DEFAULT_GROUP_NAME);
                    } else {
                        vo.setGroupName(roleGroupRepoProc.getName(t.getGroupId()));
                    }

                    return vo;
                })
                .map(ApiResult::ok)
                .orElse(ApiResult.noData());
    }

    @Override
    public ApiResult<PagingVO<RolePageRespVO>> page(RolePageQueryVO queryVO) {
        var businessKeyNameMap = super.udcMap(new SysRoleBusiness());

        Set<Long> groupIds = new HashSet<>(16);
        PagingVO<RolePageRespVO> pageResult = roleRepoProc.pageMng(BelongType.getBelonger(), queryVO)
                .map(t -> {
                    var vo = CONVERT.do2PageRespVO(t);
                    vo.setBusinessKeyName(businessKeyNameMap.get(vo.getBusinessKey()));
                    if (t.getGroupId() == null || SysRoleDO.DEFAULT_GROUP_ID == t.getGroupId()) {
                        vo.setGroupId(SysRoleDO.DEFAULT_GROUP_ID);
                        vo.setGroupName(SysRoleDO.DEFAULT_GROUP_NAME);
                    } else {
                        groupIds.add(t.getGroupId());
                    }
                    return vo;
                });

        // 查询分钟
        if (!groupIds.isEmpty()) {
            var groupMap = roleGroupRepoProc.queryByList(groupIds).stream().collect(Collectors.toMap(IdCodeNameParam::getId, IdCodeNameParam::getName, (t1, t2) -> t1));
            pageResult.each(t -> {
                if (t.getGroupId() != null && t.getGroupId() != SysRoleDO.DEFAULT_GROUP_ID) {
                    t.setGroupName(groupMap.get(t.getGroupId()));
                }
            });
        }

        return ApiResult.ok(pageResult);
    }

    @Override
    public ApiResult<List<GroupRoleRespVO>> listGroupRole(Boolean showDisabled, BelongType.Belonger belonger) {
        if (showDisabled == null) {
            showDisabled = false;
        }

        var roleList = roleManager.listGroupRole(showDisabled, BelongType.getBelongerPersonal());
        return ApiResult.ok(roleList);
    }
}
