package com.elitescloud.cloudt.system.service.impl;

/**
 * @author : chen.niu
 * @description :
 * @date : 2024-05-13 16:15
 */

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.model.entity.QSysPositionDO;
import com.elitescloud.cloudt.system.model.entity.SysPositionDO;
import com.elitescloud.cloudt.system.model.entity.SysPositionOrgDO;
import com.elitescloud.cloudt.system.model.entity.SysPositionRoleDO;
import com.elitescloud.cloudt.system.model.vo.query.position.*;
import com.elitescloud.cloudt.system.service.OrgQueryService;
import com.elitescloud.cloudt.system.service.RoleQueryService;
import com.elitescloud.cloudt.system.service.SysPositionService;
import com.elitescloud.cloudt.system.service.repo.SysPositionOrgRepo;
import com.elitescloud.cloudt.system.service.repo.SysPositionRepository;
import com.elitescloud.cloudt.system.service.repo.SysPositionRoleRepo;
import com.elitescloud.cloudt.system.service.util.JpaPredicateBuilder;
import com.querydsl.core.types.Predicate;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

@Service
@Slf4j
public class SysPositionServiceImpl implements SysPositionService {


    private final SysPositionRepository sysPositionRepository;
    private final SysPositionOrgRepo sysPositionOrgRepo;
    private final SysPositionRoleRepo sysPositionRoleRepo;
    private final RoleQueryService roleQueryService;
    private final OrgQueryService orgQueryService;

    public SysPositionServiceImpl(SysPositionRepository sysPositionRepository, SysPositionOrgRepo sysPositionOrgRepo, SysPositionRoleRepo sysPositionRoleRepo, RoleQueryService roleQueryService, OrgQueryService orgQueryService) {
        this.sysPositionRepository = sysPositionRepository;
        this.sysPositionOrgRepo = sysPositionOrgRepo;
        this.sysPositionRoleRepo = sysPositionRoleRepo;
        this.roleQueryService = roleQueryService;
        this.orgQueryService = orgQueryService;
    }

    @Override
    public ApiResult<List<SysPositionVo>> getAllPositions(SysPositionQueryVo positionParam) {
        QSysPositionDO qdo = QSysPositionDO.sysPositionDO;
        Predicate predicate = JpaPredicateBuilder.builder()
                .and(qdo.positionCode::eq, positionParam.getPositionCode())
                .and(qdo.positionName::eq, positionParam.getPositionName())
                .and(qdo.positionDescription::like, StringUtils.hasText(positionParam.getPositionDescription()) ? "%" + positionParam.getPositionDescription() + "%" : null)
                .getPredicate();
        List<SysPositionDO> sysPositionDOS = new java.util.ArrayList<>();

        sysPositionRepository.findAll(predicate).iterator().forEachRemaining(positionDO -> {
            sysPositionDOS.add(positionDO);
        });

        if (sysPositionDOS != null) {
            List<SysPositionVo> sysPositionVos = sysPositionDOS.stream().map(sysPositionDO -> {
                SysPositionVo sysPositionVo = SysPositionConvert.INSTANCE.doToVo(sysPositionDO);
                List<OrgVo> orgCode = findOrgsByPosition(sysPositionDO.getPositionCode());
                List<RoleVo> roleCode = findRolesByPosition(sysPositionDO.getPositionCode());
                sysPositionVo.setRoleList(roleCode);
                sysPositionVo.setOrgList(orgCode);
                return sysPositionVo;
            }).collect(Collectors.toList());
            return ApiResult.ok(sysPositionVos);
        } else {
            return ApiResult.ok(new ArrayList<>());
        }

    }
    @Override
    public List<RoleVo> findRolesByPosition(String positionCode) {
        Set<String> roleCodes = sysPositionRoleRepo.findByPositionCode(positionCode).stream().map(sysPositionRoleDO -> {
            return sysPositionRoleDO.getRoleCode();
        }).collect(Collectors.toSet());
        if (roleCodes == null || roleCodes.size() == 0) {
            return new ArrayList<>();
        }
        ApiResult<List<IdCodeNameParam>> apiResult = roleQueryService.listCodeNamesByCode(roleCodes);
        if (apiResult.isSuccess() && apiResult.getData() != null && apiResult.getData().size() > 0) {
            return apiResult.getData().stream().map(idCodeNameParam -> {
                RoleVo roleVo = new RoleVo();
                roleVo.setRoleCode(idCodeNameParam.getCode());
                roleVo.setRoleName(idCodeNameParam.getName());
                roleVo.setRoleId(idCodeNameParam.getId());
                return roleVo;
            }).collect(Collectors.toList());
        } else {
            log.error("角色查询失败：" + roleCodes);
            return new ArrayList<>();
        }
    }



    @Override
    public List<OrgVo> findOrgsByPosition(String positionCode) {
        Set<String> orgCodes = sysPositionOrgRepo.findByPositionCode(positionCode).stream().map(sysPositionRoleDO -> {
            return sysPositionRoleDO.getOrgCode();
        }).collect(Collectors.toSet());
        if (orgCodes == null || orgCodes.size() == 0) {
            return new ArrayList<>();
        }
        ApiResult<List<IdCodeNameParam>> apiResult = orgQueryService.queryIdAndName(orgCodes);
        if (apiResult.isSuccess() && apiResult.getData() != null && apiResult.getData().size() > 0) {
            return apiResult.getData().stream().map(idCodeNameParam -> {
                OrgVo orgVo = new OrgVo();
                orgVo.setOrgCode(idCodeNameParam.getCode());
                orgVo.setOrgName(idCodeNameParam.getName());
                orgVo.setOrgId(idCodeNameParam.getId());
                return orgVo;
            }).collect(Collectors.toList());
        } else {
            log.error("崗位查询失败：" + orgCodes);
            return new ArrayList<>();
        }

    }

    @Override
    public ApiResult<SysPositionVo> getPositionById(Long id) {
        var sysPositionDO = sysPositionRepository.findById(id).orElse(null);
        if (sysPositionDO != null) {
            SysPositionVo sysPositionVo = SysPositionConvert.INSTANCE.doToVo(sysPositionDO);
            List<OrgVo> orgCode = findOrgsByPosition(sysPositionDO.getPositionCode());
            List<RoleVo> roleCode = findRolesByPosition(sysPositionDO.getPositionCode());
            sysPositionVo.setRoleList(roleCode);
            sysPositionVo.setOrgList(orgCode);
            return ApiResult.ok(sysPositionVo);
        }else{
            return ApiResult.fail("id岗位不存在:"+id);
        }
    }
    @Override
    public ApiResult<SysPositionVo> getPositionByCode(String positionCode) {
        var sysPositionDO = sysPositionRepository.findOneByPositionCode(positionCode);
        if (sysPositionDO != null) {
            SysPositionVo sysPositionVo = SysPositionConvert.INSTANCE.doToVo(sysPositionDO);
            List<OrgVo> orgCode = findOrgsByPosition(sysPositionDO.getPositionCode());
            List<RoleVo> roleCode = findRolesByPosition(sysPositionDO.getPositionCode());
            sysPositionVo.setRoleList(roleCode);
            sysPositionVo.setOrgList(orgCode);
            return ApiResult.ok(sysPositionVo);
        }else{
            return ApiResult.fail("岗位編碼不存在:"+positionCode);
        }
    }
    @Override
    @Transactional
    public ApiResult<Long> createPosition(SysPositionSaveVo position) {
        List<SysPositionDO> sysPositionDOS =
                sysPositionRepository.findByPositionCode(position.getPositionCode());
        if (sysPositionDOS != null && sysPositionDOS.size() > 0) {
            throw new BusinessException("岗位编码已存在");
        }

        SysPositionDO sysPositionDO = new SysPositionDO();
        sysPositionDO.setPositionCode(position.getPositionCode());
        sysPositionDO.setPositionName(position.getPositionName());
        sysPositionDO.setPositionDescription(position.getPositionDescription());
        sysPositionRepository.save(sysPositionDO);
        //添加岗位和组织关联
        List<String> orgCodes = position.getOrgCodes();
        if (orgCodes != null && orgCodes.size() > 0) {
            List<SysPositionOrgDO> sysPositionOrgDOS = getSysPositionOrgDOS(position.getPositionCode(), orgCodes);
            sysPositionOrgRepo.saveAll(sysPositionOrgDOS);
        }
        //添加岗位和角色关联
        List<String> roleCode = position.getRoleCodes();
        if (roleCode != null && roleCode.size() > 0) {
            List<SysPositionRoleDO> sysPositionRoleDOS = getSysPositionRoleDOS(position.getPositionCode(), roleCode);
            sysPositionRoleRepo.saveAll(sysPositionRoleDOS);
        }



        return ApiResult.ok(sysPositionDO.getId());
    }

    @Override
    @Transactional
    public ApiResult<Boolean> updatePosition(Long id, SysPositionSaveVo updateParam) {
        SysPositionDO existingPosition = sysPositionRepository.findById(id)
                .orElseThrow(() -> new BusinessException("岗位ID不存在:" + id));
//            existingPosition.setPositionCode(newPosition.getPositionCode());
        existingPosition.setPositionName(updateParam.getPositionName());
        existingPosition.setPositionDescription(updateParam.getPositionDescription());
        sysPositionRepository.save(existingPosition);
        //修改,全删全插岗位和组织关联
        List<String> orgCodes = updateParam.getOrgCodes();
        if (orgCodes != null && orgCodes.size() > 0) {
            sysPositionOrgRepo.deleteByPositionCode(existingPosition.getPositionCode());
            List<SysPositionOrgDO> sysPositionOrgDOS = getSysPositionOrgDOS(existingPosition.getPositionCode(), orgCodes);
            sysPositionOrgRepo.saveAll(sysPositionOrgDOS);
        }
        //修改,全删全插岗位和角色关联
        List<String> roleCode = updateParam.getRoleCodes();
        if (roleCode != null && roleCode.size() > 0) {
            sysPositionRoleRepo.deleteByPositionCode(existingPosition.getPositionCode());
            List<SysPositionRoleDO> sysPositionRoleDOS = getSysPositionRoleDOS(existingPosition.getPositionCode(), roleCode);
            sysPositionRoleRepo.saveAll(sysPositionRoleDOS);
        }

        return ApiResult.ok(true);
    }

    @NotNull
    private static List<SysPositionOrgDO> getSysPositionOrgDOS(String positionCode, List<String> orgCodes) {
        List<SysPositionOrgDO> sysPositionOrgDOS = orgCodes.stream()
                .map(s -> {
                    SysPositionOrgDO sysPositionOrgDO = new SysPositionOrgDO();
                    sysPositionOrgDO.setOrgCode(s);
                    sysPositionOrgDO.setPositionCode(positionCode);
                    return sysPositionOrgDO;
                }).collect(Collectors.toList());
        return sysPositionOrgDOS;
    }

    @NotNull
    private static List<SysPositionRoleDO> getSysPositionRoleDOS(String positionCode, List<String> roleCode) {
        List<SysPositionRoleDO> sysPositionRoleDOS = roleCode.stream()
                .map(s -> {
                    SysPositionRoleDO sysPositionOrgDO = new SysPositionRoleDO();
                    sysPositionOrgDO.setRoleCode(s);
                    sysPositionOrgDO.setPositionCode(positionCode);
                    return sysPositionOrgDO;
                }).collect(Collectors.toList());
        return sysPositionRoleDOS;
    }

    @Override
    public ApiResult<Boolean> deletePosition(Long id) {
        sysPositionRepository.deleteById(id);
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<Boolean> deleteAllById(List<Long> id) {
        sysPositionRepository.deleteAllById(id);
        return ApiResult.ok(true);
    }
}
