package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.cacheable.SysCacheSettingRpcService;
import com.elitescloud.cloudt.system.convert.SysSettingConvert;
import com.elitescloud.cloudt.system.model.entity.SysSettingDO;
import com.elitescloud.cloudt.system.param.SysSettingQParam;
import com.elitescloud.cloudt.system.service.ISysSettingService;
import com.elitescloud.cloudt.system.service.dto.SysSettingDTO;
import com.elitescloud.cloudt.system.service.repo.SysSettingRepo;
import com.elitescloud.cloudt.system.service.repo.SysSettingRepoProc;
import com.elitescloud.cloudt.system.vo.SysSettingVO;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.transaction.Transactional;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author mir
 */
@Service
public class SysSettingServiceImpl implements ISysSettingService {

    private final SysSettingRepo sysSettingRepo;
    private final SysSettingRepoProc sysSettingRepoProc;

    @Autowired
    private SysCacheSettingRpcService cacheSettingRpcService;

    public SysSettingServiceImpl(SysSettingRepo sysSettingRepo,
                                 SysSettingRepoProc sysSettingRepoProc) {
        this.sysSettingRepo = sysSettingRepo;
        this.sysSettingRepoProc = sysSettingRepoProc;
    }

    @Override
    public PagingVO<SysSettingVO> search(SysSettingQParam param) {
        return sysSettingRepoProc.pageMng(param)
                .map(SysSettingConvert.INSTANCE::doToVO);
    }

    @Override
    @Cacheable(value = "sys-setting", key = "#root.methodName")
    public List<SysSettingVO> all() {
        return sysSettingRepo.findAll().stream().map(SysSettingConvert.INSTANCE::doToVO)
                .collect(Collectors.toList());
    }

    @Override
    public SysSettingVO oneById(Long id) {
        return sysSettingRepo.findById(id).map(SysSettingConvert.INSTANCE::doToVO).orElse(null);
    }

    @Override
    public SysSettingVO oneByNo(String no) {
        return sysSettingRepo.findBySettingNo(no).map(SysSettingConvert.INSTANCE::doToVO).orElse(null);
    }

    @Override
    @Transactional(rollbackOn = {Exception.class})
    @CacheEvict(value = "sys-setting", allEntries = true)
    public Long create(SysSettingVO sysSetting) {
        sysSettingRepo.findBySettingNo(sysSetting.getSettingNo()).ifPresent(s -> {
            throw new BusinessException("配置：" + s.getSettingNo() + ", 已经存在");
        });

        val newSetting = sysSettingRepo.<SysSettingDO>save(SysSettingConvert.INSTANCE.voToDO(sysSetting));

        // 清空缓存
        clearCache();

        return newSetting.getId();
    }

    @Override
    @Transactional(rollbackOn = {Exception.class})
    @CacheEvict(value = "sys-setting", allEntries = true)
    public void update(SysSettingVO sysSettingVO) {
        var oldSetting = sysSettingRepo.findById(sysSettingVO.getId());
        if (oldSetting.isEmpty()) {
            throw new BusinessException("设置：" + sysSettingVO.getSettingNo() + ", 未找到");
        }

        var saveSetting = oldSetting.get();
        Assert.isTrue(saveSetting.getSettingNo().equals(sysSettingVO.getSettingNo()), "编号不可修改");

        saveSetting.setSettingVal(sysSettingVO.getSettingVal());
        saveSetting.setSettingName(sysSettingVO.getSettingName());
        saveSetting.setSettingType(sysSettingVO.getSettingType());
        saveSetting.setSettingType2(sysSettingVO.getSettingType2());
        saveSetting.setSettingStatus(sysSettingVO.getSettingStatus());
        saveSetting.setSettingDesc(sysSettingVO.getSettingDesc());
        saveSetting.setDefaultValue(sysSettingVO.getDefaultValue());
        saveSetting.setSortNo(sysSettingVO.getSortNo());

        sysSettingRepo.save(saveSetting);

        // 清空缓存
        clearCache();
    }

    @Override
    @Transactional(rollbackOn = {Exception.class})
    @CacheEvict(value = "sys-setting", allEntries = true)
    public void removeByIds(List<Long> ids) {
        sysSettingRepo.deleteAllByIdIn(ids);

        // 清空缓存
        clearCache();
    }

    @Override
    @Transactional(rollbackOn = {Exception.class})
    @CacheEvict(value = "sys-setting", allEntries = true)
    public void removeByNos(List<String> nos) {
        sysSettingRepo.deleteAllBySettingNoIn(nos);

        // 清空缓存
        clearCache();
    }

    @Override
    public List<SysSettingDTO> findSettingsBySettingNo(String settingNo) {
        return sysSettingRepoProc.findDtosBySettingNo(settingNo);
    }

    private void clearCache() {
        cacheSettingRpcService.clearCache();
    }
}
