package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.exceptions.UtilException;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.lang.UUID;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ZipUtil;
import cn.hutool.poi.excel.cell.CellUtil;
import com.alibaba.fastjson.JSONObject;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.CloudtIdCreator;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.util.*;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.common.BasicRecordStateEnum;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.convert.BasicRecordConvert;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.model.entity.SysBasicRecordDO;
import com.elitescloud.cloudt.system.model.vo.query.devops.BasicDataRecordPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.devops.BasicRecordExportPageRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.devops.BasicRecordImportPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.devops.ExportBasicDataSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.devops.ImportBasicDataSaveVO;
import com.elitescloud.cloudt.system.service.SystemDataService;
import com.elitescloud.cloudt.system.service.common.constant.BasicRecordTypeEnum;
import com.elitescloud.cloudt.system.service.devops.init.AbstractBasicDataInitProvider;
import com.elitescloud.cloudt.system.service.devops.init.BasicDataInitProviderFactory;
import com.elitescloud.cloudt.system.service.repo.BasicRecordRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.WorkbookUtil;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.util.StopWatch;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
@Service
@Slf4j
public class SystemDataServiceImpl extends BaseServiceImpl implements SystemDataService {
    private static final String SUFFIX_ZIP = "zip";
    private static final String SUFFIX_XLSX = "xlsx";
    private static final String FILENAME_MANIFEST = "MANIFEST.properties";
    private static final String PROP_EXPORT_FILE = "_exportFile";
    private static final String PROP_EXPORT_TIME = "_exportTime";
    private static final String PROP_RECORD_ID = "_recordId";
    private static final String PROP_TENANT_CODE = "_tenantCode";
    private static final String PROP_ALL_VER = "_all_version";
    private static final String PROP_STD = "_std";
    private static final String PROP_VERSION = "_version";
    private static final String PROP_ALL_TYPE = "_all_type";
    private static final String PROP_APP_CODES = "_app_codes";
    private static final String PROP_TYPE = "_types";
    private static final String PROP_DESCRIPTION = "_description";
    private final List<CodeNameParam> dataTypeList = BasicDataInitProviderFactory.getExportProviderGroupList();

    @Autowired
    private BasicRecordRepoProc recordRepoProc;
    @Autowired
    private NamedParameterJdbcTemplate jdbcTemplate;
    @Autowired
    private PlatformTransactionManager transactionManager;
    @Autowired
    private CloudtIdCreator idCreator;
    @Autowired
    private TenantClientProvider tenantClientProvider;
    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private FileService<String> fileService;

    @Override
    public ApiResult<List<CodeNameParam>> listDataType() {
        return ApiResult.ok(dataTypeList);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> exportData(ExportBasicDataSaveVO saveVO) {
        // 保存导出记录
        SysBasicRecordDO recordDO = new SysBasicRecordDO();
        recordDO.setRecordType(BasicRecordTypeEnum.EXPORT);
        recordDO.setAllVersion(ObjUtil.defaultIfNull(saveVO.getAllVersion(), true));
        if (!recordDO.getAllVersion()) {
            recordDO.setDataVersion(saveVO.getDataVersion());
            Assert.notBlank(saveVO.getDataVersion(), "请选择数据版本");
        }
        recordDO.setAllType(Boolean.TRUE.equals(saveVO.getAllType()));
        if (!recordDO.getAllType()) {
            var dataTypes = CollUtil.isEmpty(saveVO.getDataTypes()) ? null : saveVO.getDataTypes().stream().filter(StringUtils::hasText).collect(Collectors.joining(","));
            Assert.notBlank(dataTypes, "请选择数据类型");
            recordDO.setDataType(dataTypes);
        }
        recordDO.setTenantCode(CharSequenceUtil.blankToDefault(saveVO.getTenantCode(), systemProperties.getStdProject().getTenantCode()));

        Set<String> appCodes = CollUtil.isEmpty(saveVO.getAppCodes()) ? Collections.emptySet() : saveVO.getAppCodes().stream().filter(StringUtils::hasText).collect(Collectors.toSet());
        if (CollUtil.isNotEmpty(saveVO.getAppCodes())) {
            recordDO.setAppCodes(String.join(",", appCodes));
        }
        recordDO.setOperateTime(LocalDateTime.now());
        recordDO.setDescription(saveVO.getDescription());
        recordDO.setSuccess(false);
        recordDO.setFinished(false);
        recordDO.setState(BasicRecordStateEnum.PREPARING.name());
        recordDO.setStd(Boolean.TRUE.equals(systemProperties.getStdProject().getEnabled()));
        recordRepoProc.save(recordDO);

        // 开始导出
        taskExecutor.execute(() -> this.startExport(recordDO));

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<PagingVO<BasicRecordExportPageRespVO>> pageExportedRecord(BasicDataRecordPageQueryVO queryVO) {
        var dataTypeMap = dataTypeList.stream().collect(Collectors.toMap(CodeNameParam::getCode, CodeNameParam::getName, (t1, t2) -> t1));

        var pageData = recordRepoProc.queryByPage(queryVO, BasicRecordTypeEnum.EXPORT)
                .map(t -> {
                    BasicRecordExportPageRespVO respVO = BasicRecordConvert.INSTANCE.do2ExportPageRespVO(t);

                    if (Boolean.FALSE.equals(respVO.getAllType()) && StringUtils.hasText(t.getDataType())) {
                        var dataTypeList = Arrays.stream(t.getDataType().split(","))
                                .filter(dataTypeMap::containsKey)
                                .map(code -> new CodeNameParam(code, dataTypeMap.get(code)))
                                .collect(Collectors.toList());
                        respVO.setDataTypes(dataTypeList);
                    }

                    return respVO;
                });
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<PagingVO<BasicRecordImportPageRespVO>> pageImportedRecord(BasicDataRecordPageQueryVO queryVO) {
        var dataTypeMap = dataTypeList.stream().collect(Collectors.toMap(CodeNameParam::getCode, CodeNameParam::getName, (t1, t2) -> t1));

        var pageData = recordRepoProc.queryByPage(queryVO, BasicRecordTypeEnum.IMPORT)
                .map(t -> {
                    BasicRecordImportPageRespVO respVO = BasicRecordConvert.INSTANCE.do2ImportPageRespVO(t);

                    if (Boolean.FALSE.equals(respVO.getAllType()) && StringUtils.hasText(t.getDataType())) {
                        var dataTypeList = Arrays.stream(t.getDataType().split(","))
                                .filter(dataTypeMap::containsKey)
                                .map(code -> new CodeNameParam(code, dataTypeMap.get(code)))
                                .collect(Collectors.toList());
                        respVO.setDataTypes(dataTypeList);
                    }

                    return respVO;
                });
        return ApiResult.ok(pageData);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> importData(MultipartFile file, ImportBasicDataSaveVO saveVO) {
        if (file == null || file.isEmpty()) {
            return ApiResult.fail("导入文件为空");
        }
        var importFileName = file.getOriginalFilename();
        var suffix = FileUtil.getSuffix(importFileName);
        var isSupport = SUFFIX_ZIP.equals(suffix);
        if (!isSupport) {
            return ApiResult.fail("解析导入文件失败，请使用导出的原文件");
        }

        // 转成临时文件
        File tempFile = null;
        try {
            tempFile = FileUtil.createTempFile(importFileName, UUID.fastUUID().toString(true));
            file.transferTo(tempFile);
        } catch (IOException e) {
            log.error("转存临时文件异常：", e);
            return ApiResult.fail("导入失败：" + e.getMessage());
        }

        // 解析导入的文件
        StopWatch watch = new StopWatch();
        watch.start();
        try {
            ImportedData importData = this.analyzeImportFile(tempFile, saveVO);
            this.importFromExcel(importData, tempFile);
        } catch (Exception e) {
            log.error("导入数据异常：", e);
            return ApiResult.fail("导入数据异常：" + e.getMessage());
        } finally {
            tempFile.delete();
            watch.stop();
            log.info("导入结束，耗时：{}s", watch.getTotalTimeSeconds());
        }

        return ApiResult.ok(true);
    }

    private Long queryTenantId(String tenantCode) {
        if (CharSequenceUtil.isBlank(tenantCode)) {
            return null;
        }
        return jdbcTemplate.queryForObject("select id from sys_tenant where tenant_code = :tenant_code",
                Map.of("tenant_code", tenantCode), Long.class);
    }

    private List<Long> allTenantId() {
        var allTenants = tenantClientProvider.getAllTenants();
        if (allTenants.isEmpty()) {
            return Collections.emptyList();
        }
        return allTenants.stream().map(SysTenantDTO::getId).collect(Collectors.toList());
    }

    private String sheetName(AbstractBasicDataInitProvider provider) {
        return provider.tableName();
    }

    private void validateExportProvider(List<AbstractBasicDataInitProvider> providers) {
        Set<String> existsProviderNames = new HashSet<>();
        for (AbstractBasicDataInitProvider provider : providers) {
            // 校验是否重复
            var dataTypeName = provider.typeName();
            Assert.notBlank(dataTypeName, () -> new IllegalArgumentException(provider.getClass().getName() + "的typeName为空"));
            if (existsProviderNames.contains(dataTypeName)) {
                throw new IllegalArgumentException("存在重复的数据初始化：" + dataTypeName);
            }
            existsProviderNames.add(dataTypeName);

            // 校验sheetName
            var sheetName = sheetName(provider);
            try {
                WorkbookUtil.validateSheetName(sheetName);
            } catch (Exception e) {
                throw new IllegalArgumentException("sheet名称不合法：" + e.getMessage());
            }

            Assert.notEmpty(provider.fields(), () -> new IllegalArgumentException(provider.getClass().getName() + "的fields为空"));

            Assert.notEmpty(provider.fieldTitles(), () -> new IllegalArgumentException(provider.getClass().getName() + "的fieldTitles为空"));

            // 校验是否有死循环
            if (CollUtil.isNotEmpty(provider.thenImport())) {
                validateChildren(Collections.emptySet(), provider.thenImport());
            }
        }
    }

    private void validateChildren(Set<String> parentNames, List<AbstractBasicDataInitProvider> children) {
        for (AbstractBasicDataInitProvider child : children) {
            var childName = child.getClass().getName();
            if (parentNames.contains(childName)) {
                throw new IllegalArgumentException(childName + "可能出现循环调用");
            }
            if (CollUtil.isNotEmpty(child.thenImport())) {
                var tempParentNames = new HashSet<>(parentNames);
                tempParentNames.add(childName);
                this.validateChildren(tempParentNames, child.thenImport());
            }
        }
    }

    private ImportedData analyzeImportFile(File importFile, ImportBasicDataSaveVO saveVO) throws Exception {
        ImportedData importedData = new ImportedData();
        importedData.setIncremental(saveVO.getAllData() == null || !saveVO.getAllData());
        importedData.setSyncTenant(saveVO.getSyncTenant() == null || saveVO.getSyncTenant());

        // 解压文件
        File zipDir = null;
        try {
            zipDir = ZipUtil.unzip(importFile);
        } catch (UtilException e) {
            log.error("解压上传文件异常：", e);
            throw new IllegalArgumentException("解压上传文件异常：" + e.getMessage());
        }

        // 资源清单
        var manifest = this.loadManifest(zipDir);
        importedData.setManifest(manifest);

        // 导入的excel文件
        Workbook exportFile = this.loadImportFile(zipDir, manifest);
        importedData.setWorkbook(exportFile);

        // 其它文件
        importedData.setOtherFiles(new HashMap<>(128));
        Set<String> excludeFiles = Set.of(FILENAME_MANIFEST, manifest.getProperty(PROP_EXPORT_FILE));
        this.loadOtherFiles(zipDir, "", excludeFiles, importedData.getOtherFiles());

        // 版本
        importedData.setVersion(manifest.getProperty(PROP_VERSION));

        // 应用
        var appCodeStr = manifest.getProperty(PROP_APP_CODES);
        Set<String> appCodes = StringUtils.hasText(appCodeStr) ? Arrays.stream(appCodeStr.split(",")).collect(Collectors.toSet()) : Collections.emptySet();
        importedData.setAppCodes(appCodes);

        return importedData;
    }

    private void loadOtherFiles(File dir, String keyPrefix, Set<String> excludeFileNames, Map<String, File> fileMap) {
        var childFiles = dir.listFiles((dir1, name) -> excludeFileNames == null || !excludeFileNames.contains(name));
        if (ArrayUtil.isEmpty(childFiles)) {
            return;
        }
        for (File childFile : childFiles) {
            if (childFile.isDirectory()) {
                this.loadOtherFiles(childFile, keyPrefix + childFile.getName() + File.separator, null, fileMap);
                continue;
            }

            fileMap.put(keyPrefix + childFile.getName(), childFile);
        }
    }

    private Workbook loadImportFile(File zipDir, Properties manifest) {
        var exportFileName = manifest.getProperty(PROP_EXPORT_FILE);
        if (CharSequenceUtil.isBlank(exportFileName)) {
            throw new IllegalArgumentException("文件异常，缺少资源清单");
        }
        File exportFile = new File(zipDir, exportFileName);
        if (!exportFile.exists()) {
            throw new IllegalArgumentException("文件异常，资源清单不存在");
        }

        try {
            return new XSSFWorkbook(exportFile);
        } catch (Exception e) {
            log.error("加载excel文件异常：", e);
            throw new IllegalArgumentException("加载导入文件异常");
        }
    }

    private Properties loadManifest(File zipDir) {
        Properties properties = new Properties();
        try (FileReader fileReader = new FileReader(new File(zipDir, FILENAME_MANIFEST))) {
            properties.load(fileReader);
        } catch (IOException e) {
            log.error("加载资源清单失败：", e);
            throw new IllegalArgumentException("加载资源清单失败");
        }
        return properties;
    }

    private SysBasicRecordDO saveImportRecord(ImportedData importData, File importFile) {
        var uploadResult = fileService.upload(importFile);
        if (!uploadResult.isSuccess()) {
            throw new BusinessException("备份文件异常，" + uploadResult.getMsg());
        }

        var fileCode = uploadResult.getData().getFileCode();
        SysBasicRecordDO recordDO = new SysBasicRecordDO();
        recordDO.setRecordType(BasicRecordTypeEnum.IMPORT);
        recordDO.setAllData(!importData.isIncremental());
        recordDO.setAllVersion(Boolean.parseBoolean(importData.getManifest().getProperty(PROP_ALL_VER, "false")));
        recordDO.setDataVersion(importData.getManifest().getProperty(PROP_VERSION));
        if (!recordDO.getAllVersion()) {
            Assert.notBlank(recordDO.getDataVersion(), "导入失败，未知导入数据的版本");
        }
        recordDO.setAllType(Boolean.parseBoolean(importData.getManifest().getProperty(PROP_ALL_TYPE, "false")));
        recordDO.setDataType(importData.getManifest().getProperty(PROP_TYPE));
        if (!recordDO.getAllType()) {
            Assert.notBlank(recordDO.getDataType(), "导入失败，未知导入数据的类型");
        }
        recordDO.setTenantCode(importData.getManifest().getProperty(PROP_TENANT_CODE));
        recordDO.setAppCodes(importData.getManifest().getProperty(PROP_APP_CODES, ""));
        recordDO.setSyncTenant(importData.getSyncTenant());
        recordDO.setDataFileCode(fileCode);
        recordDO.setAttribute(JSONUtil.toJsonString(importData.getManifest()));
        recordDO.setOperateTime(LocalDateTime.now());
        recordDO.setSuccess(false);
        recordDO.setFinished(false);
        recordDO.setState(BasicRecordStateEnum.PREPARING.name());
        recordDO.setStd(Boolean.parseBoolean(importData.getManifest().getProperty(PROP_STD, "false")));

        recordRepoProc.save(recordDO);
        return recordDO;
    }

    private void importFromExcel(ImportedData importData, File importFile) throws Exception {
        // 先持久化
        var recordDO = this.saveImportRecord(importData, importFile);
        // 加入线程队列
        List<CompletableFuture<Void>> threads = new ArrayList<>();
        List<Long> tenantIdList = recordDO.getSyncTenant() ? this.allTenantId() : Collections.emptyList();
        var dataTypeList = recordDO.getAllType() ? null : Arrays.asList(recordDO.getDataType().split(","));
        var providerGroups = BasicDataInitProviderFactory.getImportProviderList(dataTypeList);
        Assert.notEmpty(providerGroups, "未发现有效的数据类型");
        List<String> expList = new ArrayList<>(providerGroups.size());
        for (var providerGroup : providerGroups) {
            var thread = createThreadToImportSheet(importData, providerGroup, tenantIdList, expList);
            threads.add(thread);
        }

        CompletableFuture.allOf(threads.toArray(CompletableFuture[]::new))
                .whenComplete((v, e) -> {
                    if (e != null) {
                        log.error("导入基础数据异常：", e);
                        expList.add(e.getMessage());
                    }
                    if (!expList.isEmpty()) {
                        recordRepoProc.updateFailResult(recordDO.getId(), String.join(",", expList));
                        return;
                    }
                    log.info("导入基础数据成功：{}", recordDO.getId());
                    recordRepoProc.updateImportSuccessfully(recordDO.getId());
                })
        ;
        recordRepoProc.updateState(recordDO.getId(), BasicRecordStateEnum.PREPARING);
    }

    private CompletableFuture<Void> createThreadToImportSheet(ImportedData importData, List<AbstractBasicDataInitProvider> dataInitProviders,
                                                              List<Long> tenantIds, List<String> expList) {
        return CompletableFuture.runAsync(() -> {
                    for (AbstractBasicDataInitProvider dataInitProvider : dataInitProviders) {
                        createImportRunner(importData, dataInitProvider, Collections.emptyList(), tenantIds);
                    }
                }
        ).whenComplete((r, e) -> {
            if (e != null) {
                log.error("导入异常：", e);
                expList.add(e.getMessage());
            }
        });
    }

    private List<Map<String, Object>> createImportRunner(ImportedData importData, AbstractBasicDataInitProvider dataInitProvider,
                                                         List<Map<String, Object>> preDataList, List<Long> tenantIds) {
        log.info("开始处理导入数据：{}", dataInitProvider.typeName());

        var workbook = importData.getWorkbook();
        var sheetName = this.sheetName(dataInitProvider);
        var sheet = workbook.getSheet(sheetName);
        if (sheet == null) {
            log.info("不存在sheet：{}", sheetName);
            return Collections.emptyList();
        }
        // 读取属性
        var fieldMap = this.readFields(sheet);
        if (fieldMap.isEmpty()) {
            log.info("未读取到属性：{}", sheetName);
            return Collections.emptyList();
        }

        // 从excel里读取数据
        var dataList = this.readDataFromSheet(sheet, fieldMap);
        if (dataList.isEmpty()) {
            log.info("数据为空：{}", sheetName);
            return Collections.emptyList();
        }
        var fields = dataInitProvider.fields();
        String sqlInsert = this.generateSqlInsert(fieldMap, fields, dataInitProvider.tableName());
        String sqlUpdate = this.generateSqlUpdate(fieldMap, fields, dataInitProvider.tableName());
        if (CharSequenceUtil.hasBlank(sqlInsert, sqlUpdate)) {
            log.info("未生成有效SQL：{}，{}，{}", dataInitProvider.typeName(), sqlInsert, sqlUpdate);
            return dataList;
        }

        if (!dataInitProvider.isTenantData()) {
            return this.wrapImportForTenant(importData, dataInitProvider, TenantConstant.DEFAULT_TENANT_ID, tenantIds,
                    dataList, preDataList, sqlInsert, sqlUpdate);
        }

        // 租户的数据
        for (Long tenantId : tenantIds) {
            this.wrapImportForTenant(importData, dataInitProvider, tenantId, tenantIds,
                    dataList, preDataList, sqlInsert, sqlUpdate);
        }
        return dataList;
    }

    private List<Map<String, Object>> wrapImportForTenant(ImportedData importData, AbstractBasicDataInitProvider dataInitProvider,
                                                          Long tenantId, List<Long> allTenantIds, List<Map<String, Object>> importDataList,
                                                          List<Map<String, Object>> preDataList, String sqlInsert, String sqlUpdate) {
        // 当前所有数据
        var currentData = this.queryCurrentData(importData.isIncremental(), dataInitProvider, tenantId, importData.getAppCodes());

        // 开始更新数据库
        TransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        TransactionStatus transactionStatus = transactionManager.getTransaction(transactionDefinition);
        Throwable throwable = null;
        try {
            // 导入前准备
            dataInitProvider.prepareForImport(importData, tenantId);

            this.executeImportData(importData, dataInitProvider, currentData, importDataList, preDataList, tenantId,
                    sqlInsert, sqlUpdate);
            transactionManager.commit(transactionStatus);

            log.info("处理导入数据{}成功", dataInitProvider.typeName());
        } catch (Throwable e) {
            log.error("更新数据异常：", e);
            throwable = e;
            transactionManager.rollback(transactionStatus);
            return importDataList;
        } finally {
            dataInitProvider.afterExecute(false, throwable);
        }

        // 依赖项的处理
        var children = dataInitProvider.thenImport();
        if (CollUtil.isNotEmpty(children)) {
            for (AbstractBasicDataInitProvider child : children) {
                createImportRunner(importData, child, importDataList, allTenantIds);
            }
        }
        return importDataList;
    }

    private void executeImportData(ImportedData importData, AbstractBasicDataInitProvider dataInitProvider,
                                   List<Map<String, Object>> currentDataList, List<Map<String, Object>> importDataList,
                                   List<Map<String, Object>> preDataList, Long tenantId, String sqlInsert, String sqlUpdate) {
        if (!importData.isIncremental()) {
            // 全量时，先删除
            Map<String, Object> sqlParams = new HashMap<>(4);
            String sqlDel = "delete from " + dataInitProvider.tableName() + " where tenant_id = :tenant_id";
            sqlParams.put("tenant_id", tenantId);
//            if (StringUtils.hasText(importData.getVersion())) {
//                sqlDel += " and std_version = :std_version";
//                sqlParams.put("std_version", importData.getVersion());
//            }
            if (CollUtil.isNotEmpty(importData.getAppCodes()) && StringUtils.hasText(dataInitProvider.fieldAppCode())) {
                sqlDel += " and " + dataInitProvider.fieldAppCode() + " in (:" + dataInitProvider.fieldAppCode() + ")";
                sqlParams.put(dataInitProvider.fieldAppCode(), importData.getAppCodes());
            }
            jdbcTemplate.update(sqlDel, sqlParams);
        }

        List<Map<String, Object>> sqlValuesAdd = new ArrayList<>(50);
        List<Map<String, Object>> sqlValuesUpdate = new ArrayList<>(50);

        AbstractBasicDataInitProvider.UpdateType updateType = null;
        LocalDateTime nowTime = LocalDateTime.now();
        for (Map<String, Object> data : importDataList) {
            data.put("tenant_id", tenantId);
            data.put("modify_time", nowTime);

            updateType = dataInitProvider.convertForImport(data, currentDataList, preDataList);
            if (updateType == null || updateType == AbstractBasicDataInitProvider.UpdateType.IGNORE) {
                // 忽略
                continue;
            }
            if (updateType == AbstractBasicDataInitProvider.UpdateType.ADD) {
                data.put("id", idCreator.create());
                data.put("create_time", nowTime);
                data.put("std_version", importData.getVersion());
                sqlValuesAdd.add(normalizeSqlParam(data));
                if (sqlValuesAdd.size() == 20) {
                    jdbcTemplate.batchUpdate(sqlInsert, sqlValuesAdd.toArray(Map[]::new));
                    sqlValuesAdd.clear();
                }
                continue;
            }

            sqlValuesUpdate.add(normalizeSqlParam(data));
            if (data.get("id") == null) {
                log.info("更新失败数据：{}", JSONObject.toJSONString(data));
                throw new IllegalStateException(dataInitProvider.typeName() + "更新数据时ID为空");
            }
            if (sqlValuesUpdate.size() == 20) {
                jdbcTemplate.batchUpdate(sqlUpdate, sqlValuesUpdate.toArray(Map[]::new));
                sqlValuesUpdate.clear();
            }
        }

        if (!sqlValuesAdd.isEmpty()) {
            jdbcTemplate.batchUpdate(sqlInsert, sqlValuesAdd.toArray(Map[]::new));
            sqlValuesAdd.clear();
        }
        if (!sqlValuesUpdate.isEmpty()) {
            jdbcTemplate.batchUpdate(sqlUpdate, sqlValuesUpdate.toArray(Map[]::new));
            sqlValuesUpdate.clear();
        }
    }

    private Map<String, Object> normalizeSqlParam(Map<String, Object> param) {
        Map<String, Object> newParam = new HashMap<>(param);
        for (Map.Entry<String, Object> entry : param.entrySet()) {
            if (entry.getValue() instanceof Boolean) {
                newParam.put(entry.getKey(), ((Boolean) entry.getValue()) ? 1 : 0);
                continue;
            }
            if (entry.getValue() instanceof String && CharSequenceUtil.isBlank(entry.getValue().toString())) {
                newParam.put(entry.getKey(), null);
                continue;
            }

            newParam.put(entry.getKey(), entry.getValue());
        }
        return newParam;
    }

    private List<Map<String, Object>> queryCurrentData(boolean incremental, AbstractBasicDataInitProvider dataInitProvider,
                                                       Long tenantId, Set<String> appCodes) {
        if (!incremental) {
            // 不是增量即全量，直接删除原所有数据，无需查询当前数据
            return Collections.emptyList();
        }
        return this.queryData(dataInitProvider, tenantId, null, appCodes);
    }

    private String generateSqlInsert(Map<String, Integer> fieldMap, List<String> fields, String tableName) {
        var tableColumns = fieldMap.entrySet().stream()
                .filter(t -> fields.contains(t.getKey()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
        if (tableColumns.isEmpty()) {
            return null;
        }
        if (!tableColumns.contains("id")) {
            tableColumns.add(0, "id");
        }
        if (!tableColumns.contains("tenant_id")) {
            tableColumns.add(1, "tenant_id");
        }
        if (!tableColumns.contains("create_time")) {
            tableColumns.add("create_time");
        }
        if (!tableColumns.contains("std_version")) {
            tableColumns.add("std_version");
        }

        String sqlValue = tableColumns.stream().map(t -> ":" + t).collect(Collectors.joining(", "));

        return "insert into " + tableName + "(" + String.join(", ", tableColumns) + ") values (" + sqlValue + ")";
    }

    private String generateSqlUpdate(Map<String, Integer> fieldMap, List<String> fields, String tableName) {
        var tableColumns = fieldMap.entrySet().stream()
                .filter(t -> fields.contains(t.getKey()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
        if (tableColumns.isEmpty()) {
            return null;
        }
        if (tableColumns.contains("create_time")) {
            tableColumns.remove("create_time");
        }

        String sqlValue = tableColumns.stream()
                .map(t -> t + " = :" + t).collect(Collectors.joining(", "));

        return "update " + tableName + " set " + sqlValue + " where id = :id";
    }

    private List<Map<String, Object>> readDataFromSheet(Sheet sheet, Map<String, Integer> fieldMap) {
        var lastRowNum = sheet.getLastRowNum();
        if (lastRowNum < 2) {
            return Collections.emptyList();
        }

        List<Map<String, Object>> dataList = new ArrayList<>(lastRowNum + 1);
        var fieldEntry = fieldMap.entrySet();

        Row row = null;
        Cell cell = null;
        Object cellValue = null;
        Map<String, Object> rowData = null;
        boolean hasData = false;
        for (int rowNum = 2; rowNum <= lastRowNum; rowNum++) {
            row = sheet.getRow(rowNum);
            rowData = new HashMap<>(fieldMap.size());
            hasData = false;

            // 读取行数据
            for (Map.Entry<String, Integer> entry : fieldEntry) {
                cell = row.getCell(entry.getValue());
                if (cell == null) {
                    continue;
                }
                cellValue = getCellValue(cell);
                if (cellValue == null) {
                    continue;
                }

                rowData.put(entry.getKey(), cellValue);
                hasData = true;
            }
            if (hasData) {
                dataList.add(rowData);
            }
        }
        return dataList;
    }

    private Object getCellValue(Cell cell) {
        var cellType = cell.getCellTypeEnum();
        return CellUtil.getCellValue(cell, cellType, true);
    }

    private void startExport(SysBasicRecordDO recordDO) {
        var tenantCode = recordDO.getTenantCode();
        // 准备工作目录
        var workDir = FileUtil.createTempDir(CharSequenceUtil.format("system_{}", DatetimeUtil.currentTimeLong()));
        // 导出并压缩
        File zipFile = null;
        try {
            recordRepoProc.updateState(recordDO.getId(), BasicRecordStateEnum.PROCESSING);
            Long tenantId = this.queryTenantId(tenantCode);
            log.info("导出所在工作目录:{}", workDir.getAbsolutePath());

            // 开始导出
            Properties props = new Properties();
            props.setProperty(PROP_EXPORT_TIME, DatetimeUtil.currentTimeStr());
            props.setProperty(PROP_RECORD_ID, recordDO.getId().toString());
            props.setProperty(PROP_TENANT_CODE, ObjectUtil.defaultIfNull(recordDO.getTenantCode(), ""));
            props.setProperty(PROP_ALL_VER, recordDO.getAllVersion().toString());
            props.setProperty(PROP_VERSION, ObjectUtil.defaultIfNull(recordDO.getDataVersion(), ""));
            props.setProperty(PROP_STD, recordDO.getStd().toString());
            props.setProperty(PROP_ALL_TYPE, recordDO.getAllType().toString());
            props.setProperty(PROP_TYPE, ObjectUtil.defaultIfNull(recordDO.getDataType(), ""));
            props.setProperty(PROP_APP_CODES, ObjectUtil.defaultIfNull(recordDO.getAppCodes(), ""));
            props.setProperty(PROP_DESCRIPTION, ObjectUtil.defaultIfNull(recordDO.getDescription(), ""));
            this.exportToExcel(workDir, tenantId, props, recordDO);
            zipFile = ZipUtil.zip(workDir, StandardCharsets.UTF_8);

            // 上传导出的文件
            var uploadResult = fileService.upload(zipFile);
            Assert.isTrue(uploadResult.isSuccess(), "上传导出文件失败，" + uploadResult.getMsg());
            recordRepoProc.updateExportSuccessfully(recordDO.getId(), uploadResult.getData().getFileCode(), props.toString());

        } catch (Throwable e) {
            recordRepoProc.updateFailResult(recordDO.getId(), ExceptionsUtil.stacktraceToString(e));
            log.error("导出系统的初始化数据异常：", e);
        } finally {
            FileUtil.del(workDir);
            FileUtil.del(zipFile);
        }
    }

    private void exportToExcel(File workDir, Long tenantId, Properties props, SysBasicRecordDO recordDO) throws Exception {
        // 创建导出文件
        File excelFile = new File(workDir, workDir.getName() + "." + SUFFIX_XLSX);
        var createResult = excelFile.createNewFile();
        Assert.isTrue(createResult, "创建导出文件失败");
        props.setProperty(PROP_EXPORT_FILE, excelFile.getName());

        var exportProviders = BasicDataInitProviderFactory.getExportProviderList(recordDO.getAllType() ? null : Arrays.stream(recordDO.getDataType().split(",")).collect(Collectors.toList()));
        var version = recordDO.getAllVersion() ? null : recordDO.getDataVersion();
        Set<String> appCodes = StringUtils.hasText(recordDO.getAppCodes()) ? Arrays.stream(recordDO.getAppCodes().split(",")).collect(Collectors.toSet()) : Collections.emptySet();

        try (var workbook = new SXSSFWorkbook(); var output = new FileOutputStream(excelFile)) {
            // 加入线程队列
            List<CompletableFuture<Void>> threads = new ArrayList<>();
            for (var providers : exportProviders) {
                var thread = createThreadToExportSheet(workbook, providers, workDir, tenantId, version, appCodes);
                threads.add(thread);
            }
            CompletableFuture.allOf(threads.toArray(CompletableFuture[]::new))
                    .exceptionally(e -> {
                        log.error("导出基础数据异常：", e);
                        return null;
                    })
                    .get();

            workbook.write(output);
        }

        // 资源文件清单
        File manifest = new File(workDir, FILENAME_MANIFEST);
        try (var fileWriter = new FileWriter(manifest)) {
            props.store(fileWriter, "resource list, exported by cloudt");
        }
    }

    private CompletableFuture<Void> createThreadToExportSheet(Workbook workbook, List<AbstractBasicDataInitProvider> dataInitProviders,
                                                              File workDir, Long tenantIdReal, String version, Set<String> appCodes) {
        Map<String, Sheet> sheetMap = new HashMap<>(dataInitProviders.size());
        for (AbstractBasicDataInitProvider dataInitProvider : dataInitProviders) {
            var sheetName = this.sheetName(dataInitProvider);
            var sheet = workbook.createSheet(sheetName);
            sheetMap.put(dataInitProvider.typeName(), sheet);
        }

        return CompletableFuture.runAsync(() -> {
            for (AbstractBasicDataInitProvider dataInitProvider : dataInitProviders) {
                var sheet = sheetMap.get(dataInitProvider.typeName());
                Exception exp = null;
                try {
                    this.executeExportSheet(sheet, dataInitProvider, tenantIdReal, workDir, version, appCodes);
                    log.info("导出{}结束", dataInitProvider.typeName());
                } catch (Exception e) {
                    exp = e;
                    log.error("导出{}异常：", dataInitProvider.typeName(), e);
                } finally {
                    dataInitProvider.afterExecute(true, exp);
                }
            }
        }).whenComplete((r, e) -> {
            if (e != null) {
                log.error("导出异常：", e);
            }
        });
    }

    private void executeExportSheet(Sheet sheet, AbstractBasicDataInitProvider dataInitProvider, Long tenantIdReal, File workDir,
                                    String version, Set<String> appCodes) {
        var dataTypeName = sheet.getSheetName();
        var tenantId = dataInitProvider.isTenantData() ? tenantIdReal : TenantConstant.DEFAULT_TENANT_ID;

        sheet.setDefaultColumnWidth(22);

        log.info("开始导出{}...", dataTypeName);
        // 表格头部
        var fieldTitles = dataInitProvider.fieldTitles();
        exportRowTitle(sheet, fieldTitles);

        dataInitProvider.prepareExport(workDir);

        // 查询数据
        var dataList = queryData(dataInitProvider, tenantId, version, appCodes);
        if (!dataList.isEmpty()) {
            int rowIndex = 2;
            int cellIndex = 0;
            Row row = null;
            Cell cell = null;
            var cellStyle = createCommonStyle(sheet);

            var fields = fieldTitles.keySet();
            for (var rowData : dataList) {
                // 过滤和转换数据
                var realData = dataInitProvider.convertForExport(rowData, null);
                if (CollUtil.isEmpty(realData)) {
                    continue;
                }
                row = sheet.createRow(rowIndex);

                cellIndex = 0;
                for (String field : fields) {
                    cell = row.createCell(cellIndex);
                    cell.setCellValue(convertFieldValue(realData.get(field)));
                    cell.setCellStyle(cellStyle);

                    cellIndex++;
                }

                rowIndex++;
            }
        }
    }

    private void exportRowTitle(Sheet sheet, Map<String, String> fieldTitles) {
        var cellStyle = createTitleStyle(sheet);

        // 创建标题
        var row = sheet.createRow(0);
        int cellIndex = 0;
        Cell cell = null;
        for (String value : fieldTitles.values()) {
            cell = row.createCell(cellIndex);
            cell.setCellValue(value);

            // 样式
            cell.setCellStyle(cellStyle);

            cellIndex++;
        }
        // 创建属性
        var fields = fieldTitles.keySet();
        row = sheet.createRow(1);
        cellIndex = 0;
        for (String value : fields) {
            cell = row.createCell(cellIndex);
            cell.setCellValue(value);
            cell.setCellStyle(cellStyle);

            cellIndex++;
        }
    }

    private Map<String, Integer> readFields(Sheet sheet) {
        var row = sheet.getRow(1);
        if (row == null) {
            return Collections.emptyMap();
        }

        var lastCellNum = row.getLastCellNum();
        Map<String, Integer> fieldMap = new HashMap<>(lastCellNum + 1);
        Cell cell = null;
        String cellValue = null;
        for (int colNum = 0; colNum <= lastCellNum; colNum++) {
            cell = row.getCell(colNum);
            if (cell == null) {
                continue;
            }
            cellValue = cell.getStringCellValue();
            if (cellValue != null) {
                cellValue = cellValue.trim();
            }
            if (CharSequenceUtil.isBlank(cellValue)) {
                continue;
            }
            fieldMap.put(cellValue, colNum);
        }

        return fieldMap;
    }

    private CellStyle createCommonStyle(Sheet sheet) {
        CellStyle cellStyle = sheet.getWorkbook().createCellStyle();
        var font = sheet.getWorkbook().createFont();
        cellStyle.setFont(font);
        cellStyle.setAlignment(HorizontalAlignment.LEFT);
        cellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        cellStyle.setBorderLeft(BorderStyle.THIN);
        cellStyle.setBorderTop(BorderStyle.THIN);
        cellStyle.setBorderRight(BorderStyle.THIN);
        cellStyle.setBorderBottom(BorderStyle.THIN);
        cellStyle.setWrapText(true);

        return cellStyle;
    }

    private CellStyle createTitleStyle(Sheet sheet) {
        CellStyle cellStyle = sheet.getWorkbook().createCellStyle();
        var font = sheet.getWorkbook().createFont();
        cellStyle.setFont(font);
        cellStyle.setAlignment(HorizontalAlignment.CENTER);
        cellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        cellStyle.setBorderLeft(BorderStyle.THIN);
        cellStyle.setBorderTop(BorderStyle.THIN);
        cellStyle.setBorderRight(BorderStyle.THIN);
        cellStyle.setBorderBottom(BorderStyle.THIN);
        cellStyle.setWrapText(true);

        font.setBold(true);
        return cellStyle;
    }

    private List<Map<String, Object>> queryData(AbstractBasicDataInitProvider dataInitProvider, Long tenantId, String version,
                                                Set<String> appCodes) {
        var fields = new ArrayList<>(dataInitProvider.fields());
        if (!fields.contains("id")) {
            fields.add("id");
        }
        String sql = "select " + String.join(", ", fields) + " from " + dataInitProvider.tableName();

        // where 条件
        Map<String, Object> sqlParams = new HashMap<>(8);
        List<String> conditionList = new ArrayList<>(8);

        if (tenantId != null) {
            conditionList.add(dataInitProvider.tenantField() + " = :tenant_id");
            sqlParams.put("tenant_id", tenantId);
        }
//        if (StringUtils.hasText(version)) {
//            conditionList.add("std_version = :std_version");
//            sqlParams.put("std_version", version);
//        }
        if (CollUtil.isNotEmpty(appCodes) && StringUtils.hasText(dataInitProvider.fieldAppCode())) {
            conditionList.add(dataInitProvider.fieldAppCode() + " in (:" + dataInitProvider.fieldAppCode() + ")");
            sqlParams.put(dataInitProvider.fieldAppCode(), appCodes);
        }

        conditionList.add("delete_flag = :delete_flag");
        sqlParams.put("delete_flag", 0);

        sql += " where ";
        sql += String.join(" and ", conditionList);

        // 排序字段
        var orderFieldList = dataInitProvider.orderItems();
        if (CollUtil.isNotEmpty(orderFieldList)) {
            var order = orderFieldList.stream()
                    .filter(t -> CharSequenceUtil.isNotBlank(t.getColumn()))
                    .map(t -> t.getColumn() + " " + (t.isAsc() ? "asc" : "desc"))
                    .collect(Collectors.joining(", "));
            if (CharSequenceUtil.isNotBlank(order)) {
                sql += (" order by " + order);
            }
        }

        var dataList = jdbcTemplate.queryForList(sql, sqlParams);
        return ObjectUtil.defaultIfNull(dataList, Collections.emptyList());
    }

    private String convertFieldValue(Object value) {
        if (value == null) {
            return null;
        }
        if (value instanceof LocalDateTime) {
            return DatetimeUtil.FORMATTER_DATETIME.format((LocalDateTime) value);
        }
        if (value instanceof LocalDate) {
            return DatetimeUtil.FORMATTER_DATE.format((LocalDate) value);
        } else if (value instanceof Date) {
            return DatetimeUtil.FORMAT_DATETIME.format((Date) value);
        }

        return value.toString();
    }

    public static class ImportedData {
        private boolean incremental;
        private boolean syncTenant;
        private String version;
        private Set<String> appCodes;
        private Properties manifest;
        private Workbook workbook;
        private Map<String, File> otherFiles;

        public boolean isIncremental() {
            return incremental;
        }

        public void setIncremental(boolean incremental) {
            this.incremental = incremental;
        }

        public boolean getSyncTenant() {
            return syncTenant;
        }

        public void setSyncTenant(boolean syncTenant) {
            this.syncTenant = syncTenant;
        }

        public String getVersion() {
            return version;
        }

        public void setVersion(String version) {
            this.version = version;
        }

        public boolean isSyncTenant() {
            return syncTenant;
        }

        public Set<String> getAppCodes() {
            return appCodes;
        }

        public void setAppCodes(Set<String> appCodes) {
            this.appCodes = appCodes;
        }

        public Properties getManifest() {
            return manifest;
        }

        public void setManifest(Properties manifest) {
            this.manifest = manifest;
        }

        public Workbook getWorkbook() {
            return workbook;
        }

        public void setWorkbook(Workbook workbook) {
            this.workbook = workbook;
        }

        public Map<String, File> getOtherFiles() {
            return otherFiles;
        }

        public void setOtherFiles(Map<String, File> otherFiles) {
            this.otherFiles = otherFiles;
        }
    }
}
