package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.CloudtContextProperties;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.task.retry.AbstractRetryService;
import com.elitescloud.boot.task.retry.RetryTask;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.LockUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.constant.SysConstant;
import com.elitescloud.cloudt.system.dto.ThirdApiRetryParamDTO;
import com.elitescloud.cloudt.system.model.entity.SysThirdApiLogDO;
import com.elitescloud.cloudt.system.service.ThirdApiRetrySupportService;
import com.elitescloud.cloudt.system.service.repo.ThirdApiBusinessRepoProc;
import com.elitescloud.cloudt.system.service.repo.ThirdApiLogRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.validation.constraints.NotBlank;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * 三方接口重试服务.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/11
 */
@Slf4j
@Component
@ConditionalOnClass(AbstractRetryService.class)
class ThirdApiRetry extends AbstractRetryService<ThirdApiRetry.RetryParam> {

    private final SystemProperties.ThirdApiLog properties;
    private final RedissonClient redissonClient;
    private final RestTemplate restTemplate = RestTemplateFactory.instance();
    @Autowired
    private ThirdApiLogRepoProc repoProc;
    @Autowired
    private ThirdApiBusinessRepoProc businessRepoProc;
    @Autowired
    @Lazy
    private ThirdApiRetrySupportService service;
    @Autowired
    private CloudtContextProperties cloudtContextProperties;

    public ThirdApiRetry(ThirdApiRetryTaskProvider retryTaskProvider,
                         SystemProperties systemProperties,
                         RedissonClient redissonClient,
                         TenantDataIsolateProvider tenantDataIsolateProvider) {
        super(retryTaskProvider, tenantDataIsolateProvider);
        this.properties = systemProperties.getThirdApiLog();
        this.redissonClient = redissonClient;
    }

    @Override
    protected boolean supportRetry() {
        return Boolean.TRUE.equals(properties.getEnabledRetry());
    }

    @Override
    protected int retryTimes() {
        return ObjectUtil.defaultIfNull(properties.getRetryTimes(), 0);
    }

    @Override
    protected List<Duration> retryIntervals() {
        return properties.getRetryIntervals();
    }

    @Override
    protected void executeTask(RetryParam retryParam) {
        log.info("开始重试请求：{}", retryParam.getTaskId());
        var logId = StringUtils.hasText(retryParam.getTaskId()) ? Long.parseLong(retryParam.getTaskId()) : null;
        Assert.notNull(logId, "记录ID为空");
        var logDO = repoProc.get(logId);
        this.execute(logDO, false);
    }

    /**
     * 手动重试
     *
     * @param logDO 日志记录
     */
    public void executeByManual(SysThirdApiLogDO logDO) {
        this.execute(logDO, true);
    }

    @Override
    protected Duration scheduleDelay() {
        return Duration.ofMinutes(30);
    }

    private void execute(SysThirdApiLogDO logDO, boolean manual) {
        Assert.notNull(logDO, "日志记录已不存在");

        this.tryExecuteHttp(logDO, manual);
    }

    private void tryExecuteHttp(SysThirdApiLogDO logDO, boolean manual) {
        if (Boolean.FALSE.equals(logDO.getRestful())) {
            log.info("不需要重试：{}，{}", logDO.getId(), logDO.getRestful());
            return;
        }

        // 重试参数
        RetryConfig config = this.buildRetryConfig(logDO);
        this.normalizeRetryConfig(config);

        Supplier<Void> supplier = () -> {
            if (this.isSuccessOfBusiness(logDO)) {
                // 业务已成功
                service.updateRetryRespResult(logDO.getId(), manual, false, "其它请求已成功", null);
                return null;
            }

            // 获取url
            String url = this.buildUrl(config);
            log.info("重试请求：{} {}", config.getReqMethod(), url);
            // 获取请求头
            HttpHeaders headers = this.obtainHeaders(config, logDO);
            log.info("请求头：{}", headers);

            ResponseEntity<String> resp = null;
            try {
                resp = restTemplate.exchange(url, config.getReqMethod(), new HttpEntity<>(config.getReqBody(), headers), String.class);
            } catch (RestClientException e) {
                log.error("重试请求异常：", e);
                var reason = e.getMessage();
                service.updateRetryRespResult(logDO.getId(), manual, false, ObjectUtil.defaultIfNull(reason, "请求失败"), null);
                return null;
            }

            // 更新响应结果
            if (resp.getStatusCode().is2xxSuccessful()) {
                // 响应成功
                var apiResult = this.attemptAnalyzeForResult(resp.getBody());
                if (apiResult != null && apiResult.getTime() != null && apiResult.getCode() != 0) {
                    service.updateRetryRespResult(logDO.getId(), manual, apiResult.isSuccess(),
                            CharSequenceUtil.blankToDefault(apiResult.getMsg(), apiResult.getErrorMsg()), resp.getBody());
                    return null;
                }
                // 默认成功
                service.updateRetryRespResult(logDO.getId(), manual, true, null, resp.getBody());
                return null;
            }

            service.updateRetryRespResult(logDO.getId(), manual, false, resp.getStatusCode().toString(), resp.getBody());
            return null;
        };
        if (CharSequenceUtil.hasBlank(logDO.getBusinessType(), logDO.getBusinessKey())) {
            supplier.get();
        }

        LockUtil.executeByLock("cloudt-thirdApi-retry-" + logDO.getBusinessType() + "-" + logDO.getBusinessKey(), supplier, Duration.ofMinutes(2));
    }

    private ApiResult<String> attemptAnalyzeForResult(String respBody) {
        return JSONUtil.json2Obj(respBody, new TypeReference<>() {
        }, false, () -> "尝试使用ApiResult解析返回结果失败");
    }

    private void normalizeRetryConfig(RetryConfig config) {
        Assert.hasText(config.getServerAddr(), "服务端地址为空");
        Assert.hasText(config.getUri(), "请求接口为空");

        Assert.notNull(config.getReqMethod(), "请求方式为空");

        if (!config.getUri().startsWith("/")) {
            config.setUri("/" + config.getUri());
        }
    }

    private RetryConfig buildRetryConfig(SysThirdApiLogDO logDO) {
        String token = this.obtainToken(logDO);
        RetryConfig retryConfig = new RetryConfig();
        retryConfig.setToken(token);
        retryConfig.setTenantId(logDO.getSysTenantId());

        if (CharSequenceUtil.isBlank(logDO.getRetryParamJson())) {
            retryConfig.setServerAddr(CharSequenceUtil.blankToDefault(logDO.getServerAddr(), cloudtContextProperties.getServerAddr()));
            retryConfig.setUri(logDO.getUri());
            retryConfig.setReqMethod(logDO.getReqMethod());
            retryConfig.setReqQueryParams(this.convertMultiValueMap(logDO.getReqQueryParamsJson()));
            retryConfig.setReqBody(this.convertReqBody(logDO.getReqBody()));
            retryConfig.setReqHeaders(this.convertMultiValueMap(logDO.getReqHeadersJson()));

            return retryConfig;
        }

        ThirdApiRetryParamDTO retryParams = JSONUtil.json2Obj(logDO.getRetryParamJson(), ThirdApiRetryParamDTO.class, true, () -> "转换重试参数异常");
        retryConfig.setServerAddr(CharSequenceUtil.blankToDefault(retryParams.getServerAddr(), cloudtContextProperties.getServerAddr()));
        retryConfig.setUri(retryParams.getUri());
        retryConfig.setReqMethod(retryParams.getReqMethod());
        retryConfig.setReqQueryParams(this.convertMultiValueMap(retryParams.getReqQueryParams()));
        retryConfig.setReqBody(this.convertReqBody(retryParams.getReqBody()));
        retryConfig.setReqHeaders(this.convertMultiValueMap(retryParams.getReqHeaders()));

        return retryConfig;
    }

    private String obtainToken(SysThirdApiLogDO logDO) {
        if (!Boolean.TRUE.equals(logDO.getDetectedOperatorAuth())) {
            // 不需要获取操作人的token
            return null;
        }
        var token = SecurityContextUtil.currentToken();
        if (CharSequenceUtil.isBlank(token)) {
            log.info("未获取到当前操作人token：{}", logDO.getId());
            return null;
        }

        return token;
    }

    private boolean isSuccessOfBusiness(SysThirdApiLogDO logDO) {
        if (CharSequenceUtil.hasBlank(logDO.getBusinessType(), logDO.getBusinessKey())) {
            // 无需加锁
            return false;
        }

        var success = businessRepoProc.getSuccess(logDO.getBusinessType(), logDO.getBusinessKey());
        return Boolean.TRUE.equals(success);
    }

    private HttpHeaders obtainHeaders(RetryConfig config, SysThirdApiLogDO logDO) {
        var httpHeaders = new HttpHeaders(config.getReqHeaders());
        if (CharSequenceUtil.isNotBlank(config.getToken())) {
            httpHeaders.setBearerAuth(config.getToken());
        }
        if (config.getTenantId() != null) {
            httpHeaders.add(TenantConstant.HEADER_TENANT_ID, config.getTenantId().toString());
        }
        httpHeaders.add(SysConstant.HEADER_RETRY_ID, logDO.getId().toString());
        return httpHeaders;
    }

    private String buildUrl(RetryConfig config) {
        String uri = config.getServerAddr() + config.getUri();

        if (config.getReqQueryParams().isEmpty() || uri.contains("?")) {
            return uri;
        }

        var queryParams = this.convertReqQueryParams(config.getReqQueryParams());
        if (CharSequenceUtil.isNotBlank(queryParams)) {
            uri = uri + "?" + queryParams;
        }

        return uri;
    }

    private Object convertReqBody(String reqBody) {
        if (CharSequenceUtil.isBlank(reqBody)) {
            return null;
        }

        return JSONUtil.json2Obj(reqBody, true, () -> "解析请求头异常");
    }

    private String convertReqQueryParams(MultiValueMap<String, String> valueMap) {
        if (valueMap.isEmpty()) {
            return null;
        }

        return valueMap.entrySet().stream()
                .map(entry -> {
                    if (CollUtil.isEmpty(entry.getValue())) {
                        return entry.getKey() + "=";
                    }
                    return entry.getValue().stream().filter(Objects::nonNull)
                            .map(t -> entry.getKey() + "=" + t).collect(Collectors.joining("&"));
                }).collect(Collectors.joining("&"));
    }

    private MultiValueMap<String, String> convertMultiValueMap(Map<String, String[]> originalMap) {
        MultiValueMap<String, String> result = new LinkedMultiValueMap<>();
        if (MapUtil.isEmpty(originalMap)) {
            return result;
        }

        for (Map.Entry<String, String[]> entry : originalMap.entrySet()) {
            if (ArrayUtil.isNotEmpty(entry.getValue())) {
                for (String v : entry.getValue()) {
                    result.add(entry.getKey(), v);
                }
            }
        }

        return result;
    }

    private MultiValueMap<String, String> convertMultiValueMap(String json) {
        if (CharSequenceUtil.isBlank(json)) {
            return new LinkedMultiValueMap<>(0);
        }

        return JSONUtil.json2Obj(json, new TypeReference<LinkedMultiValueMap<String, String>>() {
        }, true, () -> "参数转换失败");
    }

    public static class RetryParam extends RetryTask {
        private static final long serialVersionUID = 3323072273214519095L;
    }

    @Data
    static class RetryConfig {
        /**
         * 自动获取操作人的token
         */
        private String token;

        /**
         * 业务所属租户
         */
        private Long tenantId;

        /**
         * 服务端地址
         * <p>
         * 为空则取当前服务的地址（不包含路由）
         */
        private String serverAddr;

        /**
         * 请求的接口地址
         * <p>
         * 包含路由
         */
        @NotBlank(message = "请求的接口为空")
        private String uri;

        /**
         * 请求方式
         */
        private HttpMethod reqMethod;

        /**
         * 请求的查询参数
         */
        private MultiValueMap<String, String> reqQueryParams;

        /**
         * 请求体
         */
        private Object reqBody;

        /**
         * 请求头
         */
        private MultiValueMap<String, String> reqHeaders;
    }
}
