package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ZipUtil;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.excel.common.param.ImportRecordDTO;
import com.elitescloud.boot.support.app.CloudtAppHolder;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.TmplImportRecordConvert;
import com.elitescloud.cloudt.system.dto.SysImportRateDTO;
import com.elitescloud.cloudt.system.dto.req.RecordResultSaveDTO;
import com.elitescloud.cloudt.system.model.vo.query.extend.TmplImportQueryParam;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TmplImportRecordRespVO;
import com.elitescloud.cloudt.system.service.TmplImportRecordService;
import com.elitescloud.cloudt.system.service.model.entity.SysTmplImportFileDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTmplImportRecordDO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.fasterxml.jackson.core.JsonProcessingException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/5/31
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Slf4j
public class TmplImportRecordServiceImpl extends BaseServiceImpl implements TmplImportRecordService {
    public static final long DEFAULT_TMPL_ID = -1L;

    @Autowired
    private SysTmplRepoProc tmplRepoProc;
    @Autowired
    private SysTmplImportRecordRepo tmplImportRecordRepo;
    @Autowired
    private SysTmplImportRecordRepoProc tmplImportRecordRepoProc;
    @Autowired
    private SysTmplImportFileRepo tmplImportFileRepo;
    @Autowired
    private SysTmplImportFileRepoProc tmplImportFileRepoProc;

    @Autowired
    private FileService<String> fileService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveImportRecord(String tmplCode, String fileCode, Map<String, Object> queryParam) {
        Long tmplId = tenantDataIsolateProvider.byDefaultDirectly(() -> tmplRepoProc.getIdByCode(tmplCode));
        if (tmplId == null) {
            log.info("未查询到模板，将使用默认：{}", tmplCode);
            tmplId = DEFAULT_TMPL_ID;
        }

        Long importRecordId = save(tmplId, tmplCode, fileCode, queryParam);
        return ApiResult.ok(importRecordId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateRecordTotal(Long id, Long numTotal) {
        tmplImportRecordRepoProc.updateNumTotal(id, numTotal);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateImportResult(RecordResultSaveDTO resultSaveDTO) {
        if (resultSaveDTO.getRecordId() == null) {
            return ApiResult.fail("记录ID为空");
        }
        long numTotal = ObjectUtil.defaultIfNull(tmplImportRecordRepoProc.getNumTotal(resultSaveDTO.getRecordId()), 0L);
        long numSuc = ObjectUtil.defaultIfNull(resultSaveDTO.getNumSuc(), 0L);
        // 导入数量与总数相同时成功
        boolean success = numSuc > 0 && numTotal <= numSuc;

        if (success) {
            // 如果多个文件则进行压缩
            String zipFileCode = zipFiles(resultSaveDTO.getRecordId());
            if (StringUtils.hasText(zipFileCode)) {
                resultSaveDTO.setFileCode(zipFileCode);
            }
        }

        if (!success && numTotal == 0 && !CharSequenceUtil.isBlank(resultSaveDTO.getFailMsg())) {
            resultSaveDTO.setFailMsg("数据为空");
        }
        tmplImportRecordRepoProc.updateImportResult(resultSaveDTO.getRecordId(), success,
                resultSaveDTO.getFailMsg(),
                resultSaveDTO.getCustomTipMsg(),
                numSuc,
                resultSaveDTO.getFileCode(),
                resultSaveDTO.getCostTimeInfo()
        );

        return ApiResult.ok(resultSaveDTO.getRecordId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveExportFile(Long id, String fileCode, Integer order) {
        var tmplId = tmplImportRecordRepoProc.getTmplId(id);
        if (tmplId == null) {
            return ApiResult.fail("未查询到导入记录");
        }

        SysTmplImportFileDO importFileDO = new SysTmplImportFileDO();
        importFileDO.setRecordId(id);
        importFileDO.setTmplId(tmplId);
        importFileDO.setFileCode(fileCode);
        importFileDO.setFileOrder(ObjectUtil.defaultIfNull(order, 0));

        if (StringUtils.hasText(fileCode)) {
            var file = fileService.getForDto(fileCode).getData();
            if (file != null) {
                importFileDO.setFileName(file.getOriginalName());
            }
        }
        tmplImportFileRepo.save(importFileDO);

        return ApiResult.ok(importFileDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveImportFailRecord(Long recordId, String failFileCode) {
        tmplImportRecordRepoProc.updateFailFileCode(recordId, failFileCode);
        return ApiResult.ok(recordId);
    }

    @Override
    public ApiResult<SysImportRateDTO> getImportRateDto(Long id) {
        var rateDto = tmplImportRecordRepo.findById(id).map(t -> {
            var dto = new SysImportRateDTO();
            dto.setFinish(t.getFinish());
            dto.setTmplCode(tenantDataIsolateProvider.byDefaultDirectly(() -> tmplRepoProc.getCode(t.getTmplId())));
            dto.setTotal(t.getNumTotal());
            dto.setCount(t.getNumSuc());
            dto.setNumSuccess(t.getNumSuc());
            dto.setFailFileCode(t.getFailFileCode());
            dto.setFailMsgCustom(t.getCustomTipMsg());

            return dto;
        });
        if (rateDto.isPresent()) {
            return ApiResult.ok(rateDto.get());
        }

        return ApiResult.fail("导入记录不存在");
    }

    @Override
    public ApiResult<String> getFileCode(Long id) {
        return ApiResult.ok(tmplImportRecordRepoProc.getFileCode(id));
    }

    @Override
    public ApiResult<List<Long>> queryUnFinished(String appCode) {
        return ApiResult.ok(tmplImportRecordRepoProc.getUnfinished(appCode));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateResultForSysError(String appCode, String failMsg) {
        var recordIds = tmplImportRecordRepoProc.getUnfinished(appCode);
        if (CollUtil.isEmpty(recordIds)) {
            // 没有需要更新的
            return ApiResult.ok(true);
        }

        failMsg = StringUtils.hasText(failMsg) ? failMsg : "服务停止";
        tmplImportRecordRepoProc.updateResultForError(recordIds, failMsg);

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<PagingVO<TmplImportRecordRespVO>> searchImport(Long tmplId, TmplImportQueryParam queryParam) {
        var currentUser = super.currentUser(true);
        if (!currentUser.isSystemAdmin() && !currentUser.isTenantAdmin()) {
            queryParam.setUserId(currentUser.getUserId());
        }
        PagingVO<SysTmplImportRecordDO> recordDoPage = tmplImportRecordRepoProc.pageMng(tmplId, queryParam);
        if (recordDoPage.isEmpty()) {
            return ApiResult.ok(PagingVO.empty());
        }

        // do 转 vo
        List<String> fileCodeList = new ArrayList<>(queryParam.getSize());
        List<Long> idList = new ArrayList<>(queryParam.getSize());
        List<TmplImportRecordRespVO> respVoList = new ArrayList<>(queryParam.getSize());
        TmplImportRecordConvert convert = TmplImportRecordConvert.INSTANCE;
        for (var doo : recordDoPage.getRecords()) {
            var vo = convert.do2Vo(doo);
            vo.setUserName(CharSequenceUtil.blankToDefault(doo.getCreator(), doo.getUserName()));
            vo.setFailMsgCustom(doo.getCustomTipMsg());
            respVoList.add(vo);
            if (StringUtils.hasText(doo.getFileCode())) {
                fileCodeList.add(doo.getFileCode());
            } else {
                idList.add(doo.getId());
            }
        }

        // 查询记录
        Map<Long, List<String>> fileCodeMap = idList.isEmpty() ? Collections.emptyMap() : tmplImportFileRepoProc.queryByRecordId(idList);
        fileCodeMap.forEach((k, v) -> fileCodeList.addAll(v));

        Map<String, FileObjRespVO<String>> fileInfoMap = null;
        if (!fileCodeList.isEmpty()) {
            var fileList = fileService.query(fileCodeList).getData();
            if (CollUtil.isNotEmpty(fileList)) {
                fileInfoMap = fileList.stream().collect(Collectors.toMap(FileObjRespVO::getFileCode, t -> t, (t1, t2) -> t1));
            }
        }
        if (fileInfoMap == null) {
            fileInfoMap = Collections.emptyMap();
        }

        for (TmplImportRecordRespVO respVO : respVoList) {
            if (StringUtils.hasText(respVO.getFileCode())) {
                respVO.setFileInfo(fileInfoMap.get(respVO.getFileCode()));
                respVO.setFileInfoList(respVO.getFileInfo() == null ? Collections.emptyList() : List.of(respVO.getFileInfo()));
            } else {
                respVO.setFileInfoList(fileCodeMap.getOrDefault(respVO.getId(), Collections.emptyList()).stream()
                        .filter(fileInfoMap::containsKey).map(fileInfoMap::get).collect(Collectors.toList()));
            }
        }

        return ApiResult.ok(PagingVO.<TmplImportRecordRespVO>builder().total(recordDoPage.getTotal()).records(respVoList).build());
    }

    @Override
    public ApiResult<List<FileObjRespVO<String>>> listFileOfExport(Long recordId) {
        var record = tmplImportRecordRepoProc.get(recordId);
        if (record == null) {
            return ApiResult.fail("导出记录不存在，请稍后再试");
        }
        if (Boolean.FALSE.equals(record.getSucc())) {
            // 导出失败
            return ApiResult.fail("导出失败，" + CharSequenceUtil.blankToDefault(record.getFailReason(), "请联系管理员"));
        }

        var fileCode = record.getFileCode();
        if (StringUtils.hasText(fileCode)) {
            // 有单一的文件
            var fileResult = fileService.get(fileCode);
            if (fileResult.getData() != null) {
                return ApiResult.ok(List.of(fileResult.getData()));
            }
            return ApiResult.ok();
        }

        // 是否是多个文件的列表
        var fileCodes = tmplImportFileRepoProc.queryFileCodeByRecordId(recordId);
        if (fileCodes.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        var fileList = fileService.query(fileCodes).getData();
        if (CollUtil.isEmpty(fileList)) {
            return ApiResult.ok();
        }
        var fileInfoMap = fileList.stream().collect(Collectors.toMap(FileObjRespVO::getFileCode, t -> t, (t1, t2) -> t1));
        var fileInfoList = fileCodes.stream().filter(fileInfoMap::containsKey).map(fileInfoMap::get).collect(Collectors.toList());
        return ApiResult.ok(fileInfoList);
    }

    @Override
    public ApiResult<List<ImportRecordDTO>> queryRecord(String tmplCode, Integer hours, Boolean self) {
        Assert.hasText(tmplCode, "模板编码不能为空");
        hours = hours == null ? 72 : hours;
        Long userId = null;
        if (self == null || self) {
            userId = super.currentUser(true).getUserId();
        }

        var tmplId = tenantDataIsolateProvider.byDefaultDirectly(() -> tmplRepoProc.getIdByCode(tmplCode));
        if (tmplId == null) {
            // 未查询到模板
            return ApiResult.ok(Collections.emptyList());
        }

        TmplImportRecordConvert convert = TmplImportRecordConvert.INSTANCE;
        var recordList = tmplImportRecordRepoProc.queryRecentlyList(tmplId, hours, userId).stream()
                .sorted(Comparator.comparing(SysTmplImportRecordDO::getTimeImport).reversed())
                .map(t -> {
                    var dto = convert.do2Dto(t);
                    dto.setFailMsgCustom(t.getCustomTipMsg());
                    return dto;
                })
                .collect(Collectors.toList());
        return ApiResult.ok(recordList);
    }

    private Long save(Long tmplId, String tmplCode, String fileCode, Map<String, Object> param) {
        var recordDO = new SysTmplImportRecordDO();
        recordDO.setTmplId(tmplId);
        recordDO.setTmplCode(tmplCode);
        recordDO.setFileCode(fileCode);
        recordDO.setExport(tenantDataIsolateProvider.byDefaultDirectly(() -> tmplRepoProc.getExport(tmplId)));

        var user = SecurityContextUtil.currentUser();
        if (user != null) {
            recordDO.setUserId(user.getUserId());
            recordDO.setUserName(user.getUsername());
        }

        recordDO.setTimeImport(LocalDateTime.now());
        recordDO.setFinish(false);
        try {
            recordDO.setQueryParam(objectMapper.writeValueAsString(param));
        } catch (JsonProcessingException e) {
            log.error("转换导入导出记录的查询参数失败", e);
        }

        return tmplImportRecordRepo.save(recordDO).getId();
    }

    private String zipFiles(Long id) {
        boolean isZip = SpringContextHolder.getProperty("elitesland.excel.export.zip", Boolean.class, true);
        if (!isZip) {
            // 不压缩
            return null;
        }

        var files = tmplImportFileRepoProc.queryByRecordId(id);
        if (files.isEmpty()) {
            return null;
        }

        String[] fileNames = new String[files.size()];
        InputStream[] fileStreams = new InputStream[files.size()];
        int i = 0;
        for (SysTmplImportFileDO file : files) {
            fileNames[i] = file.getFileName();
            try {
                var fileStream = fileService.download(file.getFileCode(), null).getBody();
                fileStreams[i] = super.streamingResponseBodyToInputStream(fileStream);
            } catch (IOException e) {
                log.error("压缩文件失败", e);
                return null;
            }

            i++;
        }

        String fileName = fileNames[0].substring(0, fileNames[0].lastIndexOf("_")) + ".zip";
        var zipFile = createTempFile(fileName);
        ZipUtil.zip(zipFile, fileNames, fileStreams);

        return fileService.upload(zipFile).getData().getFileCode();
    }

    private File createTempFile(String fileName) {
        File file = new File(System.getProperty("java.io.tmpdir"), fileName);
        file.delete();
        try {
            file.createNewFile();
        } catch (IOException e) {
            throw new RuntimeException("创建临时文件失败", e);
        }
        return file;
    }
}
