package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.UdcConvert;
import com.elitescloud.cloudt.system.dto.SysUdcDTO;
import com.elitescloud.cloudt.system.model.bo.SysUdcBO;
import com.elitescloud.cloudt.system.model.bo.SysUdcValueBO;
import com.elitescloud.cloudt.system.model.vo.query.udc.UdcPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.udc.UdcWithValuesPageRespVO;
import com.elitescloud.cloudt.system.service.UdcQueryService;
import com.elitescloud.cloudt.system.service.repo.AppRepoProc;
import com.elitescloud.cloudt.system.service.repo.UdcRepoProc;
import com.elitescloud.cloudt.system.service.repo.UdcValueRepoProc;
import com.google.common.base.Functions;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/24
 */
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Service
@Log4j2
public class UdcQueryServiceImpl extends BaseServiceImpl implements UdcQueryService {
    @Autowired
    private UdcRepoProc udcRepoProc;
    @Autowired
    private UdcValueRepoProc udcValueRepoProc;
    @Autowired
    private AppRepoProc appRepoProc;

    @Override
    public ApiResult<SysUdcDTO> get(String appCode, String udcCode) {
        if (CharSequenceUtil.hasBlank(appCode, udcCode)) {
            return ApiResult.fail("应用编码或UDC编码为空");
        }
        // 先查询值
        var valueBoList = udcValueRepoProc.listBoByUdc(appCode, udcCode, false);
        if (valueBoList.isEmpty()) {
            return ApiResult.noData();
        }
        // 查询UDC项
        var udcBo = udcRepoProc.getBoByAppCodeAndUdcCode(appCode, udcCode);
        if (udcBo == null) {
            return ApiResult.noData();
        }

        // 转dto
        var udcDTO = bo2Dto(udcBo, valueBoList);

        return ApiResult.ok(udcDTO);
    }

    @Override
    public ApiResult<List<SysUdcDTO>> listByUdcCode(String appCode, Set<String> udcCodes) {
        if (!StringUtils.hasText(appCode)) {
            return ApiResult.fail("应用编码为空");
        }

        // 查询值
        var valueBoList = udcValueRepoProc.queryBoByUdc(appCode, udcCodes, true);
        if (valueBoList.isEmpty()) {
            return ApiResult.noData();
        }
        // 转换dto
        var udcDtoList = udcValue2Dto(valueBoList);

        return ApiResult.ok(udcDtoList);
    }

    @Override
    public ApiResult<PagingVO<UdcWithValuesPageRespVO>> pageQuery(UdcPageQueryVO queryVO) {
        // 查询UDC
        var pageData = udcRepoProc.pageQuery(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        // 查询UDC值
        Map<String, Set<String>> appCodeAndUdcCodeMap = new HashMap<>(pageData.size());
        pageData.each(t -> appCodeAndUdcCodeMap.computeIfAbsent(t.getAppCode(), a -> new HashSet<>(8))
                .add(t.getUdcCode()));
        Map<String, List<SysUdcValueBO>> udcValueMap = new HashMap<>(pageData.size());
        for (Map.Entry<String, Set<String>> entry : appCodeAndUdcCodeMap.entrySet()) {
            udcValueRepoProc.queryBoByUdc(entry.getKey(), entry.getValue(), false)
                    .forEach(t -> {
                        udcValueMap.computeIfAbsent(t.getAppCode() + ":_:" + t.getUdcCode(), k -> new ArrayList<>(16)).add(t);
                    });
        }

        // 转为VO
        var appMap = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.allParams(null)).stream()
                        .collect(Collectors.toMap(IdCodeNameParam::getCode, Functions.identity(), (t1, t2) -> t1));
        pageData.each(t -> {
            ObjUtil.ifNotNull(appMap.get(t.getAppCode()), app -> t.setAppName(app.getName()));

            var valueList = udcValueMap.get(t.getAppCode() + ":_:" + t.getUdcCode());
            if (CollUtil.isEmpty(valueList)) {
                t.setValueList(Collections.emptyList());
                return;
            }
            var valueRespList = valueList.stream().map(value -> {
                UdcWithValuesPageRespVO.UdcValue valueResp = new UdcWithValuesPageRespVO.UdcValue();
                valueResp.setUdcValueCode(value.getUdcValueCode());
                valueResp.setUdcValueName(CharSequenceUtil.blankToDefault(value.getValueAliasName(), value.getUdcValueName()));
                valueResp.setUdcOrder(value.getUdcOrder());

                return valueResp;
            }).collect(Collectors.toList());
            t.setValueList(valueRespList);
        });

        return ApiResult.ok(pageData);
    }

    private SysUdcDTO.UdcValue valueBo2Dto(SysUdcValueBO valueBO) {
        SysUdcDTO.UdcValue value = new SysUdcDTO.UdcValue();
        value.setUdcValueCode(valueBO.getUdcValueCode());
        value.setUdcValueName(CharSequenceUtil.blankToDefault(valueBO.getValueAliasName(), valueBO.getUdcValueName()));
        value.setUdcOrder(ObjectUtil.defaultIfNull(valueBO.getUdcOrder(), 0));
        value.setUdcValueDescribe(valueBO.getUdcValueDescribe());
        value.setEnabled(ObjectUtil.defaultIfNull(valueBO.getAllowStart(), true));
        value.setParentUdcValueCode(valueBO.getParentUdcValueCode());

        return value;
    }

    private SysUdcDTO udcBo2Dto(SysUdcBO udcBO) {
        SysUdcDTO udcDTO = new SysUdcDTO();
        udcDTO.setAppCode(udcBO.getAppCode());
        udcDTO.setUdcCode(udcBO.getUdcCode());
        udcDTO.setUdcName(udcBO.getUdcName());
        udcDTO.setUdcDescribe(udcBO.getUdcDescribe());
        udcDTO.setParentUdcCode(udcBO.getParentUdcCode());

        return udcDTO;
    }

    private SysUdcDTO bo2Dto(SysUdcBO udcBO, List<SysUdcValueBO> valueBoList) {
        var udcDto = udcBo2Dto(udcBO);
        var valueDtoList = valueBoList.stream()
                .map(this::valueBo2Dto)
                .sorted(Comparator.comparingInt(SysUdcDTO.UdcValue::getUdcOrder))
                .collect(Collectors.toList());
        udcDto.setValueList(valueDtoList);

        return udcDto;
    }

    private List<SysUdcDTO> udcValue2Dto(List<SysUdcValueBO> valueBos) {
        if (CollectionUtils.isEmpty(valueBos)) {
            return Collections.emptyList();
        }

        List<SysUdcDTO> dtoList = new ArrayList<>(valueBos.size());
        // 根据应用分组
        var udcAppCodeMap = valueBos.stream().collect(Collectors.groupingBy(SysUdcValueBO::getAppCode));
        for (Map.Entry<String, List<SysUdcValueBO>> appCodeEntry : udcAppCodeMap.entrySet()) {
            // 根据UDC编码分组，查询对应的UDC信息
            var udcCodeMap = appCodeEntry.getValue().stream().collect(Collectors.groupingBy(SysUdcValueBO::getUdcCode));
            var udcBoMap = udcRepoProc.queryBoByUdcCode(appCodeEntry.getKey(), udcCodeMap.keySet()).stream()
                    .collect(Collectors.toMap(SysUdcBO::getUdcCode, t -> t, (t1, t2) -> t1));

            var udcDtoList = udcCodeMap.entrySet().stream()
                    .filter(t -> udcBoMap.containsKey(t.getKey()))
                    .map(t -> bo2Dto(udcBoMap.get(t.getKey()), t.getValue())).collect(Collectors.toList());
            dtoList.addAll(udcDtoList);
        }
        return dtoList;
    }
}
