package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.UserSyncConvert;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.model.vo.query.user.sync.UserSyncRecordPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.sync.UserSyncRecordDetailRespVO;
import com.elitescloud.cloudt.system.provider.usersync.SyncUserResult;
import com.elitescloud.cloudt.system.provider.usersync.UserSyncProvider;
import com.elitescloud.cloudt.system.service.UserSyncMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysUserSyncDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserSyncRecordDO;
import com.elitescloud.cloudt.system.service.repo.UserRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserSyncRecordRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserSyncRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/7/30
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
public class UserSyncMngServiceImpl implements UserSyncMngService {
    private static final Logger logger = LoggerFactory.getLogger(UserSyncMngServiceImpl.class);

    @Autowired
    private UserSyncRepoProc syncRepoProc;
    @Autowired
    private UserSyncRecordRepoProc syncRecordRepoProc;
    @Autowired
    private UserRepoProc userRepoProc;

    @Autowired
    private List<UserSyncProvider<? extends Serializable, ? extends Serializable>> syncProviderList;

    @Override
    public ApiResult<PagingVO<UserSyncRecordDetailRespVO>> pageQuery(UserSyncRecordPageQueryVO queryVO) {
        var defaultOuterSysCode = getDefaultOuterSysCode();
        var pageData = syncRepoProc.pageMng(queryVO, defaultOuterSysCode);
        return ApiResult.ok(pageData);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> syncUser(String outerSysCode, Long userId, Object syncData) {
        Assert.notNull(userId, "账号ID为空");
        Assert.hasText(outerSysCode, "外部系统编码为空");

        UserSyncProvider syncProvider = this.findSyncProvider(outerSysCode);
        Assert.notNull(syncProvider, "尚未配置该同步方式");

        var userDTO = userRepoProc.getBasicDto(userId);
        Assert.notNull(userDTO, "同步账号不存在");

        // 保存同步记录
        Serializable toSyncData = syncProvider.convertSyncData(HttpServletUtil.currentRequest(), userDTO, (Serializable) syncData);
        var record = this.saveSyncRecord(syncProvider, userDTO, toSyncData);
        var userSync = this.saveSync(record);

        // 开始同步
        Exception exp = null;
        SyncUserResult syncUserResult = null;
        try {
            syncUserResult = syncProvider.syncUser(userId, toSyncData);
        } catch (Exception e) {
            logger.error("同步账号异常：{}, {}, {}", outerSysCode, userId, userDTO.getUsername(), e);
            exp = e;
        } finally {
            String failReasonDetail = exp == null ? null : ExceptionsUtil.stackTraceAllToString(exp);
            if (syncUserResult == null) {
                syncUserResult = new SyncUserResult(userId);
                syncUserResult.setSuccess(exp == null);
                syncUserResult.setFailMsg(exp == null ? null : exp.getMessage());
            }

            syncRepoProc.updateSyncResult(userSync.getId(), syncUserResult, failReasonDetail);
            syncRecordRepoProc.updateSyncResult(record.getId(), syncUserResult, failReasonDetail);
        }

        return ApiResult.ok(userId);
    }

    @Override
    public ApiResult<List<UserSyncRecordDetailRespVO>> listSyncRecord(String outerSysCode, Long userId) {
        Assert.notNull(userId, "账号ID为空");
        Assert.hasText(outerSysCode, "系统编码为空");

        // 查询记录
        var recordDoList = syncRecordRepoProc.listSyncRecord(outerSysCode, userId);
        if (recordDoList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 转换对象
        var respVoList = recordDoList.stream()
                .sorted(Comparator.comparing(SysUserSyncRecordDO::getSyncTime).reversed())
                .map(t -> {
                    var respVO = UserSyncConvert.INSTANCE.do2respVO(t);
                    respVO.setId(t.getUserId());
                    respVO.setFullName(t.getFullName());
                    respVO.setSync(Boolean.TRUE.equals(t.getSyncSuccess()));
                    return respVO;
                })
                .collect(Collectors.toList());

        return ApiResult.ok(respVoList);
    }

    private SysUserSyncRecordDO saveSyncRecord(UserSyncProvider<?, ?> provider, SysUserBasicDTO userBasicDTO, Object syncData) {
        SysUserSyncRecordDO recordDO = new SysUserSyncRecordDO();
        recordDO.setOuterSysCode(provider.sysCode());
        recordDO.setUserId(userBasicDTO.getId());
        recordDO.setUsername(userBasicDTO.getUsername());
        recordDO.setLastName(userBasicDTO.getLastName());
        recordDO.setFirstName(userBasicDTO.getFirstName());
        recordDO.setMobile(userBasicDTO.getMobile());
        recordDO.setEmail(userBasicDTO.getEmail());
        recordDO.setIdCard(userBasicDTO.getIdCard());
        recordDO.setSyncTime(LocalDateTime.now());
        recordDO.setSyncSuccess(false);
        recordDO.setSyncUrl(provider.getSyncUrl());
        recordDO.setSyncData(JSONUtil.convertObj(syncData, new TypeReference<>() {
        }));
        syncRecordRepoProc.save(recordDO);

        return recordDO;
    }

    private SysUserSyncDO saveSync(SysUserSyncRecordDO recordDO) {
        // 先判断是否已有同步
        var userSyncDO = syncRepoProc.getUserSync(recordDO.getUserId(), recordDO.getOuterSysCode());
        if (userSyncDO == null) {
            // 不存在则新建
            userSyncDO = new SysUserSyncDO();
            userSyncDO.setUserId(recordDO.getUserId());
            userSyncDO.setOuterSysCode(recordDO.getOuterSysCode());
        }

        userSyncDO.setSyncTime(recordDO.getSyncTime());
        userSyncDO.setSyncSuccess(false);
        userSyncDO.setFailReason(null);
        userSyncDO.setSyncUrl(recordDO.getSyncUrl());
        userSyncDO.setSyncData(recordDO.getSyncData());
        syncRepoProc.save(userSyncDO);

        return userSyncDO;
    }

    private UserSyncProvider<?, ?> findSyncProvider(String outerSysCode) {
        if (CollUtil.isEmpty(syncProviderList) || CharSequenceUtil.isBlank(outerSysCode)) {
            return null;
        }

        for (var s : syncProviderList) {
            if (outerSysCode.equals(s.sysCode())) {
                return s;
            }
        }
        return null;
    }

    private String getDefaultOuterSysCode() {
        if (CollUtil.isEmpty(syncProviderList)) {
            return null;
        }

        for (var s : syncProviderList) {
            if (Boolean.FALSE.equals(s.enabled())) {
                continue;
            }
            return s.sysCode();
        }
        return null;
    }
}
