package com.elitescloud.cloudt.system.service.manager;

import com.elitescloud.boot.model.entity.BaseTreeModel;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.constant.OrgTreeNodeType;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.resp.OrgTreeNodeRespVO;
import com.elitescloud.cloudt.system.service.common.constant.ConstantCache;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepoProc;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/9
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class OrgTreeManager {

    @Autowired
    private OrgRepoProc orgRepoProc;
    @Autowired
    private EmployeeRepoProc employeeRepoProc;

    @Autowired
    private RedisUtils redisUtils;

    /**
     * 获取组织树
     *
     * @param orgIdBelong     所能看到的顶级组织ID
     * @param tree            是否返回树形数据，false则返回成list
     * @param includeDisabled 是否包含已禁用的
     * @param includeEmployee 是否包含员工
     * @return 组织树
     */
    @SuppressWarnings("unchecked")
    public List<OrgTreeNodeRespVO> getTreeCacheable(Long orgIdBelong, boolean tree, boolean includeDisabled, boolean includeEmployee) {
        StringJoiner keyJoiner = new StringJoiner("__")
                .add("orgIdBelong=" + (orgIdBelong == null ? "" : orgIdBelong.toString()))
                .add("tree=" + tree)
                .add("includeDisabled=" + includeDisabled)
                .add("includeEmployee=" + includeEmployee);
        String key = keyJoiner.toString();

        // 优先从缓存获取
        Object result = redisUtils.hget(ConstantCache.HASH_KEY_ORG_TREE, key);
        if (result != null) {
            return (List<OrgTreeNodeRespVO>) result;
        }

        var treeData = getTree(orgIdBelong, tree, includeDisabled, includeEmployee);
        redisUtils.hset(ConstantCache.HASH_KEY_ORG_TREE, key, treeData);

        return treeData;
    }

    public List<OrgTreeNodeRespVO> getTree(Long orgIdBelong, boolean tree, boolean includeDisabled, boolean includeEmployee) {
        List<OrgTreeNodeRespVO> orgList = orgRepoProc.getOrgTree(orgIdBelong, includeDisabled);
        if (orgList.isEmpty()) {
            return orgList;
        }

        // 查询员工
        List<OrgTreeNodeRespVO> employeeNodeList = null;
        if (includeEmployee) {
            var orgIdCodeMap = orgList.stream().collect(Collectors.toMap(OrgTreeNodeRespVO::getId, OrgTreeNodeRespVO::getCode, (t1, t2) -> t1));

            employeeNodeList = employeeRepoProc.queryEmployee(null, orgIdBelong, null, includeDisabled)
                    .stream()
                    .filter(t -> t.getOrgId() != null)
                    .map(t -> employee2OrgTreeNode(t, orgIdCodeMap))
                    .collect(Collectors.toList());
            orgList.addAll(employeeNodeList);
        }

        if (tree) {
            // 需要转为树形结构
            TreeDataUtil<OrgTreeNodeRespVO> treeDataUtil = new TreeDataUtil<>(orgList, OrgTreeNodeRespVO::getId,
                    OrgTreeNodeRespVO::getParentId, OrgTreeNodeRespVO::setChildren, Comparator.comparingInt(OrgTreeNodeRespVO::getSortNo));
            var roots = (List<OrgTreeNodeRespVO>) treeDataUtil.getRoots();
            for (OrgTreeNodeRespVO root : roots) {
                // 顶级的去掉parentCode
                root.setParentCode(null);
                root.setParentId(BaseTreeModel.DEFAULT_PARENT);
            }
            return roots;
        }

        // 顶级的去掉parentCode
        TreeDataUtil<OrgTreeNodeRespVO> treeDataUtil = new TreeDataUtil<>(orgList, OrgTreeNodeRespVO::getId,
                OrgTreeNodeRespVO::getParentId, OrgTreeNodeRespVO::setChildren);
        var roots = (List<OrgTreeNodeRespVO>) treeDataUtil.getRoots();
        for (OrgTreeNodeRespVO root : roots) {
            root.setParentCode(null);
            root.setParentId(BaseTreeModel.DEFAULT_PARENT);
        }

        return CollectionUtil.expandTree(roots, OrgTreeNodeRespVO::getChildren);
    }

    /**
     * 异步获取组织树
     *
     * @param parentId        父节点ID
     * @param includeDisabled 是否包含已禁用的
     * @param includeEmployee 是否包含员工
     * @return 是否包含员工
     */
    public List<OrgTreeNodeRespVO> getTreeAsync(Long parentId, boolean includeDisabled, boolean includeEmployee) {
        if (parentId == null) {
            parentId = BaseTreeModel.DEFAULT_PARENT;
        }

        // 先获取组织
        var orgList = orgRepoProc.getOrgTreeByParentId(parentId, includeDisabled);
        if (orgList.isEmpty()) {
            // 没有下级组织
            if (parentId == BaseTreeModel.DEFAULT_PARENT) {
                // 查找的是根组织
                return orgList;
            }

            // 查找员工
            if (includeEmployee) {
                // 查询组织编码
                String orgCode = orgRepoProc.getCodeById(parentId);
                Map<Long, String> orgIdCodeMap = new HashMap<>(4);
                orgIdCodeMap.put(parentId, orgCode);

                return employeeRepoProc.queryEmployee(parentId, null, null, includeDisabled)
                        .stream()
                        .map(t -> employee2OrgTreeNode(t, orgIdCodeMap))
                        .collect(Collectors.toList());
            }
        }

        if (includeEmployee) {
            // 没有子组织的，判断是否有员工
            var orgIdOfNoChild = orgList.stream().filter(t -> !t.getHasChildren()).map(OrgTreeNodeRespVO::getId).collect(Collectors.toSet());
            if (!orgIdOfNoChild.isEmpty()) {
                var countMap = employeeRepoProc.countEmployee(orgIdOfNoChild, includeDisabled);
                for (OrgTreeNodeRespVO respVO : orgList) {
                    if (respVO.getHasChildren()) {
                        continue;
                    }
                    respVO.setHasChildren(countMap.getOrDefault(respVO.getId(), 0) > 0);
                }
            }
            // 查询组织编码
            String orgCode = orgRepoProc.getCodeById(parentId);
            Map<Long, String> orgIdCodeMap = new HashMap<>(4);
            orgIdCodeMap.put(parentId, orgCode);

            // 查找员工
            var employeeList = employeeRepoProc.queryEmployee(parentId, null, null, includeDisabled)
                    .stream()
                    .map(t -> employee2OrgTreeNode(t, orgIdCodeMap))
                    .collect(Collectors.toList());
            orgList.addAll(employeeList);
        }
        return orgList;
    }

    public void clearCache() {
        redisUtils.hdel(ConstantCache.HASH_KEY_ORG_TREE);
    }

    private OrgTreeNodeRespVO employee2OrgTreeNode(SysEmployeeBasicDTO employee, Map<Long, String> orgIdCodeMap) {
        OrgTreeNodeRespVO respVO = new OrgTreeNodeRespVO();
        respVO.setId(employee.getId());
        respVO.setCode(employee.getCode());
        respVO.setName(employee.getFullName());
        respVO.setNodeType(OrgTreeNodeType.EMPLOYEE.getValue());
        respVO.setNodeTypeName(OrgTreeNodeType.EMPLOYEE.getDescription());
        respVO.setDataType(employee.getType());
        respVO.setSortNo(employee.getSortNo());
        respVO.setParentId(employee.getOrgId());
        respVO.setParentCode(orgIdCodeMap.get(employee.getOrgId()));
        respVO.setHasChildren(false);
        respVO.setChildren(Collections.emptyList());
        respVO.setUserId(employee.getUserId());

        return respVO;
    }
}
