package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.param.IdCodeNameCheckParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleConditionEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleRelationEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprSysInternallyEnum;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.LockUtil;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.platform.model.vo.resp.MenuOperationRespVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.common.BusinessObjectNodeType;
import com.elitescloud.cloudt.system.constant.DataPermissionType;
import com.elitescloud.cloudt.system.convert.RoleConvert;
import com.elitescloud.cloudt.system.convert.SysDpcRoleApiFieldsConvert;
import com.elitescloud.cloudt.system.convert.SysDprRoleApiRuleConvert;
import com.elitescloud.cloudt.system.model.bo.*;
import com.elitescloud.cloudt.system.model.vo.resp.api.SysMenuApiRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.permission.DataPermissionTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RolePermissionRespVO;
import com.elitescloud.cloudt.system.model.vo.save.dpr.DataPermissionFieldSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.dpr.DataPermissionRuleSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.dpr.DataPermissionSaveVO;
import com.elitescloud.cloudt.system.service.common.constant.BelongType;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectRefTypeEnum;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.model.bo.*;
import com.elitescloud.cloudt.system.service.model.entity.*;
import com.elitescloud.cloudt.system.service.repo.*;
import com.google.common.base.Functions;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 权限管理.
 *
 * @author Kaiser（wang shao）
 * 2022/10/16
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@Log4j2
public class PermissionMngManager extends BasePermissionManager {

    @Autowired
    private RolePermissionRepoProc rolePermissionRepoProc;
    @Autowired
    private TenantMenuTreeRepoProc tenantMenuTreeRepoProc;
    @Autowired
    private TenantMenuRepoProc tenantMenuRepoProc;
    @Autowired
    private ApiRepoProc apiRepoProc;
    @Autowired
    private MenuApiRepoProc menuApiRepoProc;
    @Autowired
    private SysDprRoleApiRowRuleRepoProc roleApiRowRuleRepoProc;
    @Autowired
    private SysDpcrApiFieldsRepoProc apiFieldsRepoProc;

    /**
     * 保存用户角色
     *
     * @param userId  用户ID
     * @param roleIds 角色ID
     */
    public void saveUserRole(@NotNull Long userId, List<Long> roleIds) {
        var userRoles = userRoleRepoProc.getByUser(userId, BelongType.getBelonger());
        if (CollectionUtils.isEmpty(roleIds)) {
            // 没有指定角色，则删除所有
            var ids = userRoles.stream().map(SysUserRoleDO::getId).collect(Collectors.toList());
            if (!ids.isEmpty()) {
                userRoleRepoProc.delete(ids);
            }
            return;
        }

        // 过滤存在的角色ID
        roleIds = roleRepoProc.exists(roleIds);
        if (roleIds.isEmpty()) {
            throw new BusinessException("部分角色不存在");
        }

        var userRoleMap = userRoles.stream().collect(Collectors.toMap(SysUserRoleDO::getRoleId, t -> t, (t1, t2) -> t1));
        // 新增的角色
        var userRoleToAdd = roleIds.stream()
                .filter(t -> !userRoleMap.containsKey(t))
                .map(t -> {
                    SysUserRoleDO userRoleDO = new SysUserRoleDO();
                    userRoleDO.setRoleId(t);
                    userRoleDO.setUserId(userId);
                    return userRoleDO;
                }).collect(Collectors.toList());
        if (!userRoleToAdd.isEmpty()) {
            userRoleRepoProc.save(userRoleToAdd);
        }

        // 需要删除的
        List<Long> finalRoleIds = roleIds;
        var idsToDel = userRoles.stream()
                .filter(t -> !finalRoleIds.contains(t.getRoleId()))
                .map(SysUserRoleDO::getId)
                .collect(Collectors.toList());
        if (!idsToDel.isEmpty()) {
            userRoleRepoProc.delete(idsToDel);
        }
    }

    /**
     * 获取用户角色
     *
     * @param userId  用户ID
     * @param onlyOwn 是否仅仅返回用户所拥有的
     * @return 角色列表
     */
    public List<IdCodeNameCheckParam> getUserRole(@NotNull Long userId, boolean onlyOwn) {
        return userRoleRepoProc.getByUser(userId, BelongType.getBelonger(), onlyOwn);
    }

    /**
     * 新增用户角色
     */
    public void addUserRole(Set<Long> userIds, Set<String> roleCodes) {
        if (CollUtil.isEmpty(userIds) || CollUtil.isEmpty(roleCodes)) {
            return;
        }

        // 获取已有角色
        var existsUserRoleMap = userRoleRepoProc.getRoleCodesOfUser(userIds, null);
        var roleCodeIdMap = roleRepoProc.getIdAndCodeByCode(roleCodes);

        List<SysUserRoleDO> userRoleList = new ArrayList<>();
        for (Long userId : userIds) {
            var existsUserRoleCodes = existsUserRoleMap.get(userId);
            for (String roleCode : roleCodes) {
                if (existsUserRoleCodes != null && existsUserRoleCodes.contains(roleCode)) {
                    continue;
                }
                if (!roleCodeIdMap.containsKey(roleCode)) {
                    continue;
                }
                SysUserRoleDO userRoleDO = new SysUserRoleDO();
                userRoleDO.setUserId(userId);
                userRoleDO.setRoleId(roleCodeIdMap.get(roleCode));

                userRoleList.add(userRoleDO);
            }
        }
        if (!userRoleList.isEmpty()) {
            userRoleRepoProc.save(userRoleList);
        }
    }

    /**
     * 保存角色下的用户
     *
     * @param roleId       角色ID
     * @param userIds      用户ID
     * @param incremental  是否为增量
     * @param employeeOnly 是否都为员工账号
     */
    public void saveRoleUser(@NotNull Long roleId, List<Long> userIds, boolean incremental, boolean employeeOnly) {
        if (CollUtil.isEmpty(userIds)) {
            if (incremental) {
                return;
            }
            // 非增量的，则需要删除当前的
            userRoleRepoProc.deleteByRole(roleId, employeeOnly);
            return;
        }

        // 先查询目前角色下的所有账号ID
        var userIdsCurrent = userRoleRepoProc.getUserIdsByRole(roleId, employeeOnly);
        var userRoleDoList = userIds.stream().filter(t -> !userIdsCurrent.contains(t))
                .map(t -> {
                    SysUserRoleDO userRoleDO = new SysUserRoleDO();
                    userRoleDO.setUserId(t);
                    userRoleDO.setRoleId(roleId);

                    return userRoleDO;
                }).collect(Collectors.toList());
        if (!userRoleDoList.isEmpty()) {
            userRoleRepoProc.save(userRoleDoList);
        }

        // 如果不是增量，则需要删除不存在的
        if (!incremental) {
            Set<Long> toDelUserIds = userIdsCurrent.isEmpty() ? Collections.emptySet() :
                    userIdsCurrent.stream().filter(t -> !userIds.contains(t)).collect(Collectors.toSet());
            userRoleRepoProc.deleteRoleUser(roleId, toDelUserIds);
        }
    }

    /**
     * 删除角色下的指定账号
     *
     * @param roleId  角色ID
     * @param userIds 用户ID
     */
    public void removeRoleUser(@NotNull Long roleId, List<Long> userIds) {
        if (CollUtil.isEmpty(userIds)) {
            return;
        }
        userRoleRepoProc.deleteRoleUser(roleId, userIds);
    }

    /**
     * 保存权限资源
     *
     * @param permissionResList 权限列表
     * @param assigner          分配者
     */
    public void savePermissionRes(@NotEmpty List<PermissionResBO> permissionResList, BelongType.Belonger assigner) {
        Assert.notEmpty(permissionResList, "资源列表为空");
        Assert.notNull(assigner, "分配人为空");

        // 检查资源
        this.checkPermissionRes(permissionResList);

        // 转换对象
        var permissionResDoList = permissionResList.stream()
                .flatMap(t -> {
                    if (CollUtil.isEmpty(t.getResList())) {
                        return Stream.empty();
                    }

                    return t.getResList().stream()
                            .map(tt -> {
                                SysPermissionResDO permissionResDO = new SysPermissionResDO();
                                permissionResDO.setResType(tt.getResTypeEnum().getValue());
                                permissionResDO.setResId(tt.getResId());
                                permissionResDO.setOwnerType(t.getOwnerTypeEnum().name());
                                permissionResDO.setOwnerId(t.getOwnerId());
                                permissionResDO.setAssignerType(assigner.getBelongType().getValue());
                                permissionResDO.setAssignerId(assigner.getBelongId());

                                return permissionResDO;
                            });
                }).collect(Collectors.toList());
        // 先清除已有的
        for (PermissionResBO permissionResBO : permissionResList) {
            permissionResRepoProc.delete(assigner, permissionResBO.getOwnerTypeEnum(), permissionResBO.getOwnerId());
        }
        // 保存新的
        if (!permissionResDoList.isEmpty()) {
            permissionResRepoProc.save(permissionResDoList);
        }
    }

    /**
     * 保存角色与菜单的关联
     * <p>
     * 包含按钮等
     *
     * @param roleId     角色ID
     * @param saveVOList 权限信息
     */
    public void saveRoleMenu(@NotNull Long roleId, List<SysAppPermissionSaveBO> saveVOList) {
        Assert.notNull(roleId, "角色ID为空");
//        boolean hasCustom = hasCustomMenuTree(null);
        boolean hasCustom = false;

        saveRolePermission(roleId, saveVOList, List.of(MenuTreeNodeType.APP, MenuTreeNodeType.MENU, MenuTreeNodeType.ACTION), hasCustom);
    }

    /**
     * 添加角色与菜单的关联
     *
     * @param roleId 角色ID
     * @param saveBO 权限信息
     */
    public void addRoleMenu(long roleId, SysAppPermissionSaveBO saveBO) {
//        boolean hasCustom = hasCustomMenuTree(null);
        Boolean hasCustom = null;

        addRolePermission(roleId, saveBO, List.of(MenuTreeNodeType.APP, MenuTreeNodeType.MENU, MenuTreeNodeType.ACTION), hasCustom);
    }

    /**
     * 移除角色与菜单的关联
     *
     * @param roleId 角色ID
     * @param saveBO 权限信息
     */
    public void removeRoleMenu(long roleId, SysAppPermissionSaveBO saveBO) {
//        boolean hasCustom = hasCustomMenuTree(null);
        Boolean hasCustom = null;

        saveBO.getPermissionList().stream().collect(Collectors.groupingBy(SysPermissionSaveBO::getNodeType))
                .forEach((nodeType, permissionList) -> {
                    var codes = permissionList.stream().map(SysPermissionSaveBO::getCode).collect(Collectors.toSet());
                    rolePermissionRepoProc.deleteByRole(roleId, saveBO.getAppCode(), nodeType, codes, hasCustom);
                });
    }

    /**
     * 根据角色获取权限菜单
     * <p>
     * 如果传递了appCode，则仅查询对应应用下的权限菜单，否则查询所有（或租户所拥有）得权限菜单
     *
     * @param roleId        角色ID
     * @param appCode       应用编码，如果为空则是租户的所有应用
     * @param includeApp    是否返回应用
     * @param includeAction 是否返回按钮
     * @param tree          是否返回树形数据
     * @return 权限菜单列表
     */
    public List<RolePermissionRespVO> getPermissionMenuByRole(@NotNull Long roleId, String appCode, boolean includeApp,
                                                              boolean includeAction, boolean tree) {
        // 获取有效应用
        Map<String, AppBO> tenantApp = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            // 获取应用列表
            return super.tenantApps(appCode);
        });
        var appCodes = tenantApp.keySet();

        // 应用和菜单
        List<RolePermissionRespVO> respVOList = this.queryAppAndMenu(tenantApp, false, includeApp);
        if (respVOList.isEmpty()) {
            return Collections.emptyList();
        }
        // 挂载的接口
        this.fillApiNumOfMenu(appCodes, respVOList);

        // 按钮
        if (includeAction) {
            var menuCodes = respVOList.stream().filter(t -> MenuTreeNodeType.MENU.getValue().equals(t.getNodeType())).map(RolePermissionRespVO::getCode).collect(Collectors.toSet());
            if (!menuCodes.isEmpty()) {
                var actionList = this.queryActionList(appCodes).stream().filter(t -> StringUtils.hasText(t.getParentCode()) && menuCodes.contains(t.getParentCode())).collect(Collectors.toList());
                respVOList.addAll(actionList);
            }
        }
        // 查询已分配的权限
        var checkCodes = rolePermissionRepoProc.getCodeByRole(roleId, Set.of(MenuTreeNodeType.MENU.getValue(), MenuTreeNodeType.ACTION.getValue()));
        return convertPermissionTreeRespVO(respVOList, checkCodes, tree);
    }

    /**
     * 根据角色获取数据权限
     *
     * @param roleId     角色ID
     * @param appCode    应用编码
     * @param includeApp 是否返回应用
     * @param tree       是否返回树状数据
     * @return 权限列表
     */
    public List<DataPermissionTreeNodeRespVO> getDataPermissionByRole(Long roleId, String appCode, boolean includeApp,
                                                                      boolean tree) {
        var currentTenant = tenantClientProvider.getCurrentTenant();
        var tenantId = currentTenant == null ? TenantConstant.DEFAULT_TENANT_ID : currentTenant.getId();
        Map<String, AppBO> tenantApp = new LinkedHashMap<>();

        Map<String, List<SysMenuApiRespVO>> menuApiMap = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            // 获取应用列表
            var tenantApps = super.tenantApps(appCode);
            if (tenantApps.isEmpty()) {
                return Collections.emptyMap();
            }
            tenantApp.putAll(tenantApps);

            // 查询接口
            return menuApiRepoProc.queryApiDetailOfMenu(tenantApp.keySet());
        });
        if (menuApiMap.isEmpty()) {
            return Collections.emptyList();
        }

        // 菜单
        var custom = this.hasCustomMenuTree(tenantId);
        List<RolePermissionRespVO> respVOList = this.queryAppAndMenu(tenantApp, custom, includeApp);
        if (respVOList.isEmpty()) {
            return Collections.emptyList();
        }

        // 分配的权限规则
        List<RoleApiRuleBO> roleApiRuleList = roleId == null ? Collections.emptyList() : roleApiRowRuleRepoProc.queryByRole(roleId);
        var treeNodeList = respVOList.stream().map(this::convertDataPermissionTreeNode).collect(Collectors.toList());
        treeNodeList.addAll(this.convertDataPermissionTreeNode(menuApiMap, roleApiRuleList));

        return this.convertDataPermissionTreeRespVO(treeNodeList, tree, MenuTreeNodeType.API);
    }

    /**
     * 根据角色获取数据权限
     *
     * @param roleCode       角色编码
     * @param permissionType 权限类型
     * @param appCode        应用编码
     * @param includeApp     是否返回应用
     * @param tree           是否返回树状数据
     * @return 权限列表
     */
    public List<DataPermissionTreeNodeRespVO> getDataPermissionByRole(String roleCode, @NotNull DataPermissionType permissionType,
                                                                      boolean withRef, String appCode, boolean includeApp, boolean tree) {
        var currentTenant = tenantClientProvider.getCurrentTenant();
        var tenantId = currentTenant == null ? TenantConstant.DEFAULT_TENANT_ID : currentTenant.getId();

        List<DataPermissionTreeNodeRespVO> treeNodeList = null;
        Assert.notNull(permissionType, "未知权限类型");
        MenuTreeNodeType lastTreeNodeType = null;
        switch (permissionType) {
            case MENU_OPERATION_RULE:
                // 菜单操作权限
                treeNodeList = this.queryDataPermissionTreeOfMenuOperation(roleCode, appCode, tenantId, includeApp);
                lastTreeNodeType = MenuTreeNodeType.BUSINESS_OPERATION;
                break;
            case BUSINESS_OPERATION_RULE:
                // 业务对象的操作权限
                treeNodeList = this.queryDataPermissionTreeOfBusinessOperation(roleCode, appCode, withRef, includeApp);
                lastTreeNodeType = MenuTreeNodeType.BUSINESS_OPERATION;
                break;
            case BUSINESS_OBJECT_RULE:
                // 业务对象的权限
                treeNodeList = this.queryDataPermissionTreeOfBusinessObject(roleCode, appCode, includeApp, false);
                lastTreeNodeType = MenuTreeNodeType.BUSINESS_OBJECT;
                break;
            case BUSINESS_RESOURCE_RULE:
                // 业务对象资源
                treeNodeList = this.queryDataPermissionTreeOfBusinessObject(roleCode, appCode, includeApp, true);
                lastTreeNodeType = MenuTreeNodeType.BUSINESS_OBJECT;
                break;
            default:
                throw new BusinessException("暂不支持的权限类型");
        }

        return this.convertDataPermissionTreeRespVO(treeNodeList, tree, lastTreeNodeType);
    }

    /**
     * 保存数据权限
     *
     * @param saveBO 数据权限信息
     */
    public void saveDataPermission(SysDprSaveBO saveBO) {
        AtomicLong appId = new AtomicLong();
        AtomicReference<List<IdCodeNameParam>> apiList = new AtomicReference<>();
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            var tempAppId = menuRepoProc.getAppIdByMenuCode(saveBO.getMenuCode());
            Assert.notNull(tempAppId, "菜单信息异常，请稍后再试");
            appId.set(tempAppId);
            var apis = menuApiRepoProc.queryApiOfMenu(saveBO.getMenuCode()).stream()
                    .map(t -> {
                        IdCodeNameParam param = new IdCodeNameParam();
                        param.setId(t.getId());
                        param.setCode(t.getApiCode());
                        param.setName(t.getApiName());

                        return param;
                    }).collect(Collectors.toList());
            apiList.set(apis);
            return null;
        });
        var apiCodesAll = apiList.get().stream().map(IdCodeNameParam::getCode).collect(Collectors.toSet());

        // 保存自定义规则
        this.saveDataPermissionCustomRule(appId.get(), saveBO, DprRuleRelationEnum.DPR_RULE_RELATION_AND, apiCodesAll);

        // 保存字段权限
        this.saveDataPermissionField(appId.get(), saveBO);
    }

    /**
     * 保存数据权限
     *
     * @param saveVO 数据权限信息
     */
    public void saveDataPermission(DataPermissionSaveVO saveVO) {
        // 保存参数的检查
        this.check(saveVO);

        // 转换和保存
        this.savePermissionRule(saveVO);

        // 字段权限
        this.savePermissionField(saveVO);
    }

    /**
     * 保存批量数据权限
     *
     * @param saveBO 数据权限信息
     */
    public void saveDataPermissionBatch(SysDprBatchSaveBO saveBO) {
        // 查询菜单信息
        var menuCodes = saveBO.getMenuApiList().stream().map(SysDprBatchSaveBO.MenuApiSaveBO::getMenuCode).collect(Collectors.toList());
        Map<String, MenuSimpleBO> menuMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.listSimpleByMenuCode(menuCodes))
                .stream().collect(Collectors.toMap(MenuSimpleBO::getMenusCode, t -> t, (t1, t2) -> t1));

        // 保存
        Map<String, Long> existsData = null;
        if (SysDprBatchSaveBO.Strategy.APPEND == saveBO.getStrategy()) {
            // 直接追加的，无需做什么
        } else if (SysDprBatchSaveBO.Strategy.REPLACE == saveBO.getStrategy()) {
            // 替换的，先查询已有数据
            existsData = roleApiRowRuleRepoProc.listByRole(saveBO.getRoleId()).stream()
                    .collect(Collectors.toMap(t -> t.getMenuCode() + ":" + t.getApiCode() + ":" + t.getDprRuleField(),
                            SysDprRoleApiRowRuleDO::getId, (t1, t2) -> t1));
        } else if (SysDprBatchSaveBO.Strategy.CLEAR == saveBO.getStrategy()) {
            // 清空已有数据
            for (SysDprBatchSaveBO.MenuApiSaveBO menuApiSaveBO : saveBO.getMenuApiList()) {
                roleApiRowRuleRepoProc.deleteByApi(saveBO.getRoleId(), menuApiSaveBO.getMenuCode(), menuApiSaveBO.getApiCode());
            }
        }

        List<SysDprRoleApiRowRuleDO> ruleList = new ArrayList<>(16);
        SysDprRoleApiRowRuleDO ruleDO = null;
        for (SysDprBatchSaveBO.MenuApiSaveBO menuApiSaveBO : saveBO.getMenuApiList()) {
            SysDprSaveBO dprSaveBO = new SysDprSaveBO();
            dprSaveBO.setRoleId(saveBO.getRoleId());
            dprSaveBO.setMenuCode(menuApiSaveBO.getMenuCode());
            dprSaveBO.setApiCode(menuApiSaveBO.getApiCode());
            dprSaveBO.setCustomRuleList(saveBO.getCustomRuleList());

            var menu = menuMap.get(menuApiSaveBO.getMenuCode());
            Assert.notNull(menu, "菜单不存在");
            int order = 0;
            for (SysDprApiCustomRuleSaveBO customRuleSaveBO : saveBO.getCustomRuleList()) {
                ruleDO = this.convertDprRoleApiRowRuleDO(menu.getMenusAppCode(), dprSaveBO, customRuleSaveBO, order++, DprRuleRelationEnum.DPR_RULE_RELATION_AND);

                if (existsData != null) {
                    var id = existsData.get(menuApiSaveBO.getMenuCode() + ":" + menuApiSaveBO.getApiCode() + ":" + customRuleSaveBO.getDprRuleField());
                    ruleDO.setId(id);
                }

                ruleList.add(ruleDO);
            }
        }

        if (!ruleList.isEmpty()) {
            roleApiRowRuleRepoProc.save(ruleList);
        }
    }

    /**
     * 复制角色权限
     *
     * @param originalId 原始角色ID
     * @param roleId     新角色ID
     */
    public void cloneRolePermission(@NotNull Long originalId, @NotNull Long roleId) {
        Assert.notNull(originalId, "原始角色ID为空");
        Assert.notNull(roleId, "角色ID为空");
        // 先复制角色权限信息
        this.cloneRoleMenu(originalId, roleId);
        // 保存数据权限
        this.cloneRoleDataPermission(originalId, roleId);
    }

    private void checkPermissionRes(List<PermissionResBO> permissionResList) {
        Map<MenuTreeNodeType, Set<String>> resMap = new HashMap<>(8);
        for (PermissionResBO permissionResBO : permissionResList) {
            Assert.notNull(permissionResBO.getOwnerTypeEnum(), "持有者类型为空");
            Assert.hasText(permissionResBO.getOwnerId(), "持有者标识为空");

            if (CollUtil.isEmpty(permissionResBO.getResList())) {
                continue;
            }
            for (PermissionResBO.Res res : permissionResBO.getResList()) {
                Assert.notNull(res.getResTypeEnum(), "资源类型为空");
                Assert.hasText(res.getResId(), "资源标识为空");

                resMap.computeIfAbsent(res.getResTypeEnum(), rt -> new HashSet<>(256))
                        .add(res.getResId());
            }
        }

        // 检查菜单
        Set<String> menuCodesAll = new HashSet<>();
        if (CollUtil.isNotEmpty(resMap.get(MenuTreeNodeType.MENU))) {
            tenantDataIsolateProvider.byDefaultDirectly(() -> {
                menuCodesAll.addAll(menuRepoProc.getMenuCodes(true, true, true, true));
                return null;
            });
        }

        for (PermissionResBO permissionResBO : permissionResList) {
            if (CollUtil.isEmpty(permissionResBO.getResList())) {
                continue;
            }

            List<PermissionResBO.Res> resList = new ArrayList<>(permissionResBO.getResList().size());
            for (PermissionResBO.Res res : permissionResBO.getResList()) {
                if (res.getResTypeEnum() == MenuTreeNodeType.MENU) {
                    // 菜单
                    if (!menuCodesAll.contains(res.getResId())) {
                        continue;
                    }
                }

                resList.add(res);
            }
            permissionResBO.setResList(resList);
        }
    }

    private void savePermissionField(DataPermissionSaveVO saveVO) {
        List<SysRoleFieldPermissionDO> permissionDoList = null;
        switch (saveVO.getPermissionType()) {
            case MENU_OPERATION_RULE:
                // 菜单操作权限
                permissionDoList = this.convertFieldForMenuOperation(saveVO);
                // 删除现有
                fieldPermissionRepoProc.deleteByRoleForMenuOperation(saveVO.getRoleCode(), saveVO.getMenuCode(), saveVO.getOperationCode());
                // 保存新的
                if (!permissionDoList.isEmpty()) {
                    fieldPermissionRepoProc.save(permissionDoList);
                }
                break;
            case BUSINESS_OPERATION_RULE:
                // 业务对象的操作权限
                permissionDoList = this.convertFieldForBusinessOperation(saveVO);
                // 删除现有
                fieldPermissionRepoProc.deleteByRoleForBusinessOperation(saveVO.getRoleCode(), saveVO.getOperationCode());
                // 保存新的
                if (!permissionDoList.isEmpty()) {
                    fieldPermissionRepoProc.save(permissionDoList);
                }
                break;
            case BUSINESS_OBJECT_RULE:
                // 业务对象的权限
                permissionDoList = this.convertFieldForBusinessObject(saveVO);
                // 删除现有
                fieldPermissionRepoProc.deleteByRoleForBusinessObject(saveVO.getRoleCode(), saveVO.getAppCode(), saveVO.getBusinessObjectCode());
                // 保存新的
                if (!permissionDoList.isEmpty()) {
                    fieldPermissionRepoProc.save(permissionDoList);
                }
                break;
            case BUSINESS_RESOURCE_RULE:
                // 业务对象的权限
                permissionDoList = this.convertFieldForBusinessObject(saveVO);
                // 删除现有
                fieldPermissionRepoProc.deleteByRoleForBusinessObjectResource(saveVO.getRoleCode(), saveVO.getAppCode(), saveVO.getBusinessObjectCode());
                // 保存新的
                if (!permissionDoList.isEmpty()) {
                    fieldPermissionRepoProc.save(permissionDoList);
                }
                break;
            default:
                throw new BusinessException("暂不支持的权限类型");
        }
    }

    private void savePermissionRule(DataPermissionSaveVO saveVO) {
        List<SysRoleDataPermissionDO> permissionDoList = null;
        switch (saveVO.getPermissionType()) {
            case MENU_OPERATION_RULE:
                // 菜单操作权限
                permissionDoList = this.convertForMenuOperation(saveVO);
                // 删除现有
                dataPermissionRepoProc.deleteByRoleForMenuOperation(saveVO.getRoleCode(), saveVO.getMenuCode(), saveVO.getOperationCode());
                // 保存新的
                if (!permissionDoList.isEmpty()) {
                    dataPermissionRepoProc.save(permissionDoList);
                }
                break;
            case BUSINESS_OPERATION_RULE:
                // 业务对象的操作权限
                permissionDoList = this.convertForBusinessOperation(saveVO);
                // 删除现有
                dataPermissionRepoProc.deleteByRoleForBusinessOperation(saveVO.getRoleCode(), saveVO.getOperationCode());
                // 保存新的
                if (!permissionDoList.isEmpty()) {
                    dataPermissionRepoProc.save(permissionDoList);
                }
                break;
            case BUSINESS_OBJECT_RULE:
                // 业务对象的权限
                permissionDoList = this.convertForBusinessObject(saveVO);
                // 删除现有
                dataPermissionRepoProc.deleteByRoleForBusinessObject(saveVO.getRoleCode(), saveVO.getAppCode(), saveVO.getBusinessObjectCode());
                // 保存新的
                if (!permissionDoList.isEmpty()) {
                    dataPermissionRepoProc.save(permissionDoList);
                }
                break;
            case BUSINESS_RESOURCE_RULE:
                // 业务对象的权限
                permissionDoList = this.convertForBusinessObject(saveVO);
                // 删除现有
                dataPermissionRepoProc.deleteByRoleForRefResourceBusinessObject(saveVO.getRoleCode(), saveVO.getAppCode(), saveVO.getBusinessObjectCode());
                // 保存新的
                if (!permissionDoList.isEmpty()) {
                    dataPermissionRepoProc.save(permissionDoList);
                }
                break;
            default:
                throw new BusinessException("暂不支持的权限类型");
        }
    }

    private List<SysRoleDataPermissionDO> convertForMenuOperation(DataPermissionSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getCustomRuleList())) {
            return Collections.emptyList();
        }

        AtomicInteger order = new AtomicInteger(0);
        return saveVO.getCustomRuleList().stream().map(t -> {
            SysRoleDataPermissionDO permissionDO = new SysRoleDataPermissionDO();
            permissionDO.setRoleCode(saveVO.getRoleCode());
            permissionDO.setPermissionType(saveVO.getPermissionType().name());
            permissionDO.setAppCode(saveVO.getAppCode());
            permissionDO.setMenuCode(saveVO.getMenuCode());
            permissionDO.setBusinessObjectCode(saveVO.getBusinessObjectCode());
            permissionDO.setOperationCode(saveVO.getOperationCode());

            this.fillCustomRule(permissionDO, t, order.getAndIncrement());
            return permissionDO;
        }).collect(Collectors.toList());
    }

    private List<SysRoleFieldPermissionDO> convertFieldForMenuOperation(DataPermissionSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getFieldList())) {
            return Collections.emptyList();
        }

        return saveVO.getFieldList().stream()
                .filter(this::isAvailableField)
                .map(t -> {
                    SysRoleFieldPermissionDO permissionDO = new SysRoleFieldPermissionDO();
                    permissionDO.setRoleCode(saveVO.getRoleCode());
                    permissionDO.setPermissionType(saveVO.getPermissionType().name());
                    permissionDO.setAppCode(saveVO.getAppCode());
                    permissionDO.setMenuCode(saveVO.getMenuCode());
                    permissionDO.setBusinessObjectCode(saveVO.getBusinessObjectCode());
                    permissionDO.setOperationCode(saveVO.getOperationCode());

                    permissionDO.setFieldName(t.getFieldName());
                    permissionDO.setReadable(t.getReadable() == null || t.getReadable());
                    permissionDO.setWriteable(t.getWriteable() == null || t.getWriteable());

                    return permissionDO;
                }).collect(Collectors.toList());
    }

    private List<SysRoleDataPermissionDO> convertForBusinessOperation(DataPermissionSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getCustomRuleList())) {
            return Collections.emptyList();
        }


        AtomicInteger order = new AtomicInteger(0);
        return saveVO.getCustomRuleList().stream().map(t -> {
            SysRoleDataPermissionDO permissionDO = new SysRoleDataPermissionDO();
            permissionDO.setRoleCode(saveVO.getRoleCode());
            permissionDO.setPermissionType(saveVO.getPermissionType().name());
            permissionDO.setAppCode(saveVO.getAppCode());
            permissionDO.setMenuCode(null);
            permissionDO.setBusinessObjectCode(saveVO.getBusinessObjectCode());
            permissionDO.setOperationCode(saveVO.getOperationCode());

            this.fillCustomRule(permissionDO, t, order.getAndIncrement());
            return permissionDO;
        }).collect(Collectors.toList());
    }

    private List<SysRoleFieldPermissionDO> convertFieldForBusinessOperation(DataPermissionSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getFieldList())) {
            return Collections.emptyList();
        }


        return saveVO.getFieldList().stream()
                .filter(this::isAvailableField)
                .map(t -> {
                    SysRoleFieldPermissionDO permissionDO = new SysRoleFieldPermissionDO();
                    permissionDO.setRoleCode(saveVO.getRoleCode());
                    permissionDO.setPermissionType(saveVO.getPermissionType().name());
                    permissionDO.setAppCode(saveVO.getAppCode());
                    permissionDO.setMenuCode(null);
                    permissionDO.setBusinessObjectCode(saveVO.getBusinessObjectCode());
                    permissionDO.setOperationCode(saveVO.getOperationCode());

                    permissionDO.setFieldName(t.getFieldName());
                    permissionDO.setReadable(t.getReadable() == null || t.getReadable());
                    permissionDO.setWriteable(t.getWriteable() == null || t.getWriteable());

                    return permissionDO;
                }).collect(Collectors.toList());
    }

    private List<SysRoleDataPermissionDO> convertForBusinessObject(DataPermissionSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getCustomRuleList())) {
            return Collections.emptyList();
        }

        AtomicInteger order = new AtomicInteger(0);
        return saveVO.getCustomRuleList().stream().map(t -> {
            SysRoleDataPermissionDO permissionDO = new SysRoleDataPermissionDO();
            permissionDO.setRoleCode(saveVO.getRoleCode());
            permissionDO.setPermissionType(saveVO.getPermissionType().name());
            permissionDO.setAppCode(saveVO.getAppCode());
            permissionDO.setMenuCode(null);
            permissionDO.setBusinessObjectCode(saveVO.getBusinessObjectCode());
            permissionDO.setOperationCode(null);

            this.fillCustomRule(permissionDO, t, order.getAndIncrement());
            return permissionDO;
        }).collect(Collectors.toList());
    }

    private List<SysRoleFieldPermissionDO> convertFieldForBusinessObject(DataPermissionSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getFieldList())) {
            return Collections.emptyList();
        }

        return saveVO.getFieldList().stream()
                .filter(this::isAvailableField)
                .map(t -> {
                    SysRoleFieldPermissionDO permissionDO = new SysRoleFieldPermissionDO();
                    permissionDO.setRoleCode(saveVO.getRoleCode());
                    permissionDO.setPermissionType(saveVO.getPermissionType().name());
                    permissionDO.setAppCode(saveVO.getAppCode());
                    permissionDO.setMenuCode(null);
                    permissionDO.setBusinessObjectCode(saveVO.getBusinessObjectCode());
                    permissionDO.setOperationCode(null);

                    permissionDO.setFieldName(t.getFieldName());
                    permissionDO.setReadable(t.getReadable() == null || t.getReadable());
                    permissionDO.setWriteable(t.getWriteable() == null || t.getWriteable());

                    return permissionDO;
                }).collect(Collectors.toList());
    }

    private boolean isAvailableField(DataPermissionFieldSaveVO saveVO) {
        if (saveVO == null || CharSequenceUtil.isBlank(saveVO.getFieldName())) {
            return false;
        }

        if (Boolean.TRUE.equals(saveVO.getReadable()) && Boolean.TRUE.equals(saveVO.getWriteable())) {
            // 相当于默认的，无需保存
            return false;
        }

        return true;
    }

    private void fillCustomRule(SysRoleDataPermissionDO permissionDO, DataPermissionRuleSaveVO ruleSaveVO, int order) {
        permissionDO.setRuleGroupCode(ruleSaveVO.getRuleGroup());
        permissionDO.setRuleRelation(DprRuleRelationEnum.DPR_RULE_RELATION_AND.name());
        permissionDO.setRuleName(ruleSaveVO.getDprRuleName());
        permissionDO.setRuleOrder(order);
        permissionDO.setRuleDescription(ruleSaveVO.getDprRuleDeclare());
        permissionDO.setRuleField(ruleSaveVO.getDprRuleField());
        permissionDO.setRuleFieldType(ruleSaveVO.getRuleFieldType());
        permissionDO.setRefResource(ruleSaveVO.getRefResource());
        permissionDO.setRefBusinessObject(ruleSaveVO.getRefBusinessObject());
        permissionDO.setRefField(ruleSaveVO.getRefField());
        permissionDO.setFieldValueCondition(ruleSaveVO.getDprRuleCondition() == null ? null : ruleSaveVO.getDprRuleCondition().name());
        permissionDO.setRuleValueType(ruleSaveVO.getDprRuleValueType() == null ? null : ruleSaveVO.getDprRuleValueType().name());
        permissionDO.setRuleValue(ruleSaveVO.getDprRuleValue());
        permissionDO.setRuleValueName(ruleSaveVO.getDprRuleValueName());
        permissionDO.setDataSet(ruleSaveVO.getDataSet());
        permissionDO.setBs1(ruleSaveVO.getBs1());
        permissionDO.setBs2(ruleSaveVO.getBs2());
        permissionDO.setBs3(ruleSaveVO.getBs3());
    }

    private void check(DataPermissionSaveVO saveVO) {
        Assert.hasText(saveVO.getRoleCode(), "角色编码为空");
        Assert.notNull(saveVO.getPermissionType(), "权限类型为空");
        switch (saveVO.getPermissionType()) {
            case MENU_OPERATION_RULE:
                // 菜单操作权限
                Assert.hasText(saveVO.getMenuCode(), "菜单编码为空");
                Assert.hasText(saveVO.getOperationCode(), "操作编码为空");

                tenantDataIsolateProvider.byDefaultDirectly(() -> {
                    saveVO.setAppCode(menuRepoProc.getAppCodeByMenuCode(saveVO.getMenuCode()));
                    saveVO.setBusinessObjectCode(businessOperationRepoProc.getBusinessObjectCodeByOperationCode(saveVO.getOperationCode()));
                    return null;
                });
                break;
            case BUSINESS_OPERATION_RULE:
                // 业务对象操作
                Assert.hasText(saveVO.getOperationCode(), "操作编码为空");

                var operationInfo = tenantDataIsolateProvider.byDefaultDirectly(() -> businessOperationRepoProc.getSimpleBO(saveVO.getOperationCode()));
                Assert.notNull(operationInfo, "操作不存在");
                saveVO.setAppCode(operationInfo.getAppCode());
                saveVO.setBusinessObjectCode(operationInfo.getBusinessObjectCode());
                break;
            case BUSINESS_OBJECT_RULE:
            case BUSINESS_RESOURCE_RULE:
                Assert.hasText(saveVO.getBusinessObjectCode(), "业务对象编码为空");
                tenantDataIsolateProvider.byDefaultDirectly(() -> {
                    saveVO.setAppCode(businessObjectRepoProc.getAppCode(saveVO.getBusinessObjectCode()));
                    return null;
                });
                Assert.hasText(saveVO.getAppCode(), "未知业务对象所属应用");
                break;
            default:
                throw new BusinessException("暂不支持的权限类型");
        }
        // 规则条件
        if (CollUtil.isNotEmpty(saveVO.getCustomRuleList())) {
            Map<String, BusinessParamBO> businessParamMap = StringUtils.hasText(saveVO.getBusinessObjectCode()) ? tenantDataIsolateProvider.byDefaultDirectly(() -> businessParamRepoProc.listSimpleBoByBusinessObjectCode(saveVO.getBusinessObjectCode()))
                    .stream().collect(Collectors.toMap(BusinessParamBO::getFieldName, Functions.identity(), (t1, t2) -> t1)) : Collections.emptyMap();
            for (var ruleSaveVO : saveVO.getCustomRuleList()) {
                Assert.hasText(ruleSaveVO.getDprRuleField(), "存在规则的字段为空");
                Assert.notNull(ruleSaveVO.getDprRuleValueType(), "存在规则的值类型为空");
                var businessParam = businessParamMap.get(ruleSaveVO.getDprRuleField());
                Assert.notNull(businessParam, "参数" + ruleSaveVO.getDprRuleField() + "不存在");
                ruleSaveVO.setRuleFieldType(businessParam.getFieldJavaType());

                if (Boolean.TRUE.equals(ruleSaveVO.getRefResource())) {
                    Assert.hasText(ruleSaveVO.getRefBusinessObject(), "关联的资源业务对象为空");
                    ruleSaveVO.setRefField(businessParam.getRelatedField());
                    Assert.hasText(ruleSaveVO.getRefField(), "未知关联的业务对象的字段：" + ruleSaveVO.getRefBusinessObject() + "-" + ruleSaveVO.getDprRuleField());
                } else {
                    Assert.notNull(ruleSaveVO.getDprRuleCondition(), "存在规则的条件为空");
                    Assert.hasText(ruleSaveVO.getDprRuleValue(), "规则值不能为空");
                }
                if (ruleSaveVO.getDprRuleCondition() == null) {
                    ruleSaveVO.setDprRuleCondition(DprRuleConditionEnum.InList);
                }
            }
        }
        // 权限字段
        if (CollUtil.isNotEmpty(saveVO.getFieldList())) {
            for (DataPermissionFieldSaveVO fieldSaveVO : saveVO.getFieldList()) {
                Assert.hasText(fieldSaveVO.getFieldName(), "字段权限配置中存在字段名称为空");
            }
        }

        // 关联操作检查
        AtomicBoolean supportUseDef = new AtomicBoolean(false);
        if (Boolean.TRUE.equals(saveVO.getUseRef())) {
            if (StringUtils.hasText(saveVO.getOperationCode())) {
                tenantDataIsolateProvider.byDefaultDirectly(() -> {
                    var permissionRef = businessOperationRepoProc.getPermissionRefByOperationCode(saveVO.getOperationCode());
                    if (StringUtils.hasText(permissionRef)) {
                        saveVO.setCustomRuleList(Collections.emptyList());
                        saveVO.setFieldList(Collections.emptyList());
                        supportUseDef.set(true);
                    } else {
                        throw new IllegalArgumentException("当前业务对象操作无关联的操作，引用权限规则失败");
                    }
                    return null;
                });
            }
        }
        saveVO.setUseRef(supportUseDef.get());
    }

    private List<DataPermissionTreeNodeRespVO> queryDataPermissionTreeOfBusinessObject(String roleCode, String appCode,
                                                                                       boolean includeApp, boolean publicResource) {
        List<DataPermissionTreeNodeRespVO> respVoList = new ArrayList<>(128);
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            // 获取应用列表
            var tenantApps = super.tenantApps(appCode);
            if (tenantApps.isEmpty()) {
                return Collections.emptyList();
            }

            // 查询业务对象
            var objectList = businessObjectRepoProc.listSimpleByAppCodes(tenantApps.keySet(), true, publicResource);
            if (objectList.isEmpty()) {
                return Collections.emptyList();
            }
            int i = 0;
            DataPermissionTreeNodeRespVO respVO = null;
            Set<String> existsAppCodes = new HashSet<>(objectList.size());
            for (BusinessObjectBO businessObjectSimpleBO : objectList) {
                respVO = this.convertDataPermissionTreeNode(businessObjectSimpleBO, includeApp);
                respVO.setSortNo(i++);
                respVoList.add(respVO);

                existsAppCodes.add(businessObjectSimpleBO.getAppCode());
            }

            if (includeApp) {
                i = 0;
                for (var entry : tenantApps.entrySet()) {
                    if (!existsAppCodes.contains(entry.getKey())) {
                        continue;
                    }
                    respVO = this.convertDataPermissionTreeNode(entry.getValue());
                    respVO.setSortNo(i++);
                    respVoList.add(respVO);
                }
            }

            return null;
        });
        if (respVoList.isEmpty()) {
            return Collections.emptyList();
        }

        // 分配的权限规则
        DataPermissionType permissionType = publicResource ? DataPermissionType.BUSINESS_RESOURCE_RULE : DataPermissionType.BUSINESS_OBJECT_RULE;
        List<RolePermissionRuleBO> roleDataPermissionRuleList = CharSequenceUtil.isBlank(roleCode) ? Collections.emptyList()
                : dataPermissionRepoProc.listRolePermissionRuleByRole(roleCode, permissionType);
        if (roleDataPermissionRuleList.isEmpty()) {
            return respVoList;
        }
        var ruleCount = roleDataPermissionRuleList.stream().collect(Collectors.groupingBy(t -> t.getBusinessObjectCode(), Collectors.counting()));
        for (DataPermissionTreeNodeRespVO respVO : respVoList) {
            if (MenuTreeNodeType.BUSINESS_OBJECT.getValue().equals(respVO.getNodeType())) {
                respVO.setRuleNum(ruleCount.getOrDefault(respVO.getCode(), 0L));
            }
        }

        return respVoList;
    }

    private List<DataPermissionTreeNodeRespVO> queryDataPermissionTreeOfBusinessOperation(String roleCode, String appCode,
                                                                                          boolean withRef, boolean includeApp) {
        List<DataPermissionTreeNodeRespVO> respVoList = new ArrayList<>(128);
        Map<String, List<BusinessRefBO>> refMap = new HashMap<>(8);
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            // 获取应用列表
            var tenantApps = super.tenantApps(appCode);
            if (tenantApps.isEmpty()) {
                return Collections.emptyList();
            }

            // 查询业务对象
            var objectList = businessObjectRepoProc.listSimpleByAppCodes(tenantApps.keySet(), true, null);
            if (objectList.isEmpty()) {
                return Collections.emptyList();
            }
            int i = 0;
            DataPermissionTreeNodeRespVO respVO = null;
            Set<Long> existsBusinessIds = new HashSet<>(objectList.size());
            Set<String> existsAppCodes = new HashSet<>(objectList.size());
            for (BusinessObjectBO businessObjectSimpleBO : objectList) {
                respVO = this.convertDataPermissionTreeNode(businessObjectSimpleBO, includeApp);
                respVO.setSortNo(i++);
                respVoList.add(respVO);

                existsBusinessIds.add(businessObjectSimpleBO.getId());
                existsAppCodes.add(businessObjectSimpleBO.getAppCode());
            }

            if (includeApp) {
                i = 0;
                for (var entry : tenantApps.entrySet()) {
                    if (!existsAppCodes.contains(entry.getKey())) {
                        continue;
                    }
                    respVO = this.convertDataPermissionTreeNode(entry.getValue());
                    respVO.setSortNo(i++);
                    respVoList.add(respVO);
                }
            }

            // 查询业务对象的操作
            AtomicInteger sortNo = new AtomicInteger(0);
            businessOperationRepoProc.listSimpleByAppCode(existsAppCodes, true, true).stream()
                    .filter(t -> existsBusinessIds.contains(t.getBusinessObjectId()))
                    .forEach(t -> {
                        var vo = convertDataPermissionTreeNode(t);
                        vo.setSortNo(sortNo.getAndIncrement());

                        respVoList.add(vo);
                    });

            // 关联
            if (withRef) {
                refMap.putAll(refRepoProc.listSimpleBo().stream().collect(Collectors.groupingBy(BusinessRefBO::getBusinessObjectCode)));
            }

            return null;
        });
        if (respVoList.isEmpty()) {
            return Collections.emptyList();
        }

        // 处理挂载的
        this.dealRefNode(respVoList, refMap);

        // 分配的权限规则
        List<RolePermissionRuleBO> roleDataPermissionRuleList = CharSequenceUtil.isBlank(roleCode) ? Collections.emptyList()
                : dataPermissionRepoProc.listRolePermissionRuleByRole(roleCode, DataPermissionType.BUSINESS_OPERATION_RULE);
        if (roleDataPermissionRuleList.isEmpty()) {
            return respVoList;
        }
        var ruleCount = roleDataPermissionRuleList.stream().collect(Collectors.groupingBy(RolePermissionRuleBO::getOperationCode, Collectors.counting()));
        for (DataPermissionTreeNodeRespVO respVO : respVoList) {
            if (MenuTreeNodeType.BUSINESS_OPERATION.getValue().equals(respVO.getNodeType())) {
                if (StringUtils.hasText(respVO.getRefPermission())) {
                    respVO.setRuleNum(ruleCount.getOrDefault(respVO.getRefPermission(), 0L));
                    continue;
                }
                respVO.setRuleNum(ruleCount.getOrDefault(respVO.getCode(), 0L));
            }
        }

        return respVoList;
    }

    private void dealRefNode(List<DataPermissionTreeNodeRespVO> ruleList, Map<String, List<BusinessRefBO>> refMap) {
        if (ruleList.isEmpty() || refMap.isEmpty()) {
            return;
        }

        var operationMap = ruleList.stream()
                .filter(t -> BusinessObjectNodeType.BUSINESS_OPERATION.name().equals(t.getNodeType()))
                .collect(Collectors.toMap(DataPermissionTreeNodeRespVO::getCode, Function.identity(), (t1, t2) -> t1));

        List<DataPermissionTreeNodeRespVO> newNodeList = new ArrayList<>();
        for (DataPermissionTreeNodeRespVO respVo : ruleList) {
            if (!BusinessObjectNodeType.BUSINESS_OBJECT.name().equals(respVo.getNodeType())) {
                continue;
            }

            var refList = refMap.getOrDefault(respVo.getCode().split(":")[1], Collections.emptyList());
            if (refList.isEmpty()) {
                continue;
            }
            AtomicInteger sortNo = new AtomicInteger(0);
            var refNodeList = refList.stream()
                    .filter(t -> BusinessObjectRefTypeEnum.BUSINESS_OPERATION.name().equals(t.getRefType()) && operationMap.containsKey(t.getRef()))
                    .map(t -> {
                        var operation = operationMap.get(t.getRef());
                        DataPermissionTreeNodeRespVO refVo = new DataPermissionTreeNodeRespVO();
                        refVo.setRoute(operation.getRoute());
                        refVo.setNodeType(operation.getNodeType());
                        refVo.setNodeTypeName(operation.getNodeTypeName());
                        refVo.setRefNode(true);
                        refVo.setRuleNum(operation.getRuleNum());
                        refVo.setIcon(operation.getIcon());
                        refVo.setApiDetail(operation.getApiDetail());
                        refVo.setOperationDetail(operation.getOperationDetail());
                        refVo.setId(operation.getId());
                        refVo.setCode(operation.getCode());
                        refVo.setName(operation.getName());
                        refVo.setSortNo(sortNo.getAndIncrement());
                        refVo.setParentId(respVo.getId());
                        refVo.setParentCode(respVo.getCode());
                        refVo.setChildren(respVo.getChildren());

                        return refVo;
                    }).collect(Collectors.toList());
            if (refNodeList.isEmpty()) {
                continue;
            }
            newNodeList.addAll(refNodeList);
        }
        if (newNodeList.isEmpty()) {
            return;
        }
        ruleList.addAll(newNodeList);
    }

    private List<DataPermissionTreeNodeRespVO> queryDataPermissionTreeOfMenuOperation(String roleCode, String appCode,
                                                                                      Long tenantId, boolean includeApp) {
        Map<String, AppBO> tenantApp = new LinkedHashMap<>();
        Map<String, List<MenuOperationRespVO>> menuOperationMap = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            // 获取应用列表
            var tenantApps = super.tenantApps(appCode);
            if (tenantApps.isEmpty()) {
                return Collections.emptyMap();
            }
            tenantApp.putAll(tenantApps);

            // 查询菜单的操作
            return menuOperationRepoProc.listOperationOfMenu(tenantApps.keySet());
        });
        if (menuOperationMap.isEmpty()) {
            return Collections.emptyList();
        }

        // 菜单
//        var custom = this.hasCustomMenuTree(tenantId);
        var custom = false;
        List<RolePermissionRespVO> respVOList = this.queryAppAndMenu(tenantApp, custom, includeApp);
        if (respVOList.isEmpty()) {
            return Collections.emptyList();
        }

        // 分配的权限规则
        List<RolePermissionRuleBO> roleDataPermissionRuleList = CharSequenceUtil.isBlank(roleCode) ? Collections.emptyList()
                : dataPermissionRepoProc.listRolePermissionRuleByRole(roleCode, DataPermissionType.MENU_OPERATION_RULE);
        var treeNodeList = respVOList.stream().map(this::convertDataPermissionTreeNode).collect(Collectors.toList());
        treeNodeList.addAll(this.convertDataPermissionTreeNodeForMenuOperation(menuOperationMap, roleDataPermissionRuleList));

        return treeNodeList;
    }

    private List<DataPermissionTreeNodeRespVO> convertDataPermissionTreeNode(Map<String, List<SysMenuApiRespVO>> menuApiMap,
                                                                             List<RoleApiRuleBO> roleApiRuleList) {
        var roleApiMap = roleApiRuleList.stream().collect(Collectors.groupingBy(t -> t.getMenuCode() + ":" + t.getApiCode(), Collectors.counting()));
        List<DataPermissionTreeNodeRespVO> respVoList = new ArrayList<>();
        DataPermissionTreeNodeRespVO respVO = null;
        for (var entry : menuApiMap.entrySet()) {
            int i = 0;
            var menuCode = entry.getKey();
            for (var param : entry.getValue()) {
                respVO = new DataPermissionTreeNodeRespVO();
                respVO.setNodeType(MenuTreeNodeType.API.getValue());
                respVO.setRuleNum(roleApiMap.getOrDefault(menuCode + ":" + param.getCode(), 0L));
                respVO.setCode(param.getCode());
                respVO.setName(param.getName());
                respVO.setSortNo(i++);
                respVO.setParentCode(menuCode);
                respVO.setHasChildren(false);
                respVO.setChildren(Collections.emptyList());
                // 接口详情
                respVO.setApiDetail(param);

                respVoList.add(respVO);
            }
        }
        return respVoList;
    }

    private List<DataPermissionTreeNodeRespVO> convertDataPermissionTreeNodeForMenuOperation(Map<String, List<MenuOperationRespVO>> menuApiMap,
                                                                                             List<RolePermissionRuleBO> roleApiRuleList) {
        var roleApiMap = roleApiRuleList.stream().collect(Collectors.groupingBy(t -> t.getMenuCode() + ":" + t.getOperationCode(), Collectors.counting()));
        List<DataPermissionTreeNodeRespVO> respVoList = new ArrayList<>();
        DataPermissionTreeNodeRespVO respVO = null;
        for (var entry : menuApiMap.entrySet()) {
            int i = 0;
            var menuCode = entry.getKey();
            for (var param : entry.getValue()) {
                respVO = new DataPermissionTreeNodeRespVO();
                respVO.setId(param.getId());
                respVO.setNodeType(MenuTreeNodeType.BUSINESS_OPERATION.getValue());
                respVO.setNodeTypeName(MenuTreeNodeType.BUSINESS_OPERATION.getDescription());
                if (StringUtils.hasText(param.getPermissionRef())) {
                    respVO.setRuleNum(roleApiMap.getOrDefault(menuCode + ":" + param.getPermissionRef(), 0L));
                } else {
                    respVO.setRuleNum(roleApiMap.getOrDefault(menuCode + ":" + param.getOperationCode(), 0L));
                }
                respVO.setCode(param.getOperationCode());
                respVO.setName(param.getOperationName());
                respVO.setSortNo(i++);
                respVO.setParentCode(menuCode);
                respVO.setHasChildren(false);
                respVO.setChildren(Collections.emptyList());
                // 操作详情
                respVO.setOperationDetail(param);
                if (StringUtils.hasText(param.getPermissionRef())) {
                    respVO.setRefPermission(param.getPermissionRef());
                    respVO.setRefPermissionName(param.getPermissionRefName());
                }

                respVoList.add(respVO);
            }
        }
        return respVoList;
    }

    private DataPermissionTreeNodeRespVO convertDataPermissionTreeNode(AppBO app) {
        DataPermissionTreeNodeRespVO respVO = new DataPermissionTreeNodeRespVO();
        respVO.setNodeType(MenuTreeNodeType.APP.getValue());
        respVO.setNodeTypeName(MenuTreeNodeType.APP.getDescription());
        respVO.setRefNode(false);
        respVO.setIcon(app.getIcon());
        respVO.setId(app.getId());
        respVO.setCode(app.getAppCode());
        respVO.setName(app.getAppName());
        respVO.setSortNo(0);
        respVO.setHasChildren(false);
        respVO.setChildren(Collections.emptyList());

        return respVO;
    }

    private DataPermissionTreeNodeRespVO convertDataPermissionTreeNode(BusinessOperationBO simpleBO) {
        DataPermissionTreeNodeRespVO respVO = new DataPermissionTreeNodeRespVO();
        respVO.setNodeType(MenuTreeNodeType.BUSINESS_OPERATION.getValue());
        respVO.setNodeTypeName(MenuTreeNodeType.BUSINESS_OPERATION.getDescription());
        respVO.setRefNode(false);
        if (StringUtils.hasText(simpleBO.getPermissionRef())) {
            respVO.setRefPermission(simpleBO.getPermissionRef());
            respVO.setRefPermissionName(simpleBO.getPermissionRefName());
        }
        respVO.setRuleNum(0L);
        respVO.setId(simpleBO.getId());
        respVO.setCode(simpleBO.getOperationCode());
        respVO.setName(CharSequenceUtil.blankToDefault(simpleBO.getCustomName(), simpleBO.getOperationDescription()));
        respVO.setSortNo(0);
        respVO.setParentCode(simpleBO.getBusinessObjectCode());
        respVO.setHasChildren(false);
        respVO.setChildren(new ArrayList<>(64));
        respVO.setBusinessObjectCode(simpleBO.getBusinessObjectCode());

        MenuOperationRespVO operationDetail = new MenuOperationRespVO();
        operationDetail.setId(simpleBO.getId());
        operationDetail.setAppCode(simpleBO.getAppCode());
        operationDetail.setAppName(simpleBO.getAppName());
        operationDetail.setBusinessObjectCode(simpleBO.getBusinessObjectCode());
        operationDetail.setBusinessObjectName(simpleBO.getBusinessObjectName());
        operationDetail.setOperationCode(simpleBO.getOperationCode());
        operationDetail.setOperationName(CharSequenceUtil.blankToDefault(simpleBO.getCustomName(), simpleBO.getApiName()));
        operationDetail.setApiMethod(simpleBO.getApiMethod());
        operationDetail.setApiUrl(simpleBO.getApiUrl());
        operationDetail.setPermissionRef(simpleBO.getPermissionRef());
        operationDetail.setPermissionRefName(simpleBO.getPermissionRefName());
        respVO.setOperationDetail(operationDetail);

        return respVO;
    }

    private DataPermissionTreeNodeRespVO convertDataPermissionTreeNode(BusinessObjectBO simpleBO, boolean includeApp) {
        DataPermissionTreeNodeRespVO respVO = new DataPermissionTreeNodeRespVO();
        respVO.setNodeType(MenuTreeNodeType.BUSINESS_OBJECT.getValue());
        respVO.setNodeTypeName(MenuTreeNodeType.BUSINESS_OBJECT.getDescription());
        respVO.setRefNode(false);
        respVO.setRuleNum(0L);
        respVO.setId(simpleBO.getId());
        respVO.setCode(simpleBO.getCode());
        respVO.setName(CharSequenceUtil.blankToDefault(simpleBO.getCustomName(), simpleBO.getName()));
        respVO.setSortNo(0);
        if (includeApp) {
            respVO.setParentCode(simpleBO.getAppCode());
        }
        respVO.setHasChildren(false);
        respVO.setChildren(new ArrayList<>(64));
        respVO.setBusinessObjectCode(simpleBO.getCode());

        return respVO;
    }

    private DataPermissionTreeNodeRespVO convertDataPermissionTreeNode(RolePermissionRespVO rolePermissionRespVO) {
        DataPermissionTreeNodeRespVO treeNodeRespVO = new DataPermissionTreeNodeRespVO();
        treeNodeRespVO.setId(rolePermissionRespVO.getId());
        treeNodeRespVO.setRoute(rolePermissionRespVO.getRoute());
        treeNodeRespVO.setNodeType(rolePermissionRespVO.getNodeType());
        treeNodeRespVO.setNodeTypeName(MenuTreeNodeType.valueOf(rolePermissionRespVO.getNodeType()).getDescription());
        treeNodeRespVO.setIcon(rolePermissionRespVO.getMenusIcon());
        treeNodeRespVO.setRuleNum(0L);
        treeNodeRespVO.setCode(rolePermissionRespVO.getCode());
        treeNodeRespVO.setName(rolePermissionRespVO.getName());
        treeNodeRespVO.setSortNo(ObjectUtil.defaultIfNull(rolePermissionRespVO.getSortNo(), 0));
        treeNodeRespVO.setParentId(null);
        treeNodeRespVO.setParentCode(rolePermissionRespVO.getParentCode());
        treeNodeRespVO.setHasChildren(false);
        treeNodeRespVO.setChildren(new ArrayList<>(0));

        return treeNodeRespVO;
    }

    private void cloneRoleDataPermission(@NotNull Long originalId, @NotNull Long roleId) {
        // 行权限
        var roleDataPermissionList = roleApiRowRuleRepoProc.listByRole(originalId).stream()
                .map(t -> {
                    var ruleDO = SysDprRoleApiRuleConvert.INSTANCE.do2Do(t);
                    ruleDO.setId(null);
                    ruleDO.setRoleId(roleId);
                    return ruleDO;
                }).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(roleDataPermissionList)) {
            roleApiRowRuleRepoProc.save(roleDataPermissionList);
        }

        // 列权限
        var roleColumnPermissionList = apiFieldsRepoProc.listByRole(originalId).stream()
                .map(t -> {
                    var permissionDO = SysDpcRoleApiFieldsConvert.INSTANCE.do2Do(t);
                    permissionDO.setId(null);
                    permissionDO.setRoleId(roleId);
                    return permissionDO;
                }).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(roleColumnPermissionList)) {
            apiFieldsRepoProc.save(roleColumnPermissionList);
        }
    }

    private void cloneRoleMenu(long originalId, long roleId) {
        var roleCode = roleRepoProc.getCode(roleId);
        var rolePermissionList = rolePermissionRepoProc.listByRole(originalId).stream()
                .map(t -> {
                    var permissionDO = RoleConvert.INSTANCE.do2do(t);
                    permissionDO.setId(null);
                    permissionDO.setRoleId(roleId);
                    permissionDO.setRoleCode(roleCode);
                    return permissionDO;
                }).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(rolePermissionList)) {
            rolePermissionRepoProc.save(rolePermissionList);
        }
    }

    private Long currentTenantId() {
        var currentTenant = tenantClientProvider.getCurrentTenant();
        return currentTenant == null ? TenantConstant.DEFAULT_TENANT_ID : currentTenant.getId();
    }

    private boolean hasCustomMenuTree(Long tenantId) {
        if (tenantId == null) {
            GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
            tenantId = ObjectUtil.defaultIfNull(currentUser.getTenantId(), TenantConstant.DEFAULT_TENANT_ID);
        }

        var custom = tenantMenuRepoProc.getEnabledByTenant(tenantId);
        return custom != null && custom;
    }

    private Set<String> getApps() {
        var currentTenant = tenantClientProvider.getCurrentTenant();
        if (currentTenant != null) {
            return currentTenant.getAppCodes();
        }
        return queryAppAll(true).stream().map(IdCodeNameParam::getCode).collect(Collectors.toSet());
    }

    private List<IdCodeNameParam> queryAppAll(Boolean enabled) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.allParams(enabled));
    }

    private SysDprRoleApiRowRuleDO convertDprRoleApiRowRuleDO(String appCode, SysDprSaveBO saveBO, SysDprApiCustomRuleSaveBO ruleBO,
                                                              int ruleOrder, DprRuleRelationEnum relation) {
        SysDprRoleApiRowRuleDO ruleDO = new SysDprRoleApiRowRuleDO();
        ruleDO.setRoleId(saveBO.getRoleId());
        ruleDO.setAppCode(appCode);
        ruleDO.setMenuCode(saveBO.getMenuCode());
        ruleDO.setApiCode(saveBO.getApiCode());
        ruleDO.setRuleOrder(ruleOrder);
        ruleDO.setDprRuleRelation(relation.name());
        ruleDO.setDprRuleRelationName(relation.getValueDescription());
        ruleDO.setDprRuleName(ruleBO.getDprRuleName());
        ruleDO.setDprRuleDeclare(ruleBO.getDprRuleDeclare());
        ruleDO.setDprRuleField(ruleBO.getDprRuleField());
        if (ruleBO.getDprRuleCondition() != null) {
            ruleDO.setDprRuleCondition(ruleBO.getDprRuleCondition().name());
            ruleDO.setDprRuleConditionName(ruleBO.getDprRuleCondition().getValueDescription());
        }

        var valueType = ruleBO.getDprRuleValueType();
        Assert.notNull(valueType, "取值方式为空");
        if (DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_CUSTOM != valueType
                && DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS != valueType
                && DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS != valueType) {
            throw new IllegalArgumentException("暂不支持的取值方式：" + valueType.getValueDescription());
        }
        ruleDO.setDprRuleValueType(valueType.name());
        ruleDO.setDprRuleValueTypeName(valueType.getValueDescription());
        ruleDO.setDprRuleValue(ruleBO.getDprRuleValue());
        ruleDO.setDprRuleValueName(ruleBO.getDprRuleValueName());
        ruleDO.setRoleRuleValueType(valueType.name());
        ruleDO.setDataSet(ruleBO.getDataSet());
        ruleDO.setRoleRuleValue(ruleBO.getDprRuleValue());
        if (DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS == valueType) {
            ruleDO.setDprSysInternally(ruleBO.getDprRuleValue());
            ruleDO.setDprSysInternallyName(DprSysInternallyEnum.valueOf(ruleBO.getDprRuleValue()).getValueDescription());
        }
        ruleDO.setDataRange(false);
        ruleDO.setBs1(ruleBO.getBs1());
        ruleDO.setBs2(ruleBO.getBs2());
        ruleDO.setBs3(ruleBO.getBs3());
        return ruleDO;
    }

    private void saveDataPermissionCustomRule(Long appId, SysDprSaveBO saveBO, DprRuleRelationEnum relation, Set<String> apiCodesAll) {
        String appCode = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.getCode(appId));
        List<SysDprRoleApiRowRuleDO> rowRuleDOList = new ArrayList<>(saveBO.getCustomRuleList().size());
        // 数据范围
        int order = 0;
        SysDprRoleApiRowRuleDO ruleDO = null;
        if (saveBO.getRange() != null) {
            ruleDO = new SysDprRoleApiRowRuleDO();
            ruleDO.setRoleId(saveBO.getRoleId());
            ruleDO.setAppCode(appCode);
            ruleDO.setMenuCode(saveBO.getMenuCode());
            ruleDO.setApiCode(saveBO.getApiCode());
            ruleDO.setRuleOrder(order++);
            ruleDO.setDprRuleRelation(relation.name());
            ruleDO.setDprRuleRelationName(relation.getValueDescription());
            ruleDO.setDprRuleName("数据范围");
            ruleDO.setDprRuleValueType(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name());
            ruleDO.setDprRuleValueTypeName(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.getValueDescription());
            ruleDO.setDprSysInternally(saveBO.getRange().name());
            ruleDO.setDprSysInternallyName(saveBO.getRange().getValueDescription());
            ruleDO.setDataRange(true);
            ruleDO.setBs1(null);
            ruleDO.setBs2(null);
            ruleDO.setBs3(null);
            rowRuleDOList.add(ruleDO);
        }

        // 自定义规则
        for (SysDprApiCustomRuleSaveBO bo : saveBO.getCustomRuleList()) {
            ruleDO = this.convertDprRoleApiRowRuleDO(appCode, saveBO, bo, order++, relation);

            rowRuleDOList.add(ruleDO);
        }
        // 先删除
        roleApiRowRuleRepoProc.deleteByApi(saveBO.getRoleId(), saveBO.getMenuCode(), saveBO.getApiCode());
        // 删除菜单下已不存在的api
        roleApiRowRuleRepoProc.deleteForNotExistsApi(saveBO.getMenuCode(), apiCodesAll);
        // 再新增
        if (!rowRuleDOList.isEmpty()) {
            roleApiRowRuleRepoProc.save(rowRuleDOList);
        }
    }

    private void saveDataPermissionField(Long appId, SysDprSaveBO saveBO) {
        String appCode = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.getCode(appId));
        List<SysDpcrApiFieldsDO> fieldDoList = new ArrayList<>(saveBO.getFieldList().size());
        SysDpcrApiFieldsDO fieldsDO = null;
        for (SysDprApiFieldSaveBO bo : saveBO.getFieldList()) {
            fieldsDO = new SysDpcrApiFieldsDO();
            fieldsDO.setRoleId(saveBO.getRoleId());
            fieldsDO.setAppCode(appCode);
            fieldsDO.setMenuCode(saveBO.getMenuCode());
            fieldsDO.setApiCode(saveBO.getApiCode());
            fieldsDO.setFieldName(bo.getFieldName());
            fieldsDO.setFieldRemark(bo.getFieldRemark());
            fieldsDO.setFieldApiVisible(ObjectUtil.defaultIfNull(bo.isFieldApiVisible(), false));
            fieldsDO.setFieldFormVisible(ObjectUtil.defaultIfNull(bo.isFieldFormVisible(), false));
            fieldsDO.setFieldFormUpdate(ObjectUtil.defaultIfNull(bo.isFieldFormUpdate(), false));

            fieldDoList.add(fieldsDO);
        }
        // 先删除
        apiFieldsRepoProc.deleteByApi(saveBO.getRoleId(), saveBO.getMenuCode(), saveBO.getApiCode());
        if (!fieldDoList.isEmpty()) {
            // 再保存
            apiFieldsRepoProc.save(fieldDoList);
        }
    }

    private List<RolePermissionRespVO> queryAppAndMenu(Map<String, AppBO> appMap, boolean custom, boolean includeApp) {
        // 查询自定义菜单树
        if (custom) {
            return queryCustomMenuTree(appMap);
        }

        // 查询默认的
        return queryDefaultMenuTree(includeApp, appMap);
    }

    private <T> List<T> filterChildrenForTree(List<T> respVoList, Function<T, List<T>> childrenGetter, BiConsumer<T, List<T>> childrenSetter,
                                              Predicate<T> predicate) {
        if (CollUtil.isEmpty(respVoList)) {
            return Collections.emptyList();
        }

        List<T> dataList = new ArrayList<>();
        for (T t : respVoList) {
            var result = predicate.test(t);
            if (result) {
                dataList.add(t);
            }

            // 判断子节点是否有满足的
            var children = filterChildrenForTree(childrenGetter.apply(t), childrenGetter, childrenSetter, predicate);
            childrenSetter.accept(t, children);
            if (!result && CollUtil.isNotEmpty(children)) {
                dataList.add(t);
            }
        }
        return dataList;
    }

    @SuppressWarnings("unchecked")
    private List<DataPermissionTreeNodeRespVO> convertDataPermissionTreeRespVO(List<DataPermissionTreeNodeRespVO> respVoList,
                                                                               boolean tree, MenuTreeNodeType lastNodeType) {
        TreeDataUtil<DataPermissionTreeNodeRespVO> treeDataUtil = new TreeDataUtil<>(respVoList, DataPermissionTreeNodeRespVO::getCode,
                DataPermissionTreeNodeRespVO::getParentCode, DataPermissionTreeNodeRespVO::setChildren, Comparator.comparingInt(DataPermissionTreeNodeRespVO::getSortNo));
        respVoList = (List<DataPermissionTreeNodeRespVO>) treeDataUtil.getRoots();

        // 过滤出出有目标权限资源类型的
        if (lastNodeType != null) {
            respVoList = filterChildrenForTree(respVoList, DataPermissionTreeNodeRespVO::getChildren, DataPermissionTreeNodeRespVO::setChildren,
                    t -> lastNodeType.getValue().equals(t.getNodeType()));
        }

        return tree ? respVoList : CollectionUtil.expandTree(respVoList, DataPermissionTreeNodeRespVO::getChildren);
    }

    @SuppressWarnings("unchecked")
    private List<RolePermissionRespVO> convertPermissionTreeRespVO(List<RolePermissionRespVO> respVOList, List<String> checked, boolean tree) {
        // 设置已选择项
        for (RolePermissionRespVO respVO : respVOList) {
            if (checked.contains(respVO.getCode())) {
                respVO.setChecked(true);
            }

            if (respVO.getSortNo() == null) {
                respVO.setSortNo(1);
            }
        }

        if (!tree) {
            return respVOList;
        }

        TreeDataUtil<RolePermissionRespVO> treeDataUtil = new TreeDataUtil<>(respVOList, RolePermissionRespVO::getCode,
                RolePermissionRespVO::getParentCode, RolePermissionRespVO::setChildren, Comparator.comparingInt(RolePermissionRespVO::getSortNo));

        return (List<RolePermissionRespVO>) treeDataUtil.getRoots();
    }

    private List<RolePermissionRespVO> queryActionList(Set<String> appCodes) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryActionByAppCode(appCodes, true)).stream()
                .filter(t -> (t.getMenusState() == null || t.getMenusState()) && StringUtils.hasText(t.getMenusParentCode()))
                .map(t -> {
                    RolePermissionRespVO respVO = new RolePermissionRespVO();
                    respVO.setCode(t.getMenusCode());
                    respVO.setName(t.getMenusName());
                    respVO.setNodeType(MenuTreeNodeType.ACTION.getValue());
                    respVO.setParentCode(t.getMenusParentCode());
                    respVO.setMenusIcon(t.getMenusIcon());
                    respVO.setRoute(t.getMenusRoute());
                    respVO.setChecked(false);
                    respVO.setSortNo(t.getMenusOrder());
                    respVO.setHasChildren(false);
                    respVO.setChildren(new ArrayList<>(8));

                    return respVO;
                }).collect(Collectors.toList());
    }

    private List<RolePermissionRespVO> queryApiList(Set<String> appCodes) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> apiRepoProc.queryByAppCode(appCodes)).stream()
                .map(t -> {
                    RolePermissionRespVO respVO = new RolePermissionRespVO();
                    respVO.setCode(t.getPermissonCode());
                    respVO.setName(t.getPermissonName());
                    respVO.setNodeType(MenuTreeNodeType.API.getValue());
                    respVO.setParentCode(t.getMenusCode());
                    respVO.setChecked(false);
                    respVO.setHasChildren(false);
                    respVO.setChildren(new ArrayList<>(8));

                    return respVO;
                }).collect(Collectors.toList());
    }

    private List<RolePermissionRespVO> queryCustomMenuTree(Map<String, AppBO> appMap) {
        // 查询菜单设置
        var menuTreeDOList = tenantMenuTreeRepoProc.queryByTenantId();
        if (menuTreeDOList.isEmpty()) {
            // 尚未设置
            return Collections.emptyList();
        }

        // 查询默认菜单
        var menuDOMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryMenu(appMap.keySet(), true, true, true, true)).stream()
                .filter(t -> {
                    // 过滤业务菜单
                    if (!PlatformAppMenusTypeEnum.MENUS_TYPE_BUS.name().equals(t.getMenusType())) {
                        return false;
                    }
                    return true;
                }).collect(Collectors.toMap(MenuBO::getMenusCode, t -> t, (t1, t2) -> t1));

        // 组织vo
        List<RolePermissionRespVO> menuVOList = new ArrayList<>(menuTreeDOList.size());
        RolePermissionRespVO menuVO = null;
        for (SysTenantMenuTreeDO menuTreeDO : menuTreeDOList) {
            menuVO = new RolePermissionRespVO();
            menuVO.setCode(menuTreeDO.getMenuCode());
            menuVO.setName(menuTreeDO.getMenuName());

            // 设置节点原始信息
            var nodeType = MenuTreeNodeType.valueOf(menuTreeDO.getNodeType());
            if (nodeType == null) {
                log.error("未知菜单节点类型：" + menuTreeDO.getNodeType());
                continue;
            }
            if (MenuTreeNodeType.MENU == nodeType) {
                MenuBO menuBO = menuDOMap.get(menuTreeDO.getMenuCode());
                if (menuBO == null) {
                    // 是否是自定义菜单
                    if (!menuTreeDO.getCustom()) {
                        log.info("菜单不存在：" + menuTreeDO.getMenuCode());
                        continue;
                    }
                } else {
                    if (CharSequenceUtil.isBlank(menuVO.getName())) {
                        menuVO.setName(menuBO.getMenusName());
                    }

                    if (menuBO.getMenusState() != null && !menuBO.getMenusState()) {
                        // 菜单已禁用
                        continue;
                    }
                    menuVO.setRoute(menuBO.getMenusRoute());
                }
            } else if (MenuTreeNodeType.APP == nodeType) {
                var app = appMap.get(menuTreeDO.getMenuCode());
                if (app == null) {
                    log.info("应用已不存在：{}", menuTreeDO.getMenuCode());
                    continue;
                }
                if (CharSequenceUtil.isBlank(menuVO.getName())) {
                    menuVO.setName(app.getAppName());
                }
            } else {
                log.error("暂不支持的节点类型：{}", nodeType.getValue());
                continue;
            }
            menuVO.setNodeType(menuTreeDO.getNodeType());
            menuVO.setParentCode(menuTreeDO.getParentMenuCode());
            menuVO.setMenusIcon(menuTreeDO.getIcon());
            menuVO.setChecked(false);
            menuVO.setSortNo(menuTreeDO.getSortNo());
            menuVO.setHasChildren(false);
            menuVO.setChildren(new ArrayList<>(128));

            menuVOList.add(menuVO);
        }

        return menuVOList;
    }

    private List<RolePermissionRespVO> queryDefaultMenuTree(boolean includeApp, Map<String, AppBO> appMap) {
        List<RolePermissionRespVO> respVOList = includeApp ? convertMenuTreeRespVO(appMap) : new ArrayList<>(128);
        if (includeApp && respVOList.isEmpty()) {
            return Collections.emptyList();
        }

        // 获取菜单
        tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryMenu(appMap.keySet(), true, true, true, true)).stream()
                .filter(t -> {
                    // 过滤业务菜单
                    if (!PlatformAppMenusTypeEnum.MENUS_TYPE_BUS.name().equals(t.getMenusType())) {
                        return false;
                    }

                    return true;
                })
                .forEach(t -> {
                    RolePermissionRespVO respVO = new RolePermissionRespVO();
                    respVO.setId(t.getId());
                    respVO.setCode(t.getMenusCode());
                    respVO.setName(t.getMenusName());
                    respVO.setNodeType(MenuTreeNodeType.MENU.getValue());
                    respVO.setParentCode(t.getMenusParentCode());
                    if (CharSequenceUtil.isBlank(t.getMenusParentCode())) {
                        // 顶级的设置应用编码
                        respVO.setParentCode(t.getMenusAppCode());
                    }
                    respVO.setMenusIcon(t.getMenusIcon());
                    respVO.setRoute(t.getMenusRoute());
                    respVO.setChecked(false);
                    respVO.setSortNo(t.getMenusOrder());
                    respVO.setHasChildren(false);
                    respVO.setChildren(new ArrayList<>(128));

                    respVOList.add(respVO);
                });

        return respVOList;
    }

    private List<RolePermissionRespVO> convertMenuTreeRespVO(Map<String, AppBO> appMap) {
        List<RolePermissionRespVO> respVOList = new ArrayList<>(256);
        RolePermissionRespVO respVO = null;
        int sortNo = 1;
        for (var app : appMap.values()) {
            respVO = new RolePermissionRespVO();
            respVO.setCode(app.getAppCode());
            respVO.setName(app.getAppName());
            respVO.setNodeType(MenuTreeNodeType.APP.getValue());
            respVO.setMenusIcon(app.getIcon());
            respVO.setChecked(false);
            respVO.setSortNo(sortNo++);
            respVO.setHasChildren(false);
            respVO.setChildren(new ArrayList<>(128));

            respVOList.add(respVO);
        }

        return respVOList;
    }

    private void fillApiNumOfMenu(Set<String> appCodes, List<RolePermissionRespVO> respVOList) {
        if (CollectionUtils.isEmpty(respVOList)) {
            return;
        }

        // 查询菜单下的api数量
        var apiNumMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menuApiRepoProc.queryNumOfApi(appCodes));
        if (apiNumMap.isEmpty()) {
            return;
        }
        for (RolePermissionRespVO respVO : respVOList) {
            if (MenuTreeNodeType.MENU.getValue().equals(respVO.getNodeType())) {
                respVO.setApiNum(apiNumMap.get(respVO.getCode()));
            }
        }
    }

    private void saveRolePermission(long roleId, List<SysAppPermissionSaveBO> saveVOList, List<MenuTreeNodeType> nodeTypes, boolean custom) {
        Set<String> permissionType = nodeTypes.stream().map(MenuTreeNodeType::getValue).collect(Collectors.toSet());
        var roleCode = roleRepoProc.getCode(roleId);

        CompletableFuture.runAsync(() -> {
                    saveVOList.forEach(saveBO -> {
                        var lockKey = "cloudt_sys_role_permission:" + roleId + ":" + saveBO.getAppCode();
                        LockUtil.executeByLock(lockKey, () -> {
                            // 先删除应用下所有
                            rolePermissionRepoProc.deleteByRole(roleId, saveBO.getAppCode(), permissionType, custom);
                            if (CollectionUtils.isEmpty(saveBO.getPermissionList())) {
                                return null;
                            }

                            var doList = saveBO.getPermissionList().stream()
                                    .map(t -> {
                                        SysRolePermissionDO permissionDO = new SysRolePermissionDO();
                                        permissionDO.setRoleId(roleId);
                                        permissionDO.setRoleCode(roleCode);
                                        permissionDO.setPermissionCode(t.getCode());
                                        permissionDO.setAppCode(saveBO.getAppCode());
                                        permissionDO.setPermissionType(t.getNodeType());
                                        permissionDO.setCustom(custom);

                                        return permissionDO;
                                    }).collect(Collectors.toList());
                            rolePermissionRepoProc.save(doList);
                            return null;
                        }, Duration.ofMinutes(2), "当前角色操作人过多，请稍后再试");
                    });
                }, taskExecutor)
                .whenComplete((r, e) -> {
                    if (e == null) {
                        log.info("角色权限保存成功：{}", roleId);
                        return;
                    }
                    log.error("角色权限保存失败：{}", roleId, e);
                });
    }

    private void addRolePermission(long roleId, SysAppPermissionSaveBO saveBO, List<MenuTreeNodeType> nodeTypes, Boolean custom) {
        Set<String> permissionTypes = nodeTypes.stream().map(MenuTreeNodeType::getValue).collect(Collectors.toSet());
        var roleCode = roleRepoProc.getCode(roleId);

        // 先获取现有的
        var permissionsExists = rolePermissionRepoProc.getPermissionByRole(roleId, saveBO.getAppCode(), permissionTypes, custom).stream()
                .map(t -> t.getAppCode() + ":" + t.getPermissionType() + ":" + t.getPermissionCode())
                .collect(Collectors.toSet());
        var permissionDoList = saveBO.getPermissionList().stream()
                .filter(t -> {
                    if (permissionsExists.isEmpty()) {
                        return true;
                    }
                    return !permissionsExists.contains(saveBO.getAppCode() + ":" + t.getNodeType() + ":" + t.getCode());
                }).map(t -> {
                    SysRolePermissionDO permissionDO = new SysRolePermissionDO();
                    permissionDO.setRoleId(roleId);
                    permissionDO.setRoleCode(roleCode);
                    permissionDO.setPermissionCode(t.getCode());
                    permissionDO.setAppCode(saveBO.getAppCode());
                    permissionDO.setPermissionType(t.getNodeType());
                    permissionDO.setCustom(custom);

                    return permissionDO;
                }).collect(Collectors.toList());
        rolePermissionRepoProc.save(permissionDoList);
    }
}
