package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.zhxu.bs.BeanSearcher;
import cn.zhxu.bs.FieldOps;
import cn.zhxu.bs.util.MapUtils;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.datasecurity.config.DataSecurityProperties;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprSysInternallyEnum;
import com.elitescloud.boot.provider.PlatformAppProvider;
import com.elitescloud.boot.security.common.InnerRole;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.model.constant.PlatformAdminTypeEnum;
import com.elitescloud.cloudt.system.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.system.constant.PlatformMenusNodeEnum;
import com.elitescloud.cloudt.security.common.InnerUserEnum;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.convert.PermissionConverter;
import com.elitescloud.cloudt.system.convert.SysDpcRoleApiFieldsConvert;
import com.elitescloud.cloudt.system.dto.*;
import com.elitescloud.cloudt.system.model.bo.*;
import com.elitescloud.cloudt.system.model.vo.resp.index.UserFieldRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.index.UserMenuRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeeUserInfoRespVO;
import com.elitescloud.cloudt.system.model.vo.sbean.SysDprRoleApiDataColumnRuleListQueryBean;
import com.elitescloud.cloudt.system.model.vo.sbean.SysDprRoleApiDataRuleListQueryBean;
import com.elitescloud.cloudt.system.rpc.dpr.RoleAppApiDataPermissionRpcServiceUtil;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.common.constant.PermissionOwnerTypeEnum;
import com.elitescloud.cloudt.system.service.common.constant.SubUserPermissionTypeEnum;
import com.elitescloud.cloudt.system.service.model.bo.AppBO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.elitescloud.cloudt.system.spi.DprSysRuleSysInternallyValueSpi;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.google.common.base.Functions;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 权限查询相关接口.
 *
 * @author Kaiser（wang shao）
 * 2022/10/19
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@Log4j2
public class PermissionQueryManager extends BasePermissionManager {
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private UserRoleRepoProc userRoleRepoProc;
    @Autowired
    private EmployeeRepoProc employeeRepoProc;
    @Autowired
    private TenantMenuRepoProc tenantMenuRepoProc;
    @Autowired
    private TenantMenuTreeRepoProc tenantMenuTreeRepoProc;
    @Autowired
    private AdminMenuRepoProc adminMenuRepoProc;
    @Autowired
    private RolePermissionRepoProc rolePermissionRepoProc;
    @Autowired
    private ApiManageRepoProc apiRepoProc;
    @Autowired
    private ApiParameterRepoProc apiParameterRepoProc;
    @Autowired
    private SysDpcrApiFieldsRepoProc dpcrApiFieldsRepoProc;

    @Autowired
    private PlatformAppProvider appProvider;

    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private AuthorizationProperties authorizationProperties;
    @Autowired
    private DataSecurityProperties dataSecurityProperties;

    @Autowired
    private BeanSearcher beanSearcher;
    @Autowired
    private DprSysRuleSysInternallyValueSpi valueServiceSpi;

    /**
     * 获取用户的角色编码
     *
     * @param userDTO 用户
     * @return 角色编码
     */
    public Set<String> queryRoleByUser(SysUserDTO userDTO) {
        if (userDTO == null) {
            // 未登录
            return Set.of(InnerRole.ANONYMOUS.getValue());
        }

        Set<String> roles = new HashSet<>(16);
        // 内置角色
        // 系统管理员
        if (InnerUserEnum.ADMIN.getUsername().equals(userDTO.getUsername())) {
            roles.add(InnerRole.SYSTEM_ADMIN.getValue());
        }
        if (userDTO.getSysTenantVO() != null && Objects.equals(userDTO.getSysTenantVO().getSysUserId(), userDTO.getId())) {
            roles.add(InnerRole.TENANT_ADMIN.getValue());
        }
        if (Objects.equals(userDTO.getId(), userDTO.getTenantOrgAdminId())) {
            roles.add(InnerRole.TENANT_ORG_ADMIN.getValue());
        }
        // 分配的业务角色
        var businessRoles = userRoleRepoProc.getRoleOfUser(userDTO.getId(), userDTO.getTenantId(), userDTO.getTenantOrg() == null ? null : userDTO.getTenantOrg().getId());
        roles.addAll(businessRoles.stream().map(IdCodeNameParam::getCode).collect(Collectors.toList()));

        userDTO.setRoles(businessRoles);

        return Collections.unmodifiableSet(roles);
    }

    /**
     * 查询用户的角色
     *
     * @param userIds 用户
     * @return 角色
     */
    public Map<Long, List<IdCodeNameParam>> queryUserRoles(@NotEmpty Set<Long> userIds) {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();

        return userRoleRepoProc.getRoleOfUser(userIds, currentUser.getTenantId(), currentUser.getTenantOrgId());
    }

    /**
     * 获取用户菜单
     *
     * @param includeAction 是否包含操作按钮
     * @param tree          是否返回树状结构
     * @return 菜单列表
     */
    public List<UserMenuRespVO> queryUserMenusCacheable(boolean includeAction, boolean tree) {
        var cacheKey = "System:CurrentMenu:" + SecurityContextUtil.currentToken() + ":includeAction_" + includeAction + ":tree_" + tree;
        List<UserMenuRespVO> cacheResult = (List<UserMenuRespVO>) redisUtils.get(cacheKey);
        if (cacheResult != null) {
            return cacheResult;
        }

        var dataList = this.queryUserMenus(includeAction, tree);
        redisUtils.set(cacheKey, dataList, 20, TimeUnit.MINUTES);
        return dataList;
    }

    /**
     * 获取用户菜单
     *
     * @param includeAction 是否包含操作按钮
     * @param tree          是否返回树状结构
     * @return 菜单列表
     */
    public List<UserMenuRespVO> queryUserMenus(boolean includeAction, boolean tree) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();

        // 获取授权的应用
        var appMap = super.tenantApps(null);
        if (appMap.isEmpty()) {
            return Collections.emptyList();
        }
        // 主账号
        var masterUser = userQueryManager.getMasterUser(currentUser.getUserId());

        // 是否有自定义菜单
        var customized = hasCustomMenuTree(currentUser);

        // 所有菜单的基本信息
        CompletableFuture<Map<String, MenuBO>> menuMapFuture = this.getAllMenus(false, customized, currentUser, appMap.keySet(), includeAction);
        CompletableFuture<Map<String, MenuBO>> sysMenuMapFuture = this.getAllMenus(true, customized, currentUser, appMap.keySet(), includeAction);

        // 授权的管理员菜单
        CompletableFuture<List<UserMenuRespVO>> adminMenuFuture = this.getAdminUserPermissionMenuFuture(currentUser, masterUser, sysMenuMapFuture, customized, appMap, includeAction);
        // 授权的普通用户菜单
        CompletableFuture<List<UserMenuRespVO>> commonMenuFuture = this.getCommonUserPermissionMenuFuture(currentUser, masterUser, menuMapFuture, customized, appMap, includeAction);
        // 合并菜单
        return adminMenuFuture.thenCombineAsync(commonMenuFuture, (adminMenus, commonMenus) -> {
            List<UserMenuRespVO> menuList = new ArrayList<>(256);
            menuList.addAll(adminMenus);
            menuList.addAll(commonMenus);
            if (menuList.isEmpty()) {
                return new ArrayList<UserMenuRespVO>(0);
            }

            // 菜单去重
//            menuList = menuList.stream().collect(Collectors.collectingAndThen(
//                    Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(UserMenuRespVO::getMenuCode))),
//                    ArrayList::new
//            ));

            // 树形处理
            if (!tree) {
                return filterNoParent(menuList);
            }
            TreeDataUtil<UserMenuRespVO> treeDataUtil = new TreeDataUtil<>(menuList,
                    UserMenuRespVO::getMenuCode, UserMenuRespVO::getParentMenuCode,
                    UserMenuRespVO::setChildren, Comparator.comparingInt(UserMenuRespVO::getSortNo));
            return ((List<UserMenuRespVO>) treeDataUtil.getRoots()).stream()
                    .filter(t -> !StringUtils.hasText(t.getParentMenuCode()) && CollUtil.isNotEmpty(t.getChildren()))
                    .collect(Collectors.toList());
        }).join();
    }


    /**
     * 根据菜单查询用户的按钮
     *
     * @param menuCode 菜单编码
     * @return 按钮列表
     */
    public List<UserMenuRespVO> queryUserActionByMenu(@NotBlank String menuCode) {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        // 主账号
        var masterUser = userQueryManager.getMasterUser(currentUser.getUserId());

        // 先查询下级所有按钮
        AtomicReference<String> menuType = new AtomicReference<>();
        List<MenuBO> actionBos = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            var tempMenuType = menuRepoProc.getMenuTypeByMenuCode(menuCode);
            if (CharSequenceUtil.isBlank(tempMenuType)) {
                return Collections.emptyList();
            }
            menuType.set(tempMenuType);
            return menuRepoProc.queryActionByMenuCode(menuCode, true);
        });
        if (actionBos.isEmpty()) {
            return Collections.emptyList();
        }
        actionBos.forEach(t -> t.setNodeType(MenuTreeNodeType.ACTION.getValue()));

        // 如果是管理菜单，则无需授权，直接返回所有按钮
        if (PlatformAppMenusTypeEnum.MENUS_TYPE_PLATFORM.name().equals(menuType.get()) || PlatformAppMenusTypeEnum.MENUS_TYPE_SYS.name().equals(menuType.get())) {
            return platformMenu2Bo2Vo(actionBos);
        }

        // 普通菜单，则过滤出已授权的按钮
        var menuCodes = this.queryMenuCodeOfCommonUser(currentUser, masterUser, false, true);
        if (menuCodes.isEmpty()) {
            return Collections.emptyList();
        }
        actionBos = actionBos.stream().filter(t -> menuCodes.contains(t.getMenusCode())).collect(Collectors.toList());

        return platformMenu2Bo2Vo(actionBos);
    }

    /**
     * 查询用户的所有按钮
     *
     * @return 按钮列表
     */
    public List<UserMenuRespVO> queryAllUserAction() {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        // 获取授权的应用
        var appMap = super.tenantApps(null);
        if (appMap.isEmpty()) {
            return Collections.emptyList();
        }
        // 主账号
        var masterUser = userQueryManager.getMasterUser(currentUser.getUserId());

        // 所有按钮的基本信息
        CompletableFuture<Map<String, MenuBO>> menuMapFuture = this.getAllActions(currentUser, appMap.keySet());

        // 授权的管理员按钮
        CompletableFuture<List<UserMenuRespVO>> adminMenuFuture = this.getAdminUserPermissionActionFuture(menuMapFuture);
        // 授权的普通用户按钮
        CompletableFuture<List<UserMenuRespVO>> commonMenuFuture = this.getCommonUserPermissionActionFuture(currentUser, masterUser, menuMapFuture);
        // 合并按钮
        return adminMenuFuture.thenCombineAsync(commonMenuFuture, (adminMenus, commonMenus) -> {
            List<UserMenuRespVO> menuList = new ArrayList<>(256);
            menuList.addAll(adminMenus);
            menuList.addAll(commonMenus);

            return menuList;
        }).join();
    }

    /**
     * 查询用户授权的字段
     *
     * @param menuCode 菜单编码
     * @param apiCode  api接口编码
     * @return 字段列表
     */
    public List<UserFieldRespVO> queryUserField(@NotBlank String menuCode, @NotBlank String apiCode) {
        Assert.hasText(menuCode, "菜单编码为空");
        Assert.hasText(apiCode, "API接口编码为空");

        // 获取用户角色
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        AtomicReference<String> appCode = new AtomicReference<>();
        // 查询接口的出参信息
        List<CodeNameParam> apiParamList = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            // 查询出参
            var params = apiParameterRepoProc.queryOutParamByApiCode(apiCode);
            if (params.isEmpty()) {
                log.warn("接口{}未配置出参", apiCode);
                return Collections.emptyList();
            }
            // 查询接口是否还在
            appCode.set(apiRepoProc.getAppCodeByCode(apiCode));
            Assert.hasText(appCode.get(), "接口不存在或数据异常");
            return params;
        });
        if (apiParamList.isEmpty()) {
            return Collections.emptyList();
        }
        if (currentUser.getTenant() != null && !CollUtil.contains(currentUser.getTenant().getAppCodes(), appCode.get())) {
            // 应用无权限
            log.warn("接口{}所属应用{}未分配给租户{}", apiCode, appCode, currentUser.getTenantId());
            return this.convertUserField(apiParamList, Collections.emptyList());
        }
        var userRoles = this.obtainUserRoleId(currentUser);
        if (userRoles.isEmpty()) {
            // 未分配权限
            return this.convertUserField(apiParamList, Collections.emptyList());
        }
        // 查询角色分配的权限
        var permissionFields = dpcrApiFieldsRepoProc.queryBoByRole(userRoles, menuCode, apiCode);
        return this.convertUserField(apiParamList, permissionFields);
    }

    /**
     * 获取角色管理的用户ID
     *
     * @param roleCode 角色编码
     * @return 用户ID
     */
    public Set<Long> queryUserIdOfRole(@NotBlank String roleCode) {
        return userRoleRepoProc.getUserIdByRole(roleCode);
    }

    /**
     * 获取角色管理的用户ID
     *
     * @param roleId 角色ID
     * @return 用户ID
     */
    public Set<Long> queryUserIdOfRole(long roleId) {
        return userRoleRepoProc.getUserIdsByRole(roleId);
    }

    /**
     * 查询角色下的用户
     *
     * @param roleId       角色ID
     * @param onlyEmployee 是否仅显示员工
     * @return
     */
    public List<EmployeeUserInfoRespVO> queryUserOfRole(long roleId, boolean onlyEmployee) {
        if (onlyEmployee) {
            // 仅显示员工的话，直接查员工
            return employeeRepoProc.listEmployeeByRole(roleId).stream().map(t -> {
                EmployeeUserInfoRespVO respVO = new EmployeeUserInfoRespVO();
                respVO.setId(t.getId());
                respVO.setCode(t.getCode());
                respVO.setUserId(t.getUserId());
                respVO.setUsername(t.getUsername());
                respVO.setFullName(t.getFullName());
                respVO.setEmail(CharSequenceUtil.blankToDefault(t.getEmailWork(), t.getEmail()));
                respVO.setMobile(t.getMobile());

                return respVO;
            }).collect(Collectors.toList());
        }

        // 查询角色下的用户
        var userIds = userRoleRepoProc.getUserIdsByRole(roleId);
        if (userIds.isEmpty()) {
            return Collections.emptyList();
        }
        var employeeMap = employeeRepoProc.queryBasicByUserIds(userIds).stream().collect(Collectors.toMap(SysEmployeeBasicDTO::getUserId, Function.identity(), (t1, t2) -> t1));
        return tenantDataIsolateProvider.byDefaultDirectly(() -> userRepoProc.getBasicDto(userIds))
                .stream()
                .map(t -> {
                    EmployeeUserInfoRespVO respVO = new EmployeeUserInfoRespVO();

                    var emp = employeeMap.get(t.getId());
                    if (emp != null) {
                        respVO.setId(emp.getId());
                        respVO.setCode(emp.getCode());
                        respVO.setEmail(CharSequenceUtil.blankToDefault(emp.getEmailWork(), emp.getEmail()));
                        respVO.setMobile(emp.getMobile());
                    } else {
                        respVO.setEmail(t.getEmail());
                        respVO.setMobile(t.getMobile());
                    }

                    respVO.setUserId(t.getId());
                    respVO.setUsername(t.getUsername());
                    respVO.setFullName(t.getFullName());

                    return respVO;
                }).collect(Collectors.toList());
    }

    /**
     * 获取用户的数据权限
     *
     * @return 数据权限
     */
    public SysDprRoleApiRowColumnRuleDTO getDataPermissionOfCurrentUser() {
        SysDprRoleApiRowColumnRuleDTO roleRuleDTO = new SysDprRoleApiRowColumnRuleDTO();
        // 用户信息
        var userInfo = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        roleRuleDTO.setUserId(userInfo.getUserId());
        roleRuleDTO.setTenantId(ObjectUtil.defaultIfNull(userInfo.getTenantId(), TenantConstant.DEFAULT_TENANT_ID));
        // 角色信息
        List<String> roleCodeList = CollUtil.isEmpty(userInfo.getUser().getRoles()) ? Collections.emptyList() :
                userInfo.getUser().getRoles().stream().map(IdCodeNameParam::getCode).collect(Collectors.toList());
        roleRuleDTO.setRoleCodeList(roleCodeList);
        List<Long> roleIdList = CollUtil.isEmpty(userInfo.getUser().getRoles()) ? Collections.emptyList() :
                userInfo.getUser().getRoles().stream().map(IdCodeNameParam::getId).collect(Collectors.toList());
        roleRuleDTO.setRoelIdList(roleIdList);
        if (roleCodeList.isEmpty() ||
                (userInfo.getTenant() != null && CollUtil.isEmpty(userInfo.getTenant().getAppCodes()))) {
            // 无权限，直接返回
            roleRuleDTO.setSysDpcRoleApiFieldsDTOList(Collections.emptyList());
            roleRuleDTO.setSysDprRoleApiDataRuleListQueryDTO(Collections.emptyList());
            return roleRuleDTO;
        }

        // 查询权限规则
        List<SysDprRoleApiDataRuleListQueryDTO> dataRuleList = new ArrayList<>();
        List<SysDpcRoleApiFieldsDTO> fieldRuleList = new ArrayList<>();
        switch (dataSecurityProperties.getCompatible()) {
            case LATEST:
                // 使用最新的
                dataRuleList.addAll(this.queryDataPermission(userInfo, roleCodeList));
                fieldRuleList.addAll(this.queryFieldPermission(userInfo, roleCodeList));
                break;
            case OLDEST:
                // 使用老的
                dataRuleList.addAll(this.queryDprRoleApiDataRule(userInfo, roleCodeList));
                // 查询字段权限
                fieldRuleList.addAll(this.queryDprRoleApiFields(roleCodeList));
                break;
            case FIRST_LATEST:
                dataRuleList.addAll(this.queryDataPermission(userInfo, roleCodeList));
                dataRuleList.addAll(this.queryDprRoleApiDataRule(userInfo, roleCodeList));
                // 查询字段权限
                fieldRuleList.addAll(this.queryFieldPermission(userInfo, roleCodeList));
                fieldRuleList.addAll(this.queryDprRoleApiFields(roleCodeList));
                break;
            case FIRST_OLDEST:
                dataRuleList.addAll(this.queryDprRoleApiDataRule(userInfo, roleCodeList));
                dataRuleList.addAll(this.queryDataPermission(userInfo, roleCodeList));
                // 查询字段权限
                fieldRuleList.addAll(this.queryDprRoleApiFields(roleCodeList));
                fieldRuleList.addAll(this.queryFieldPermission(userInfo, roleCodeList));
                break;
            default:
                throw new IllegalArgumentException("暂不支持的兼容模式：" + dataSecurityProperties.getCompatible());
        }
        roleRuleDTO.setSysDprRoleApiDataRuleListQueryDTO(dataRuleList);
        roleRuleDTO.setSysDpcRoleApiFieldsDTOList(fieldRuleList);

        return roleRuleDTO;
    }

    private List<SysDpcRoleApiFieldsDTO> queryFieldPermission(GeneralUserDetails userInfo, List<String> roleCodeList) {
        // 查询权限字段列表
        var permissionList = fieldPermissionRepoProc.queryByRoleCodes(roleCodeList);
        if (permissionList.isEmpty()) {
            return permissionList;
        }

        Map<String, IdCodeNameParam> roleMap = userInfo.getUser().getRoles().stream().collect(Collectors.toMap(IdCodeNameParam::getCode, Function.identity(), (t1, t2) -> t1));

        // 操作编码
        Set<String> operationCodes = new HashSet<>(permissionList.size());
        for (var dto : permissionList) {
            if (StringUtils.hasText(dto.getApiPermissionCode())) {
                operationCodes.add(dto.getApiPermissionCode());
            }
            // 兼容老的
            dto.setRoleId(roleMap.get(dto.getRoleCode()).getId());
            dto.setFieldApiVisible(dto.getReadable());
            dto.setFieldFormVisible(dto.getReadable());
            dto.setFieldFormUpdate(dto.getWriteable());
        }
        if (operationCodes.isEmpty()) {
            return permissionList;
        }

        // 设置业务对象信息
        return this.fillByBusinessObject(permissionList, operationCodes, PermissionConverter.INSTANCE::cloneRule);
    }

    private List<SysDprRoleApiDataRuleListQueryDTO> queryDataPermission(GeneralUserDetails userInfo, List<String> roleCodeList) {
        var permissionList = dataPermissionRepoProc.queryDataPermissionByRoles(roleCodeList);
        if (permissionList.isEmpty()) {
            return permissionList;
        }
        Map<String, IdCodeNameParam> roleMap = userInfo.getUser().getRoles().stream().collect(Collectors.toMap(IdCodeNameParam::getCode, Function.identity(), (t1, t2) -> t1));

        // 操作编码
        Set<String> operationCodes = new HashSet<>(permissionList.size());
        for (var dto : permissionList) {
            if (StringUtils.hasText(dto.getApiPermissionCode())) {
                operationCodes.add(dto.getApiPermissionCode());
            }
            // 兼容老的
            dto.setRoleId(roleMap.get(dto.getRoleCode()).getId());
            dto.setRuleOrder(dto.getOrder() == null ? 0 : dto.getOrder() * 1.0);
        }
        // 设置规则的值
        RoleAppApiDataPermissionRpcServiceUtil.fillDataPermissionValue(permissionList, userInfo);

        // 设置业务对象信息
        if (operationCodes.isEmpty()) {
            return permissionList;
        }
        return this.fillByBusinessObject(permissionList, operationCodes, PermissionConverter.INSTANCE::cloneRule);
    }

    private <T extends BaseDataSecurityRuleDTO> List<T> fillByBusinessObject(List<T> ruleList, Collection<String> operationCodes,
                                                                             Function<T, T> cloner) {
        if (operationCodes.isEmpty()) {
            return ruleList;
        }

        // 操作信息
        List<OperationRequestInfoBO> operations = new ArrayList<>(operationCodes.size());
        List<BusinessOperationBO> refOperations = new ArrayList<>(operationCodes.size());
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            operations.addAll(businessOperationRepoProc.listRequestInfo(operationCodes, null));
            refOperations.addAll(businessOperationRepoProc.listSimpleByRef(operationCodes));
            return null;
        });

        // 设置操作信息
        Map<String, OperationRequestInfoBO> operationMap = operations.stream()
                .collect(Collectors.toMap(OperationRequestInfoBO::getOperationCode, Function.identity(), (t1, t2) -> t1));
        List<T> resultList = new ArrayList<>(ruleList.size());
        for (var dto : ruleList) {
            if (StringUtils.hasText(dto.getApiPermissionCode())) {
                ObjUtil.ifNotNull(operationMap.get(dto.getApiPermissionCode()), t -> {
                    dto.setApiPermissionRequestType(t.getApiMethod());
                    dto.setApiPermissionPath(t.getApiUrl());
                    dto.setPermissionRef(t.getPermissionRef());
                });
                if (!StringUtils.hasText(dto.getApiPermissionPath())) {
                    continue;
                }
            }

            resultList.add(dto);
        }

        // 设置关联的操作
        this.fillRefPermission(resultList, refOperations, cloner);

        return resultList;
    }

    private <T extends BaseDataSecurityRuleDTO> void fillRefPermission(List<T> ruleList, List<BusinessOperationBO> refOperations,
                                                                       Function<T, T> cloner) {
        if (ruleList.isEmpty() || refOperations.isEmpty()) {
            return;
        }
        var refOperationCodes = refOperations.stream().map(BusinessOperationBO::getOperationCode).collect(Collectors.toSet());

        var refRuleMap = ruleList.stream()
                .filter(t -> StringUtils.hasText(t.getApiPermissionCode()))
                .collect(Collectors.groupingBy(BaseDataSecurityRuleDTO::getApiPermissionCode));
        var existsRefRuleMap = ruleList.stream()
                .filter(t -> StringUtils.hasText(t.getApiPermissionCode()) && refOperationCodes.contains(t.getApiPermissionCode()))
                .collect(Collectors.groupingBy(t -> t.getPermissionType() + "::" + t.getApiPermissionCode()));
        for (var operation : refOperations) {
            var refRuleList = refRuleMap.get(operation.getPermissionRef());
            if (CollUtil.isEmpty(refRuleList)) {
                continue;
            }

            for (var refRule : refRuleList) {
                if (existsRefRuleMap.containsKey(refRule.getPermissionType() + "::" + operation.getOperationCode())) {
                    // 已存在自定义配置
                    continue;
                }

                var newRule = cloner.apply(refRule);
                newRule.setPermissionRef(operation.getPermissionRef());
                newRule.setAppCode(operation.getAppCode());
                newRule.setBusinessObjectCode(operation.getBusinessObjectCode());
                newRule.setApiPermissionCode(operation.getOperationCode());
                newRule.setApiPermissionPath(operation.getApiUrl());
                newRule.setApiPermissionRequestType(operation.getApiMethod());

                ruleList.add(newRule);
            }
        }
    }

    private List<SysDprRoleApiDataRuleListQueryDTO> queryDprRoleApiDataRule(GeneralUserDetails userInfo, List<String> roleCodeList) {
        var mapWhere =
                MapUtils.builder()
                        .field(SysDprRoleApiDataRuleListQueryBean::getRoleCode, roleCodeList).op(FieldOps.InList)
                        .build();
        var sysDprRoleApiDataRuleListQueryBeans
                = beanSearcher.searchAll(SysDprRoleApiDataRuleListQueryBean.class, mapWhere);

        var beanList = sysDprRoleApiDataRuleListQueryBeans.stream()
                .filter(t -> {
                    // 过滤掉【全部】 的
                    if (CharSequenceUtil.equals(DprSysInternallyEnum.DPR_SYS_INTERNALLY_ALL.name(), t.getDprSysInternally())) {
                        return false;
                    }
                    if (t.getApiId() == null) {
                        log.error("接口不存在：{}", t.getId());
                        return false;
                    }
                    if (CharSequenceUtil.isBlank(t.getMenusCode())) {
                        return false;
                    }
                    if (CharSequenceUtil.isBlank(t.getDprRuleValueType())) {
                        log.error("规则值类型不存在：{}", t.getId());
                        return false;
                    }
                    return true;
                }).collect(Collectors.toList());

        // 根据规则值类型进行分组
        Map<String, List<SysDprRoleApiDataRuleListQueryBean>> dprRuleValueTypeGroupMap =
                beanList.stream()
                        .collect(Collectors.groupingBy(SysDprRoleApiDataRuleListQueryBean::getDprRuleValueType));
        //汇总全部的规则集合
        List<SysDprRoleApiDataRuleListQueryDTO> dtoList = new ArrayList<>();
        //业务集合
        if (dprRuleValueTypeGroupMap.containsKey(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS.name())) {
            List<SysDprRoleApiDataRuleListQueryBean> businessBean =
                    dprRuleValueTypeGroupMap.get(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS.name());
            var beanValueList = RoleAppApiDataPermissionRpcServiceUtil
                    .setBusinessSysDprRoleApiRuleValue(userInfo, businessBean);
            dtoList.addAll(beanValueList);
        }
        //自定义
        if (dprRuleValueTypeGroupMap.containsKey(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_CUSTOM.name())) {
            List<SysDprRoleApiDataRuleListQueryBean> customBean =
                    dprRuleValueTypeGroupMap.get(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_CUSTOM.name());
            var beanValueList = RoleAppApiDataPermissionRpcServiceUtil.
                    setCustomSysDprRoleApiRuleValue(userInfo, customBean);
            dtoList.addAll(beanValueList);
        }
        //系统内置 重点
        if (dprRuleValueTypeGroupMap.containsKey(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name())) {
            List<SysDprRoleApiDataRuleListQueryBean> sysBean =
                    dprRuleValueTypeGroupMap.get(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name());
            var beanValueList = RoleAppApiDataPermissionRpcServiceUtil
                    .setSysSysDprRoleApiRuleValue(valueServiceSpi, userInfo, sysBean);
            dtoList.addAll(beanValueList);
        }

        return dtoList;
    }

    private List<SysDpcRoleApiFieldsDTO> queryDprRoleApiFields(List<String> roleCodeList) {
        var mapColumnWhere = MapUtils.builder()
                .field(SysDprRoleApiDataColumnRuleListQueryBean::getRoleCode, roleCodeList).op(FieldOps.InList)
                .build();
        var roleApiColumnRuleList = beanSearcher.searchAll(SysDprRoleApiDataColumnRuleListQueryBean.class, mapColumnWhere);
        if (CollUtil.isEmpty(roleApiColumnRuleList)) {
            return Collections.emptyList();
        }

        return roleApiColumnRuleList.stream()
                .map(SysDpcRoleApiFieldsConvert.INSTANCE::beanToDto)
                .collect(Collectors.toList());
    }

    private List<UserFieldRespVO> convertUserField(List<CodeNameParam> apiParamList, List<PermissionParameterBO> permissionFields) {
        Map<String, List<PermissionParameterBO>> perMap = CollectionUtils.isEmpty(permissionFields) ? Collections.emptyMap() :
                permissionFields.stream().collect(Collectors.groupingBy(PermissionParameterBO::getFieldName));
        var conflictShow = systemProperties.getPermissionConflictShow() == null || systemProperties.getPermissionConflictShow();
        return apiParamList.stream()
                .map(t -> {
                    UserFieldRespVO respVO = new UserFieldRespVO();
                    respVO.setFieldName(t.getCode());
                    respVO.setFieldRemark(t.getName());

                    if (perMap.isEmpty() || !perMap.containsKey(t.getCode())) {
                        // 默认不显示
                        respVO.setFieldApiVisible(false);
                        respVO.setFieldFormVisible(false);
                        respVO.setFieldFormUpdate(false);
                        return respVO;
                    }

                    var perList = perMap.get(t.getCode());
                    respVO.setId(CollUtil.isEmpty(perList) ? null : perList.get(0).getId());
                    respVO.setFieldApiVisible(this.authShow(conflictShow, perList, PermissionParameterBO::getFieldApiVisible));
                    respVO.setFieldFormVisible(this.authShow(conflictShow, perList, PermissionParameterBO::getFieldFormVisible));
                    respVO.setFieldFormUpdate(this.authShow(conflictShow, perList, PermissionParameterBO::getFieldFormUpdate));
                    return respVO;
                }).collect(Collectors.toList());
    }

    private boolean authShow(boolean conflictShow, List<PermissionParameterBO> parameterBOList, Function<PermissionParameterBO, Boolean> valueFunction) {
        if (CollUtil.isEmpty(parameterBOList)) {
            return false;
        }

        var show = false;
        for (PermissionParameterBO parameterBO : parameterBOList) {
            show = Boolean.TRUE.equals(valueFunction.apply(parameterBO));
            if (conflictShow) {
                // 如果是【冲突时显示】，则只要有一个显示就显示
                if (show) {
                    return true;
                }
            } else {
                // 如果是【冲突时不显示】，则只要有一个不显示就不显示
                if (!show) {
                    return false;
                }
            }
        }
        return show;
    }

    @SuppressWarnings("unchecked")
    private List<UserMenuRespVO> filterNoParent(List<UserMenuRespVO> respVos) {
        TreeDataUtil<UserMenuRespVO> treeDataUtil = new TreeDataUtil<>(respVos, UserMenuRespVO::getMenuCode, UserMenuRespVO::getParentMenuCode,
                UserMenuRespVO::setChildren);
        List<UserMenuRespVO> rootDataList = ((List<UserMenuRespVO>) treeDataUtil.getRoots()).stream()
                .filter(t -> !StringUtils.hasText(t.getParentMenuCode()) && CollUtil.isNotEmpty(t.getChildren())).collect(Collectors.toList());
        if (rootDataList.isEmpty()) {
            return Collections.emptyList();
        }

        return CollectionUtil.expandTree(rootDataList, UserMenuRespVO::getChildren);
    }

    private List<UserMenuRespVO> permissionMenuBo2Vo(List<PermissionMenuBO> menuBos) {
        if (CollectionUtils.isEmpty(menuBos)) {
            return Collections.emptyList();
        }
        List<UserMenuRespVO> respVos = new ArrayList<>(menuBos.size());
        UserMenuRespVO respVO = null;
        for (PermissionMenuBO menuBO : menuBos) {
            respVO = new UserMenuRespVO();
            respVO.setMenuCode(menuBO.getMenusCode());
            respVO.setMenuName(menuBO.getMenusName());
            respVO.setParentMenuCode(menuBO.getMenusParentCode());
            respVO.setMenusIcon(menuBO.getMenusIcon());
            respVO.setRoute(menuBO.getMenusRoute());
            respVO.setNodeType(menuBO.getNodeType());
            respVO.setDataType(menuBO.getDataType());
            respVO.setSortNo(ObjectUtil.defaultIfNull(menuBO.getMenusOrder(), 1));
            respVO.setDisplay(ObjectUtil.defaultIfNull(menuBO.getDisplay(), true));
            respVO.setOuterLink(menuBO.getOuterLink());
            respVO.setOuterLinkType(menuBO.getOuterLinkType());
            respVO.setHasChildren(false);
            respVO.setChildren(new ArrayList<>(16));

            respVos.add(respVO);
        }
        return respVos;
    }

    private List<UserMenuRespVO> platformMenu2Bo2Vo(List<MenuBO> menuBos) {
        if (CollectionUtils.isEmpty(menuBos)) {
            return Collections.emptyList();
        }
        List<UserMenuRespVO> respVos = new ArrayList<>(menuBos.size());
        UserMenuRespVO respVO = null;
        for (MenuBO menuBO : menuBos) {
            respVO = new UserMenuRespVO();
            respVO.setMenuCode(menuBO.getMenusCode());
            respVO.setMenuName(menuBO.getMenusName());
            respVO.setParentMenuCode(menuBO.getMenusParentCode());
            respVO.setMenusIcon(menuBO.getMenusIcon());
            respVO.setRoute(menuBO.getMenusRoute());
            respVO.setNodeType(menuBO.getNodeType());
            respVO.setDataType(menuBO.getMenusType());
            respVO.setSortNo(ObjectUtil.defaultIfNull(menuBO.getMenusOrder(), 1));
            respVO.setDisplay(ObjectUtil.defaultIfNull(menuBO.getDisplay(), true));
            respVO.setOuterLink(menuBO.getOuterLink());
            respVO.setOuterLinkType(menuBO.getOuterLinkType());
            respVO.setHasChildren(false);
            respVO.setChildren(new ArrayList<>(16));

            respVos.add(respVO);
        }
        return respVos;
    }

    private MenuBO tenantMenuBo2MenuBo(TenantMenuBO tenantMenuBO, MenuBO originalMenu) {
        MenuBO menuBO = new MenuBO();
        menuBO.setId(tenantMenuBO.getId());
        if (originalMenu != null) {
            menuBO.setMenusAppCode(originalMenu.getMenusAppCode());
            menuBO.setMenusType(originalMenu.getMenusType());
            menuBO.setMenusRoute(originalMenu.getMenusRoute());
            menuBO.setDisplay(originalMenu.getDisplay());
            menuBO.setOuterLink(originalMenu.getOuterLink());
            menuBO.setOuterLinkType(originalMenu.getOuterLinkType());
        } else {
            menuBO.setDisplay(true);
        }
        menuBO.setMenusName(tenantMenuBO.getMenuName());
        menuBO.setNodeType(tenantMenuBO.getNodeType());
        menuBO.setMenusCode(tenantMenuBO.getMenuCode());
        menuBO.setMenusOrder(ObjUtil.defaultIfNull(tenantMenuBO.getSortNo(), 0));
        menuBO.setMenusState(true);
        menuBO.setMenusParentCode(tenantMenuBO.getParentMenuCode());
        menuBO.setMenusIcon(tenantMenuBO.getIcon());

        return menuBO;
    }

    private CompletableFuture<Map<String, MenuBO>> getAllMenus(Boolean sys, boolean customMenu, GeneralUserDetails currentUser,
                                                               Set<String> appCodes, boolean includeAction) {
        var originalMenuFuture = CompletableFuture.supplyAsync(() -> {
            boolean includeSys = true;
            boolean includeBusiness = true;
            if (sys != null) {
                if (sys) {
                    includeBusiness = false;
                } else {
                    includeSys = false;
                }
            }
            return menuRepoProc.queryMenu(appCodes, includeSys, includeBusiness, !includeAction, true).stream()
                    .collect(Collectors.toMap(MenuBO::getMenusCode, Functions.identity(), (t1, t2) -> t1));
        });
        if (sys || !customMenu) {
            // 管理员菜单，无需判断是否有自定义
            return originalMenuFuture;
        }

        return originalMenuFuture.thenApplyAsync(originalMenuMap -> {
            if (originalMenuMap.isEmpty()) {
                return Collections.emptyMap();
            }

            // 自定义菜单
            var tenantMenuBoList = tenantDataIsolateProvider.byTenantUser(() -> tenantMenuTreeRepoProc.queryMenuBos(), currentUser.getUser());
            if (tenantMenuBoList.isEmpty()) {
                return Collections.emptyMap();
            }

            return tenantMenuBoList.stream()
                    .filter(t -> {
                        // 启用的
                        if (Boolean.FALSE.equals(t.getEnabled())) {
                            return false;
                        }
                        if (MenuTreeNodeType.MENU.getValue().equals(t.getNodeType())) {
                            var menu = originalMenuMap.get(t.getMenuCode());
                            return menu != null && Boolean.TRUE.equals(menu.getMenusState());
                        }
                        return true;
                    }).map(t -> this.tenantMenuBo2MenuBo(t, originalMenuMap.get(t.getMenuCode()))).collect(Collectors.toMap(MenuBO::getMenusCode, Function.identity(), (t1, t2) -> t1));

        }, taskExecutor);
    }

    private CompletableFuture<Map<String, MenuBO>> getAllActions(GeneralUserDetails currentUser, Set<String> appCodes) {
        return CompletableFuture.supplyAsync(() -> menuRepoProc.queryActionByAppCode(appCodes, true).stream()
                .collect(Collectors.toMap(MenuBO::getMenusCode, Functions.identity(), (t1, t2) -> t1)));
    }

    private CompletableFuture<List<UserMenuRespVO>> getAdminUserPermissionActionFuture(CompletableFuture<Map<String, MenuBO>> menuMapFuture) {
        return CompletableFuture.supplyAsync(() -> {
            Map<String, MenuBO> allMenuMap = null;
            try {
                allMenuMap = menuMapFuture.get();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            var actionBoList = allMenuMap.values().stream()
                    .filter(t -> PlatformAppMenusTypeEnum.MENUS_TYPE_SYS.name().equals(t.getMenusType()) || PlatformAppMenusTypeEnum.MENUS_TYPE_PLATFORM.name().equals(t.getMenusType()))
                    .map(this::convertPermissionMenuBoForActionBo).collect(Collectors.toList());
            return this.permissionMenuBo2Vo(actionBoList);
        });
    }

    private CompletableFuture<List<UserMenuRespVO>> getCommonUserPermissionActionFuture(GeneralUserDetails currentUser, MasterUserBO masterUser,
                                                                                        CompletableFuture<Map<String, MenuBO>> menuMapFuture) {
        return CompletableFuture.supplyAsync(() -> {
            // 获取分配的按钮
            var menuCodes = this.queryMenuCodeOfCommonUser(currentUser, masterUser, false, true);
            if (menuCodes.isEmpty()) {
                return Collections.emptyList();
            }
            Map<String, MenuBO> allMenuMap;
            try {
                allMenuMap = menuMapFuture.get();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            List<PermissionMenuBO> userMenus = menuCodes.stream()
                    .filter(allMenuMap::containsKey)
                    .map(t -> this.convertPermissionMenuBoForActionBo(allMenuMap.get(t)))
                    .collect(Collectors.toList());

            return this.permissionMenuBo2Vo(userMenus);
        }, taskExecutor);
    }

    private CompletableFuture<List<UserMenuRespVO>> getCommonUserPermissionMenuFuture(GeneralUserDetails currentUser, MasterUserBO masterUser,
                                                                                      CompletableFuture<Map<String, MenuBO>> menuMapFuture,
                                                                                      boolean customMenu, Map<String, AppBO> appMap,
                                                                                      boolean includeAction) {
        return CompletableFuture.supplyAsync(() -> {
            var appCodes = appMap.keySet();
            // 获取分配的菜单
            var menuCodes = this.queryMenuCodeOfCommonUser(currentUser, masterUser, true, includeAction);
            if (menuCodes.isEmpty()) {
                return Collections.emptyList();
            }
            Map<String, MenuBO> allMenuMap = null;
            try {
                allMenuMap = menuMapFuture.get();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            List<PermissionMenuBO> userMenus = new ArrayList<>(256);
            Set<String> existsMenuBos = new HashSet<>(256);
            for (String menuCode : menuCodes) {
                this.addPermissionMenuBO(customMenu, menuCode, allMenuMap, appCodes, userMenus, existsMenuBos, (m, mb) -> mb.setMenusParentCode(m.getMenusAppCode()));
            }

            // 应用
            if (!customMenu) {
                userMenus.addAll(this.convertPermissionMenuForApp(appMap.values()));
            }

            return this.permissionMenuBo2Vo(userMenus);
        }, taskExecutor);
    }

    private CompletableFuture<List<UserMenuRespVO>> getAdminUserPermissionMenuFuture(GeneralUserDetails currentUser, MasterUserBO masterUser,
                                                                                     CompletableFuture<Map<String, MenuBO>> menuMapFuture,
                                                                                     boolean customMenu, Map<String, AppBO> appMap,
                                                                                     boolean includeAction) {
        return CompletableFuture.supplyAsync(() -> {
            var appCodes = appMap.keySet();
            var menuCodes = this.queryMenuCodeOfAdminUser(currentUser, masterUser, appCodes);
            if (menuCodes.isEmpty()) {
                return Collections.emptyList();
            }
            Map<String, MenuBO> allMenuMap = null;
            try {
                allMenuMap = menuMapFuture.get();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            List<PermissionMenuBO> userMenus = new ArrayList<>(256);
            Set<String> existsMenuBos = new HashSet<>(256);
            for (String menuCode : menuCodes) {
                this.addPermissionMenuBO(customMenu, menuCode, allMenuMap, appCodes, userMenus, existsMenuBos, null);
            }
            if (!includeAction) {
                return this.permissionMenuBo2Vo(userMenus);
            }

            // 按钮
            var actionBoList = allMenuMap.values().stream()
                    .filter(t -> {
                        if (!PlatformMenusNodeEnum.BUTTON.name().equals(t.getNodeType())) {
                            return false;
                        }
                        if (!PlatformAppMenusTypeEnum.MENUS_TYPE_SYS.name().equals(t.getMenusType()) && !PlatformAppMenusTypeEnum.MENUS_TYPE_PLATFORM.name().equals(t.getMenusType())) {
                            return false;
                        }
                        return menuCodes.contains(t.getMenusParentCode());
                    })
                    .map(this::convertPermissionMenuBoForActionBo).collect(Collectors.toList());
            userMenus.addAll(actionBoList);
            return this.permissionMenuBo2Vo(userMenus);
        });
    }

    private Set<String> queryMenuCodeOfCommonUser(@NotNull GeneralUserDetails currentUser, MasterUserBO masterUser, boolean includeMenu, boolean includeAction) {
        Set<String> menuCodes = new HashSet<>(256);

        // 获取根据角色分配的菜单
        var roleIds = obtainUserRoleIds(currentUser, masterUser);
        if (!roleIds.isEmpty()) {
            menuCodes.addAll(rolePermissionRepoProc.queryPermissionCodesByRoles(roleIds, includeMenu, includeAction));
        }

        // 获取直接分配的
        Map<String, PermissionOwnerTypeEnum> owners = new HashMap<>(4);
        owners.put(currentUser.getUserId().toString(), PermissionOwnerTypeEnum.USER);
        if (masterUser != null && SubUserPermissionTypeEnum.EXTENDS_ALL == masterUser.getPermissionType()) {
            owners.put(masterUser.getMasterUserId().toString(), PermissionOwnerTypeEnum.USER);
        }
        menuCodes.addAll(permissionResRepoProc.listMenuCodeByUsers(owners, includeMenu, includeAction));

        return menuCodes;
    }

    private Set<String> queryMenuCodeOfAdminUser(@NotNull GeneralUserDetails currentUser, MasterUserBO masterUser, Set<String> authedAppCodes) {
        Set<String> adminTypeEnums = new HashSet<>(8);
        // 系统管理员
        if (super.isSystemAdmin(currentUser, masterUser)) {
            adminTypeEnums.add(PlatformAdminTypeEnum.SYS_ADMIN.name());
        }
        // 租户管理员
        if (super.isTenantAdmin(currentUser, masterUser) && CollUtil.isNotEmpty(authedAppCodes)) {
            adminTypeEnums.add(PlatformAdminTypeEnum.TENANT_ADMIN.name());
        }
        // 租户组织管理员
        if (super.isTenantOrgAdmin(currentUser, masterUser)) {
            adminTypeEnums.add(PlatformAdminTypeEnum.TENANT_ORG_ADMIN.name());
        }

        // 非管理员
        if (adminTypeEnums.isEmpty()) {
            return Collections.emptySet();
        }
        return tenantDataIsolateProvider.byDefaultDirectly(() -> {
            var menuCodes = adminMenuRepoProc.getMenuCodes(adminTypeEnums);
            if (!menuCodes.isEmpty()) {
                return new HashSet<>(menuCodes);
            }

            // 系统管理员的话，可能是初始化系统，暂未分配
            if (adminTypeEnums.contains(PlatformAdminTypeEnum.SYS_ADMIN.name())) {
                return new HashSet<>(menuRepoProc.getMenuCodes(true, true, false, true));
            }
            return Collections.emptySet();
        });
    }


    private void addPermissionMenuBO(boolean customMenu, @NotBlank String menuCodeAuthed, Map<String, MenuBO> menusAllMap, Set<String> appCodesAuthed,
                                     List<PermissionMenuBO> result, Set<String> existsResult, BiConsumer<MenuBO, PermissionMenuBO> rootMenuFunction) {
        if (existsResult.contains(menuCodeAuthed)) {
            return;
        }

        var menu = menusAllMap.get(menuCodeAuthed);
        if (menu == null) {
            return;
        }
        // 是否有对应应用的权限
        if (appCodesAuthed != null) {
            if (CharSequenceUtil.isBlank(menu.getMenusAppCode())) {
                if (!customMenu || MenuTreeNodeType.MENU.getValue().equals(menu.getNodeType())) {
                    // 只有自定义菜单的分钟允许所属应用为空
                    return;
                }
            } else if (!appCodesAuthed.contains(menu.getMenusAppCode())){
                return;
            }
        }
        existsResult.add(menuCodeAuthed);

        var menuBO = this.convertPermissionMenuForMenuBo(menu);
        result.add(menuBO);

        // 添加上级
        if (StringUtils.hasText(menu.getMenusParentCode())) {
            this.addPermissionMenuBO(customMenu, menu.getMenusParentCode(), menusAllMap, appCodesAuthed, result, existsResult, rootMenuFunction);
        } else if (rootMenuFunction != null) {
            rootMenuFunction.accept(menu, menuBO);
        }
    }

    private List<PermissionMenuBO> convertPermissionMenuForApp(Collection<AppBO> appBoList) {
        if (CollUtil.isEmpty(appBoList)) {
            return Collections.emptyList();
        }

        List<PermissionMenuBO> menuBoList = new ArrayList<>(appBoList.size());
        for (var app : appBoList) {
            menuBoList.add(this.convertPermissionMenuBoForAppBo(app));
        }

        return menuBoList;
    }

    private PermissionMenuBO convertPermissionMenuBoForAppBo(AppBO appBO) {
        PermissionMenuBO menuBO = new PermissionMenuBO();

        menuBO.setMenusName(appBO.getAppName());
        menuBO.setAppCode(appBO.getAppCode());
        menuBO.setNodeType(MenuTreeNodeType.APP.getValue());
        menuBO.setMenusOrder(ObjUtil.defaultIfNull(appBO.getAppOrder(), 1));
        menuBO.setDisplay(true);
        menuBO.setMenusCode(appBO.getAppCode());
        return menuBO;
    }

    private PermissionMenuBO convertPermissionMenuForMenuBo(MenuBO menu) {
        PermissionMenuBO menuBO = new PermissionMenuBO();
        menuBO.setMenusName(menu.getMenusName());
        menuBO.setAppCode(menu.getMenusAppCode());
        var nodeType = MenuTreeNodeType.valueOfPlatformMenu(menu.getNodeType());
        menuBO.setNodeType(nodeType == null ? null : nodeType.getValue());
        menuBO.setDataType(menu.getNodeType());
        menuBO.setMenusCode(menu.getMenusCode());
        menuBO.setMenusOrder(menu.getMenusOrder());
        menuBO.setMenusParentCode(menu.getMenusParentCode());
        menuBO.setMenusRoute(menu.getMenusRoute());
        menuBO.setMenusIcon(menu.getMenusIcon());
        menuBO.setDisplay(menu.getDisplay());
        menuBO.setMenusIcon(menu.getMenusIcon());
        menuBO.setOuterLink(menu.getOuterLink());
        menuBO.setOuterLinkType(menu.getOuterLinkType());
        return menuBO;
    }

    private PermissionMenuBO convertPermissionMenuBoForActionBo(MenuBO menuBO) {
        PermissionMenuBO actionBO = new PermissionMenuBO();
        actionBO.setMenusName(menuBO.getMenusName());
        actionBO.setAppCode(menuBO.getMenusAppCode());
        actionBO.setNodeType(MenuTreeNodeType.ACTION.getValue());
        actionBO.setDataType(menuBO.getNodeType());
        actionBO.setMenusCode(menuBO.getMenusCode());
        actionBO.setMenusOrder(menuBO.getMenusOrder());
        actionBO.setMenusParentCode(menuBO.getMenusParentCode());
        actionBO.setMenusRoute(menuBO.getMenusRoute());
        actionBO.setMenusIcon(menuBO.getMenusIcon());
        actionBO.setDisplay(menuBO.getDisplay());
        actionBO.setOuterLink(menuBO.getOuterLink());
        actionBO.setOuterLinkType(menuBO.getOuterLinkType());
        return actionBO;
    }

    private Set<Long> obtainUserRoleIds(GeneralUserDetails userDetails, MasterUserBO masterUser) {
        var userRoles = this.obtainUserRoleId(userDetails);
        if (masterUser == null || SubUserPermissionTypeEnum.EXTENDS_ALL != masterUser.getPermissionType()) {
            return userRoles;
        }

        // 获取主账号的角色
        var mastUserRoles = userRoleRepoProc.getRolesOfUser(masterUser.getMasterUserId(), null).stream().map(IdCodeNameParam::getId).collect(Collectors.toSet());

        Set<Long> roleCodes = new HashSet<>(16);
        roleCodes.addAll(userRoles);
        roleCodes.addAll(mastUserRoles);

        return roleCodes;
    }

    private Set<Long> obtainUserRoleId(GeneralUserDetails userDetails) {
        var roleCodes = this.normalizeUserRoleCodes(userDetails);
        if (CollectionUtils.isEmpty(roleCodes)) {
            return Collections.emptySet();
        }

        return roleRepoProc.filterEnabledId(roleCodes);
    }

    private Set<String> normalizeUserRoleCodes(GeneralUserDetails userDetails) {
        if (CollectionUtils.isEmpty(userDetails.getRoleCodes())) {
            return Collections.emptySet();
        }

        Set<String> roleCodes = null;
        if (StringUtils.hasText(authorizationProperties.getRolePrefix())) {
            // 去掉前缀
            String prefix = authorizationProperties.getRolePrefix();
            roleCodes = userDetails.getRoleCodes().stream()
                    .map(t -> t.startsWith(prefix) ? t.substring(authorizationProperties.getRolePrefix().length()) : t)
                    .collect(Collectors.toSet());
        } else {
            roleCodes = new HashSet<>(userDetails.getRoleCodes());
        }
        return roleCodes;
    }

    private boolean hasCustomMenuTree(GeneralUserDetails currentUser) {
        var testCustomMenu = HttpServletUtil.currentRequest().getParameter("testCustomMenu");
        if (Boolean.parseBoolean(testCustomMenu)) {
            return true;
        }
        var tenantId = ObjectUtil.defaultIfNull(currentUser.getTenantId(), TenantConstant.DEFAULT_TENANT_ID);
        var enabled = tenantMenuRepoProc.getEnabledByTenant(tenantId);

        return enabled != null && enabled;
    }
}