package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.cas.provider.UserTransferHelper;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.constant.CommonConstant;
import com.elitescloud.boot.constant.Gender;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.cloudt.common.constant.Terminal;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.cacheable.SysCacheUserRpcService;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.constant.TenantType;
import com.elitescloud.cloudt.system.constant.UserSourceType;
import com.elitescloud.cloudt.system.convert.UserConvert;
import com.elitescloud.cloudt.system.model.entity.SysUserTerminalDO;
import com.elitescloud.cloudt.system.service.callback.UserChangedCallback;
import com.elitescloud.cloudt.system.service.model.bo.SysUserSaveBO;
import com.elitescloud.cloudt.system.service.model.bo.SysUserTypeBO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserTypeDO;
import com.elitescloud.cloudt.system.service.repo.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 用户管理公共类.
 *
 * @author Kaiser（wang shao）
 * 2022/9/23
 */
@Slf4j
@Component
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class UserMngManager {
    private static final UserConvert CONVERT = UserConvert.INSTANCE;

    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private SysUserTerminalRepo userTerminalRepo;
    @Autowired
    private SysUserTerminalRepoProc userTerminalRepoProc;
    @Autowired
    private UserTypeRepoProc userTypeRepoProc;
    @Autowired
    private UserRoleRepoProc userRoleRepoProc;
    @Autowired
    private TenantUserRepoProc tenantUserRepoProc;
    @Autowired
    private ObjectProvider<UserChangedCallback> userChangedCallbackObjectProvider;
    @Autowired
    private PasswordEncoder passwordEncoder;
    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private AuthorizationSdkProperties authorizationSdkProperties;
    @Autowired(required = false)
    private UserTransferHelper userTransferHelper;
    @Autowired
    private TenantClientProvider tenantClientProvider;
    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;
    @Autowired
    private AreaManager areaManager;

    @Autowired
    private SysCacheUserRpcService cacheUserRpcService;

    /**
     * 保存用户
     *
     * @param saveBO 保存信息
     * @return 用户信息
     */
    public SysUserDO upsert(@NotNull SysUserSaveBO saveBO) {
        Long id = saveBO.getId();

        // 保存前的校验
        SysUserDO userDO = id == null ? checkForInsert(saveBO) : checkForUpdate(saveBO, false);

        // 保存用户基本信息
        userRepoProc.save(userDO);
        // 保存用户类型
        saveUserTypes(saveBO, userDO.getId());

        // 所属租户相关操作
        tenantDataIsolateProvider.byTenantAuto(() -> {
            // 保存用户终端
            saveUserTerminal(saveBO, userDO.getId());

            return null;
        });

        // 保存后的回调
        userChangedCallbackObjectProvider.forEach(callback -> callback.onUpsert(id == null, saveBO, userDO));

        // 清空缓存
        cacheUserRpcService.clearCacheForAllTenant();

        return userDO;
    }

    /**
     * 检查用户信息
     *
     * @param saveBO 检查信息
     */
    public void check(@NotNull SysUserSaveBO saveBO) {
        if (saveBO.getId() == null) {
            this.checkForInsert(saveBO);
            return;
        }
        this.checkForUpdate(saveBO, true);
    }

    /**
     * 更新用户启用状态
     *
     * @param id      用户ID
     * @param enabled 是否启用
     */
    public void updateEnabled(long id, Boolean enabled) {
        var tenant = tenantClientProvider.getCurrentTenant();
        if (tenant == null || tenant.getType() == TenantType.OPERATION) {
            // 修改用户账号的状态
            userRepoProc.updateEnabled(id, enabled);
        } else {
            // 如果只有一个租户，则直接更新账户状态
            var boundTenantNum = tenantUserRepoProc.listTenantIdByUserID(id, null).size();
            if (boundTenantNum <= 1) {
                userRepoProc.updateEnabled(id, enabled);
            }
        }

        // 回调
        userChangedCallbackObjectProvider.forEach(callback -> callback.onEnabled(id, enabled));
        // 清空缓存
        cacheUserRpcService.clearCacheForAllTenant();
    }

    /**
     * 重置密码
     *
     * @param id 用户ID
     */
    public void resetPassword(long id) {
        this.updatePassword(id, CommonConstant.INIT_PWD, null, true);
    }

    /**
     * 修改密码
     *
     * @param id               用户ID
     * @param password         新密码
     * @param originalPassword 原密码
     * @param needReset        是否需要用户重置密码
     */
    public void updatePassword(long id, @NotBlank String password, String originalPassword, Boolean needReset) {
        // 新密码
        Assert.hasText(password, "密码不能为空");

        // 校验密码
        var validated = this.isMatchedPassword(id, password, originalPassword);
        Assert.isTrue(validated, "当前密码不正确");

        String passwordNew = encryptPassword(password);
        userRepoProc.updatePassword(id, needReset == null || needReset, passwordNew);

        // 回调处理
        userChangedCallbackObjectProvider.forEach(callback -> callback.onUpdatePassword(id, password, originalPassword));
    }

    /**
     * 更新手机号
     *
     * @param id     用户ID
     * @param mobile 手机号
     */
    public void updateMobile(long id, String mobile) {
        if (StringUtils.hasText(mobile)) {
            var existsIds = userRepoProc.getIdByMobile(mobile);
            if (existsIds.contains(id)) {
                // 无需修改
                return;
            }
            if (systemProperties.getMobileUnique()) {
                Assert.isTrue(existsIds.isEmpty(), "手机号已存在");
            }
        }

        // 更新手机号
        userRepoProc.updateMobile(id, mobile);

        // 回调处理
        userChangedCallbackObjectProvider.forEach(callback -> callback.onUpdateMobile(id, mobile));
        // 清空缓存
        cacheUserRpcService.clearCacheForAllTenant();
    }

    /**
     * 更新邮箱
     *
     * @param id    用户ID
     * @param email 邮箱
     */
    public void updateEmail(long id, String email) {
        if (StringUtils.hasText(email)) {
            var existsIds = userRepoProc.getIdByEmail(email);
            if (existsIds.contains(id)) {
                // 无需修改
                return;
            }
            if (systemProperties.getEmailUnique()) {
                Assert.isTrue(existsIds.isEmpty(), "邮箱已存在");
            }
        }

        // 更新邮箱
        userRepoProc.updateEmail(id, email);

        // 回调处理
        userChangedCallbackObjectProvider.forEach(callback -> callback.onUpdateEmail(id, email));
        // 清空缓存
        cacheUserRpcService.clearCacheForAllTenant();
    }

    /**
     * 删除用户
     *
     * @param id 用户ID
     */
    public void delete(long id) {
        var tenant = tenantClientProvider.getCurrentTenant();

        Long casUserId;
        var tenantIdsBelong = tenantUserRepoProc.listTenantIdByUserID(id, null);
        if (tenant == null || tenant.getType() == TenantType.OPERATION ||
                tenantIdsBelong.isEmpty() ||
                (tenantIdsBelong.size() == 1 && Objects.equals(tenantIdsBelong.get(0), tenant.getId()))
        ) {
            // 未启用租户或当前是运营机构
            // 没有所属租户
            // 仅有一个租户且是当前租户
            casUserId = userRepoProc.getCasUserId(id);
            userRepoProc.delete(id);
            tenantUserRepoProc.deleteByUserId(id);
        } else {
            casUserId = null;
            // 租户时
            tenantUserRepoProc.delete(tenant.getId(), id);
            // 删除用户类型
            userTypeRepoProc.deleteByUserId(id, tenant.getId());

            // 租户下相关操作
            tenantDataIsolateProvider.byTenantDirectly(() -> {
                // 删除用户终端
                userTerminalRepoProc.deleteByUserId(id);
                // 删除用户角色
                userRoleRepoProc.deleteByUser(id);
                return null;
            }, tenant);
        }

        // 回调处理
        userChangedCallbackObjectProvider.forEach(callback -> callback.onDelete(id, casUserId));
        // 清空缓存
        cacheUserRpcService.clearCacheForAllTenant();
    }

    /**
     * 更新用户身份
     *
     * @param id
     * @param userType
     * @param identityId
     */
    public void updateUserType(long id, @NotBlank String userType, String identityId) {
        // 查询当前的用户类型
        var tenantId = currentTenantId();
        var userTypeBoList = userTypeRepoProc.getUserTypeBO(id, tenantId);

        boolean isBlankOfIdentityId = CharSequenceUtil.isBlank(identityId);
        boolean existsNoIdentityId = false;
        for (SysUserTypeBO sysUserTypeBO : userTypeBoList) {
            if (!userType.equals(sysUserTypeBO.getUserType())) {
                continue;
            }

            if (isBlankOfIdentityId) {
                // 已存在对应的类型
                return;
            }
            if (identityId.equals(sysUserTypeBO.getIdentityId())) {
                // 已存在
                return;
            }

            if (CharSequenceUtil.isBlank(sysUserTypeBO.getIdentityId())) {
                existsNoIdentityId = true;
            }
        }

        // 不存在对应的用户类型，则新增
        if (isBlankOfIdentityId) {
            SysUserTypeDO userTypeDO = new SysUserTypeDO();
            userTypeDO.setUserId(id);
            userTypeDO.setType(userType);
            userTypeDO.setIdentityId(null);
            userTypeDO.setSysTenantId(tenantId);
            userTypeRepoProc.save(userTypeDO);
            return;
        }

        // 存在为空的，则更新其身份ID
        if (existsNoIdentityId) {
            userTypeRepoProc.updateIdentityId(id, tenantId, userType, identityId);
            return;
        }

        SysUserTypeDO userTypeDO = new SysUserTypeDO();
        userTypeDO.setUserId(id);
        userTypeDO.setType(userType);
        userTypeDO.setIdentityId(identityId);
        userTypeDO.setSysTenantId(tenantId);
        userTypeRepoProc.save(userTypeDO);
    }

    /**
     * 删除用户类型
     *
     * @param id
     * @param userType
     * @param identityId
     */
    public void deleteUserType(long id, @NotBlank String userType, String identityId) {
        // 查询当前的用户类型
        var tenantId = currentTenantId();

        userTypeRepoProc.deleteIdentityId(id, tenantId, userType, identityId);
    }

    /**
     * 获取回调接口
     *
     * @return
     */
    public List<UserChangedCallback> getUserChangedCallbacks() {
        return userChangedCallbackObjectProvider.stream().collect(Collectors.toList());
    }

    /**
     * 获取保存信息
     *
     * @param id
     * @return
     */
    public SysUserSaveBO getUserSaveBO(long id) {
        var userDO = userRepoProc.get(id);
        if (userDO == null) {
            return null;
        }

        var saveBO = CONVERT.do2SaveBo(userDO);
        saveBO.setAreaBO(areaManager.getAreaBO(userDO.getProvinceCode(), userDO.getCityCode(), userDO.getCountyCode()));

        var tenantId = this.currentTenantId();
        saveBO.setTerminals(new HashSet<>(userTerminalRepoProc.getTerminalStrByUserId(id)));
        saveBO.setUserTypes(new HashSet<>(userTypeRepoProc.getUserTypeBO(id, tenantId)));

        return saveBO;
    }

    private boolean isMatchedPassword(long id, String password, String originalPassword) {
        if (Boolean.TRUE.equals(authorizationSdkProperties.getCasClient().getSupportValidatePassword())) {
            // 通过认证中心校验密码
            var casUserId = userRepoProc.getCasUserId(id);
            if (casUserId != null && userTransferHelper != null) {
                log.info("通过CAS校验密码：{}", id);
                try {
                    var validateResult = userTransferHelper.validatePwd(casUserId, originalPassword, password);
                    if (Boolean.FALSE.equals(validateResult.getSuccess())) {
                        throw new BusinessException(validateResult.getMsg());
                    }
                    return true;
                } catch (Exception e) {
                    if (e instanceof BusinessException) {
                        throw e;
                    }
                    log.error("通过CAS校验密码异常：", e);
                }
            }
        }

        if (CharSequenceUtil.isBlank(originalPassword)) {
            return true;
        }

        // 默认本地校验
        var pwdEncoded = userRepoProc.getPassword(id);
        return passwordEncoder.matches(originalPassword, pwdEncoded);
    }

    private void saveUserTerminal(SysUserSaveBO saveBO, Long userId) {
        // 获取已有终端
        var terminalOld = userTerminalRepoProc.getTerminalStrByUserId(userId);
        LocalDateTime now = LocalDateTime.now();

        if (CollUtil.isNotEmpty(saveBO.getTerminalsAdd())) {
            // 有新增项，如果没有传递所有，则直接新增
            if (CollUtil.isEmpty(saveBO.getTerminals())) {
                var terminalsToAdd = saveBO.getTerminalsAdd().stream().filter(t -> !terminalOld.contains(t))
                        .map(t -> {
                            SysUserTerminalDO userTerminalDO = new SysUserTerminalDO();
                            userTerminalDO.setUserId(userId);
                            userTerminalDO.setTerminalCode(t);
                            userTerminalDO.setTimeBind(now);

                            return userTerminalDO;
                        })
                        .collect(Collectors.toList());
                if (!terminalsToAdd.isEmpty()) {
                    userTerminalRepo.saveAll(terminalsToAdd);
                }
                return;
            }

            // 添加到总的里面
            saveBO.getTerminals().addAll(saveBO.getTerminalsAdd());
        }

        if (CollectionUtils.isEmpty(saveBO.getTerminals())) {
            // 没有终端，则需要删除已有的
            if (terminalOld.isEmpty()) {
                return;
            }
            // 删除已有的
            userTerminalRepoProc.deleteByUserId(userId);
            return;
        }

        // 组织要新增的终端
        var userTerminals = saveBO.getTerminals().stream()
                .filter(t -> !terminalOld.contains(t))
                .map(t -> {
                    SysUserTerminalDO userTerminalDO = new SysUserTerminalDO();
                    userTerminalDO.setUserId(userId);
                    userTerminalDO.setTerminalCode(t);
                    userTerminalDO.setTimeBind(now);

                    return userTerminalDO;
                }).collect(Collectors.toList());
        if (!userTerminals.isEmpty()) {
            userTerminalRepo.saveAll(userTerminals);
        }

        // 删除已没有的
        if (terminalOld.isEmpty()) {
            return;
        }
        var terminalsToDel = terminalOld.stream()
                .filter(t -> !saveBO.getTerminals().contains(t))
                .collect(Collectors.toSet());
        if (!terminalsToDel.isEmpty()) {
            userTerminalRepoProc.delete(userId, terminalsToDel);
        }
    }

    private long currentTenantId() {
        var tenant = tenantClientProvider.getSessionTenant();
        if (tenant == null) {
            tenant = tenantClientProvider.getCurrentTenant();
        }
        return tenant == null ? TenantConstant.DEFAULT_TENANT_ID : tenant.getId();
    }

    private void saveUserTypes(SysUserSaveBO saveBO, Long userId) {
        var tenantId = currentTenantId();
        // 没有类型了，则直接删除
        if (CollUtil.isEmpty(saveBO.getUserTypes()) && CollUtil.isEmpty(saveBO.getUserTypesAdd())) {
            userTypeRepoProc.deleteByUserId(userId, tenantId);
            return;
        }

        // 获取已有类型
        var typeOldList = userTypeRepoProc.getUserTypeList(userId, tenantId);
        Map<String, SysUserTypeDO> typesMapOfNoId = typeOldList.isEmpty() ? Collections.emptyMap() : typeOldList.stream()
                .filter(t -> CharSequenceUtil.isBlank(t.getIdentityId())).collect(Collectors.toMap(SysUserTypeDO::getType, t -> t, (t1, t2) -> t1));
        Map<String, SysUserTypeDO> typesMapOfHasId = typeOldList.isEmpty() ? Collections.emptyMap() : typeOldList.stream()
                .filter(t -> CharSequenceUtil.isNotBlank(t.getIdentityId())).collect(Collectors.toMap(t -> t.getType() + ":" + t.getIdentityId(), t -> t, (t1, t2) -> t1));

        Set<String> typesOfNoId = new HashSet<>();
        Set<String> typesOfHasId = new HashSet<>();
        if (CollUtil.isNotEmpty(saveBO.getUserTypesAdd())) {
            for (var sysUserTypeBO : saveBO.getUserTypesAdd()) {
                // 未设置身份ID
                if (CharSequenceUtil.isBlank(sysUserTypeBO.getIdentityId())) {
                    if (typesMapOfNoId.containsKey(sysUserTypeBO.getUserType()) ||
                            typesOfNoId.contains(sysUserTypeBO.getUserType())) {
                        // 已存在该类型
                        typesOfNoId.add(sysUserTypeBO.getUserType());
                        continue;
                    }
                    typesOfNoId.add(sysUserTypeBO.getUserType());

                    SysUserTypeDO userTypeDO = new SysUserTypeDO();
                    userTypeDO.setUserId(userId);
                    userTypeDO.setType(sysUserTypeBO.getUserType());
                    userTypeDO.setIdentityId(null);
                    userTypeDO.setSysTenantId(tenantId);
                    userTypeRepoProc.save(userTypeDO);
                    continue;
                }

                // 有身份ID
                var key = sysUserTypeBO.getUserType() + ":" + sysUserTypeBO.getIdentityId();
                if (typesMapOfHasId.containsKey(key) || typesOfHasId.contains(key)) {
                    typesOfHasId.add(key);
                    continue;
                }
                typesOfHasId.add(key);
                SysUserTypeDO userTypeDO = new SysUserTypeDO();
                userTypeDO.setUserId(userId);
                userTypeDO.setType(sysUserTypeBO.getUserType());
                userTypeDO.setIdentityId(sysUserTypeBO.getIdentityId());
                userTypeDO.setSysTenantId(tenantId);
                userTypeRepoProc.save(userTypeDO);
            }
        }

        // 未设置全部
        if (CollectionUtils.isEmpty(saveBO.getUserTypes())) {
            // 删除已有ID的空记录
            Set<String> tempTypesOfNoIds = new HashSet<>(typesMapOfNoId.keySet());
            tempTypesOfNoIds.addAll(typesOfNoId);
            Set<String> tempTypesOfHasIds = new HashSet<>(typesMapOfHasId.keySet());
            tempTypesOfHasIds.addAll(typesOfHasId);
            for (var type : tempTypesOfNoIds) {
                for (String s : tempTypesOfHasIds) {
                    if (type.equals(s.split(":")[0])) {
                        userTypeRepoProc.deleteIdentityId(userId, tenantId, type, null);
                        break;
                    }
                }
            }
            return;
        }

        for (SysUserTypeBO sysUserTypeBO : saveBO.getUserTypes()) {
            // 未设置身份ID
            if (CharSequenceUtil.isBlank(sysUserTypeBO.getIdentityId())) {
                if (typesOfNoId.contains(sysUserTypeBO.getUserType())) {
                    // 已添加过
                    continue;
                }
                typesOfNoId.add(sysUserTypeBO.getUserType());
                if (typesMapOfNoId.containsKey(sysUserTypeBO.getUserType())) {
                    // 已存在该类型
                    continue;
                }
                SysUserTypeDO userTypeDO = new SysUserTypeDO();
                userTypeDO.setUserId(userId);
                userTypeDO.setType(sysUserTypeBO.getUserType());
                userTypeDO.setIdentityId(null);
                userTypeDO.setSysTenantId(tenantId);
                userTypeRepoProc.save(userTypeDO);
                continue;
            }

            // 有身份ID
            if (typesOfHasId.contains(sysUserTypeBO.getUserType() + ":" + sysUserTypeBO.getIdentityId())) {
                // 已添加过
                continue;
            }
            typesOfHasId.add(sysUserTypeBO.getUserType() + ":" + sysUserTypeBO.getIdentityId());
            if (typesMapOfHasId.containsKey(sysUserTypeBO.getUserType() + ":" + sysUserTypeBO.getIdentityId())) {
                continue;
            }
            SysUserTypeDO userTypeDO = new SysUserTypeDO();
            userTypeDO.setUserId(userId);
            userTypeDO.setType(sysUserTypeBO.getUserType());
            userTypeDO.setIdentityId(sysUserTypeBO.getIdentityId());
            userTypeDO.setSysTenantId(tenantId);
            userTypeRepoProc.save(userTypeDO);
        }

        // 需要删除的
        if (!typesMapOfNoId.isEmpty()) {
            for (Map.Entry<String, SysUserTypeDO> entry : typesMapOfNoId.entrySet()) {
                if (typesOfNoId.contains(entry.getValue().getType())) {
                    continue;
                }
                userTypeRepoProc.delete(entry.getValue().getId());
            }
        }
        if (!typesMapOfHasId.isEmpty()) {
            for (Map.Entry<String, SysUserTypeDO> entry : typesMapOfHasId.entrySet()) {
                if (typesOfHasId.contains(entry.getValue().getType() + ":" + entry.getValue().getIdentityId())) {
                    continue;
                }
                userTypeRepoProc.delete(entry.getValue().getId());
            }
        }
        // 删除已有ID的空记录
        Set<String> tempTypesOfNoIds = new HashSet<>(typesMapOfNoId.keySet());
        tempTypesOfNoIds.addAll(typesOfNoId);
        Set<String> tempTypesOfHasIds = new HashSet<>(typesMapOfHasId.keySet());
        tempTypesOfHasIds.addAll(typesOfHasId);
        for (var type : tempTypesOfNoIds) {
            for (String s : tempTypesOfHasIds) {
                if (type.equals(s.split(":")[0])) {
                    userTypeRepoProc.deleteIdentityId(userId, tenantId, type, null);
                    break;
                }
            }
        }
    }

    private SysUserDO checkForInsert(SysUserSaveBO saveBO) {
        SysUserDO userDO = CONVERT.saveBo2Do(saveBO, saveBO.getAreaBO());
        Assert.hasText(saveBO.getUsername(), "登录账号为空");
        boolean exists = userRepoProc.existsUsername(saveBO.getUsername());
        Assert.isTrue(!exists, "登录账号已存在");

        // 密码设置
        userDO.setPassword(encryptPassword(saveBO.getPassword()));
        userDO.setNeedReset(ObjectUtil.defaultIfNull(saveBO.getNeedReset(), true));
        // 姓名
        userDO.setLastName(CharSequenceUtil.blankToDefault(saveBO.getFullName(), saveBO.getUsername()));
        userDO.setFirstName(null);
        // 性别
        if (StringUtils.hasText(saveBO.getGender())) {
            Gender gender = Gender.valueOf(saveBO.getGender());
            Assert.notNull(gender, "未知性别类型" + saveBO.getGender());
        } else {
            userDO.setGender(Gender.SECRET.getValue());
        }

        // 手机号限制唯一
        userDO.setMobile(CharSequenceUtil.nullToDefault(saveBO.getMobile(), ""));
        if (systemProperties.getMobileUnique() && StringUtils.hasText(saveBO.getMobile())) {
            exists = userRepoProc.existsMobile(saveBO.getMobile());
            Assert.isTrue(!exists, "手机号已存在");
        }
        // 邮箱限制唯一
        userDO.setEmail(CharSequenceUtil.nullToDefault(saveBO.getEmail(), ""));
        if (systemProperties.getEmailUnique() && StringUtils.hasText(saveBO.getEmail())) {
            exists = userRepoProc.existsEmail(saveBO.getEmail());
            Assert.isTrue(!exists, "邮箱已存在");
        }
        // 身份证号限制唯一
        userDO.setIdCard(CharSequenceUtil.nullToDefault(saveBO.getIdCard(), ""));
        if (systemProperties.getIdCardUnique() && StringUtils.hasText(saveBO.getIdCard())) {
            exists = userRepoProc.existsIdCard(saveBO.getIdCard());
            Assert.isTrue(!exists, "身份证号已存在");
        }
        if (saveBO.getEnabled() == null) {
            userDO.setEnabled(true);
        }
        if (StringUtils.hasText(saveBO.getSourceType())) {
            var sourceType = UserSourceType.parse(saveBO.getSourceType());
            Assert.notNull(sourceType, "未知账号来源类型");
        }
        checkUserTerminal(saveBO.getTerminals());
        checkUserTerminal(saveBO.getTerminalsAdd());

        userDO.setTenantId(TenantConstant.DEFAULT_TENANT_ID);

        // 需要同步
        userDO.setSyncCas(true);

        return userDO;
    }

    private SysUserDO checkForUpdate(SysUserSaveBO saveBO, boolean checkOnly) {
        var userDO = userRepoProc.get(saveBO.getId());
        Assert.notNull(userDO, "账号不存在");

        boolean exists = false;
        // 登录账号
        Assert.hasText(saveBO.getUsername(), "登录账号为空");
        if (systemProperties.getUsernameEditable()) {
            // 账号可修改
            if (!userDO.getUsername().equals(saveBO.getUsername())) {
                exists = userRepoProc.existsUsername(saveBO.getUsername());
                Assert.isTrue(!exists, "登录账号已存在");
            }
        } else {
            // 判断账号是否已存在
            var idTemp = userRepoProc.getIdByUsername(saveBO.getUsername());
            if (idTemp != null && idTemp.longValue() != saveBO.getId()) {
                throw new IllegalArgumentException("登录账号已存在");
            }
        }

        // 密码设置
        var pwd = userDO.getPassword();
        if (StringUtils.hasText(saveBO.getPassword())) {
            // 如果传递密码，则更新密码
            pwd = encryptPassword(saveBO.getPassword());
        }
        if (saveBO.getNeedReset() == null) {
            saveBO.setNeedReset(ObjectUtil.defaultIfNull(userDO.getNeedReset(), true));
        }
        // 姓名
        userDO.setLastName(saveBO.getFullName());
        userDO.setFirstName(null);
        // 性别
        if (StringUtils.hasText(saveBO.getGender())) {
            Gender gender = Gender.valueOf(saveBO.getGender());
            Assert.notNull(gender, "未知性别类型" + saveBO.getGender());
        } else {
            saveBO.setGender(Gender.SECRET.getValue());
        }

        // 手机号限制唯一
        if (systemProperties.getMobileUnique() && StringUtils.hasText(saveBO.getMobile())
                && !saveBO.getMobile().equals(userDO.getMobile())) {
            exists = userRepoProc.existsMobile(saveBO.getMobile());
            Assert.isTrue(!exists, "手机号已存在");
        }
        // 邮箱限制唯一
        if (systemProperties.getEmailUnique() && StringUtils.hasText(saveBO.getEmail())
                && !saveBO.getEmail().equals(userDO.getEmail())) {
            exists = userRepoProc.existsEmail(saveBO.getEmail());
            Assert.isTrue(!exists, "邮箱已存在");
        }
        // 身份证号限制唯一
        if (systemProperties.getIdCardUnique() && StringUtils.hasText(saveBO.getIdCard())
                && !saveBO.getIdCard().equals(userDO.getIdCard())) {
            exists = userRepoProc.existsIdCard(saveBO.getIdCard());
            Assert.isTrue(!exists, "身份证号已存在");
        }
        if (saveBO.getEnabled() == null) {
            saveBO.setEnabled(true);
        }
        saveBO.setSourceType(userDO.getSourceType());
        checkUserTerminal(saveBO.getTerminals());
        checkUserTerminal(saveBO.getTerminalsAdd());
        if (checkOnly) {
            // 仅仅检查，不做转换
            return userDO;
        }

        CONVERT.copySaveBo2Do(saveBO, saveBO.getAreaBO(), userDO);
        userDO.setId(saveBO.getId());
        userDO.setPassword(pwd);

        // 需要同步
        userDO.setSyncCas(true);

        return userDO;
    }

    private void checkUserTerminal(Set<String> terminals) {
        if (CollUtil.isEmpty(terminals)) {
            return;
        }
        for (String terminal : terminals) {
            Assert.notNull(Terminal.parse(terminal), "未知终端类型" + terminal);
        }
    }

    /**
     * 加密密码
     *
     * @param password
     * @return
     */
    private String encryptPassword(String password) {
        if (!StringUtils.hasText(password)) {
            // 默认密码
            password = CommonConstant.INIT_PWD;
        }
        return passwordEncoder.encode(password);
    }
}
