package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.boot.model.dto.SysBusinessOperationDTO;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.convert.BusinessObjectConvert;
import com.elitescloud.cloudt.system.model.bo.BusinessOperationBO;
import com.elitescloud.cloudt.system.model.bo.OperationRequestInfoBO;
import com.elitescloud.cloudt.system.model.vo.query.businessobject.BusinessOperationPageQueryVO;
import com.elitescloud.cloudt.system.service.common.constant.OpenApiSourceEnum;
import com.elitescloud.cloudt.system.service.model.entity.QSysBusinessObjectDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysBusinessOperationDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformAppDO;
import com.elitescloud.cloudt.system.service.model.entity.SysBusinessOperationDO;
import com.google.common.base.Functions;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/6
 */
@Repository
public class BusinessOperationRepoProc extends BaseRepoProc<SysBusinessOperationDO> {
    private static final QSysBusinessOperationDO QDO = QSysBusinessOperationDO.sysBusinessOperationDO;

    public BusinessOperationRepoProc() {
        super(QDO);
    }

    /**
     * 根据OpenApiCode删除
     *
     * @param openApiCode
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteForGatherByOpenApiCode(@NotBlank String openApiCode) {
        var predicate = QDO.openApiCode.eq(openApiCode)
                .and(QDO.dataSource.in(OpenApiSourceEnum.GATHER_PULL.name(), OpenApiSourceEnum.GATHER_PUSH.name()));
        super.delete(predicate);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByCodes(@NotBlank String openApiCode, @NotEmpty Collection<String> operationCodes) {
        super.delete(QDO.operationCode.in(operationCodes).and(QDO.openApiCode.eq(openApiCode)));
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByBusinessObjectId(long businessObjectId) {
        super.delete(QDO.businessObjectId.eq(businessObjectId));
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByBusinessObjectId(Collection<Long> businessObjectIds) {
        super.delete(QDO.businessObjectId.in(businessObjectIds));
    }

    public String getPermissionRefByOperationCode(@NotBlank String operationCode) {
        return super.getValueByValue(QDO.permissionRef, QDO.operationCode, operationCode);
    }

    public String getBusinessObjectCodeByOperationCode(@NotBlank String operationCode) {
        return super.getValueByValue(QDO.businessObjectCode, QDO.operationCode, operationCode);
    }

    /**
     * 根据OpenApi的Code获取对应的业务操作的ID和Code
     *
     * @param openApiCode
     * @return
     */
    public Map<String, Long> getIdAndCodeByOpenApiCode(@NotBlank String openApiCode) {
        return super.jpaQueryFactory.select(QDO.id, QDO.operationCode)
                .from(QDO)
                .where(QDO.openApiCode.eq(openApiCode))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.operationCode), t -> t.get(QDO.id), (t1, t2) -> t1));
    }

    /**
     * 统计业务对象对应的业务操作数量
     *
     * @param businessObjectIds
     * @return
     */
    public Map<Long, Long> countByBusinessObject(Collection<Long> businessObjectIds) {
        return super.jpaQueryFactory.select(QDO.businessObjectId, QDO.businessObjectId.count())
                .from(QDO)
                .where(QDO.businessObjectId.in(businessObjectIds))
                .groupBy(QDO.businessObjectId)
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.businessObjectId), t -> t.get(QDO.businessObjectId.count()), (t1, t2) -> t1));
    }

    /**
     * 统计支持权限的业务对象
     *
     * @return
     */
    public Map<Long, Long> countBusinessObjectOfPermissionEnabled() {
        return super.jpaQueryFactory.select(QDO.businessObjectId, QDO.businessObjectId.count())
                .from(QDO)
                .where(predicateForPermission())
                .groupBy(QDO.businessObjectId)
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.businessObjectId), t -> t.get(QDO.businessObjectId.count()), (t1, t2) -> t1));
    }

    public BusinessOperationBO getSimpleBO(@NotBlank String operationCode) {
        return super.jpaQueryFactory.select(qBeanSimpleBO())
                .from(QDO)
                .where(QDO.operationCode.eq(operationCode))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据OpenApi的Code获取对应的业务操作
     *
     * @param openApiCode
     * @return
     */
    public List<SysBusinessOperationDO> listByOpenApiCode(@NotBlank String openApiCode) {
        return super.getList(QDO.openApiCode.eq(openApiCode));
    }

    /**
     * 根据业务对象ID获取对应的业务操作
     *
     * @param businessObjectId 业务对象ID
     * @return
     */
    public List<IdCodeNameParam> listByBusinessObjectId(long businessObjectId) {
        return super.jpaQueryFactory.select(QDO.id, QDO.operationCode, QDO.operationDescription, QDO.customName)
                .from(QDO)
                .where(QDO.businessObjectId.eq(businessObjectId))
                .fetch()
                .stream()
                .map(t -> new IdCodeNameParam(t.get(QDO.id), t.get(QDO.operationCode), CharSequenceUtil.blankToDefault(t.get(QDO.customName), t.get(QDO.operationDescription))))
                .collect(Collectors.toList());
    }

    public PagingVO<SysBusinessOperationDO> pageMng(BusinessOperationPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.businessObjectId, queryVO.getBusinessObjectId())
                .andLike(QDO.operationCode, queryVO.getOperationCode())
                .andLike(QDO.operationDescription, queryVO.getOperationDescription())
                .andLike(QDO.apiUrl, queryVO.getApiUrl())
                .andEq(QDO.permissionEnabled, queryVO.getPermissionEnabled())
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest());
    }

    public List<BusinessOperationBO> listSimpleByAppCode(@NotEmpty Collection<String> appCodes,
                                                         Boolean enabled, boolean permissionOnly) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.appCode, appCodes)
                .and(permissionOnly, this::predicateForPermission)
                .andEq(QDO.enabled, enabled)
                .build();
        var dataList = super.jpaQueryFactory.select(qBeanSimpleBO())
                .from(QDO)
                .where(predicate)
                .fetch();
        this.fillOperationBo(dataList, predicate);
        return dataList;
    }

    public List<BusinessOperationBO> listSimple(@NotEmpty Collection<String> operationCodes, Boolean enabled) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.permissionRef, operationCodes)
                .andEq(QDO.enabled, enabled)
                .build();
        var dataList = super.getList(this.qBeanSimpleBO(), predicate);
        this.fillOperationBo(dataList, predicate);
        return dataList;
    }

    public List<OperationRequestInfoBO> listRequestInfo(@NotEmpty Collection<String> operationCodes, Boolean enabled) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.operationCode, operationCodes)
                .andEq(QDO.enabled, enabled)
                .build();
        return super.jpaQueryFactory.select(qBeanRequestInfoBO())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    public List<BusinessOperationBO> listSimpleByRef(@NotEmpty Collection<String> operationCodes) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.permissionRef, operationCodes)
                .andEq(QDO.enabled, true)
                .andEq(QDO.permissionEnabled, true)
                .build();
        var dataList = super.getList(this.qBeanSimpleBO(), predicate);
        this.fillOperationBo(dataList, predicate);
        return dataList;
    }

    public List<BusinessOperationBO> listSimpleByType(boolean permissionOnly) {
        var predicate = PredicateBuilder.builder()
                .and(permissionOnly, this::predicateForPermission)
                .build();
        var dataList = super.getList(this.qBeanSimpleBO(), predicate);
        this.fillOperationBo(dataList, predicate);
        return dataList;
    }

    public List<SysBusinessOperationDTO> allDtoList(Boolean enabled) {
        return super.all()
                .stream()
                .filter(t -> {
                    if (enabled == null) {
                        return true;
                    }
                    return t.getEnabled() == enabled;
                })
                .map(t -> {
                    SysBusinessOperationDTO operationDTO = BusinessObjectConvert.INSTANCE.do2Dto(t);
                    operationDTO.setOperationDescription(CharSequenceUtil.blankToDefault(t.getCustomName(), t.getOperationDescription()));

                    return operationDTO;
                }).collect(Collectors.toList());
    }

    private void fillOperationBo(List<BusinessOperationBO> operationList, Predicate predicate) {
        if (operationList.isEmpty()) {
            return;
        }

        var existsRef = operationList.stream().map(BusinessOperationBO::getPermissionRef).anyMatch(StringUtils::hasText);
        if (existsRef) {
            var operationCodeNameMap = super.jpaQueryFactory.select(QDO.operationCode, QDO.operationDescription, QDO.customName)
                    .from(QDO)
                    .where(predicate)
                    .fetch()
                    .stream()
                    .collect(Collectors.toMap(t -> t.get(QDO.operationCode), t -> CharSequenceUtil.blankToDefault(t.get(QDO.customName), t.get(QDO.operationDescription)), (t1, t2) -> t1));
            operationList.parallelStream()
                    .filter(t -> StringUtils.hasText(t.getPermissionRef()))
                    .forEach(t -> t.setPermissionRefName(operationCodeNameMap.get(t.getPermissionRef())));
        }

        // 业务对象信息
        var businessObjectCodes = operationList.stream().map(BusinessOperationBO::getBusinessObjectCode).collect(Collectors.toSet());
        if (!businessObjectCodes.isEmpty()) {
            QSysBusinessObjectDO QDO_BO = QSysBusinessObjectDO.sysBusinessObjectDO;
            QSysPlatformAppDO QDO_APP = QSysPlatformAppDO.sysPlatformAppDO;
            var obMap = jpaQueryFactory.select(QDO_BO.code, QDO_BO.name, QDO_BO.customName, QDO_APP.appName)
                    .from(QDO_BO).leftJoin(QDO_APP).on(QDO_APP.appCode.eq(QDO_BO.appCode))
                    .where(QDO_BO.code.in(businessObjectCodes))
                    .fetch()
                    .stream()
                    .collect(Collectors.toMap(t -> t.get(QDO_BO.code), Functions.identity(), (t1, t2) -> t1));
            for (BusinessOperationBO operationBO : operationList) {
                ObjUtil.ifNotNull(obMap.get(operationBO.getBusinessObjectCode()), tuple -> {
                    operationBO.setBusinessObjectName(CharSequenceUtil.blankToDefault(tuple.get(QDO_BO.customName), tuple.get(QDO_BO.name)));
                    operationBO.setAppName(tuple.get(QDO_APP.appName));
                });
            }
        }
    }

    private Predicate predicateForPermission() {
        return QDO.operationType.in(SysBusinessOperationDO.TYPES_SELECT).and(QDO.permissionEnabled.eq(true));
    }

    private QBean<BusinessOperationBO> qBeanSimpleBO() {
        return Projections.bean(BusinessOperationBO.class, QDO.id, QDO.businessObjectCode, QDO.businessObjectId,
                QDO.operationCode, QDO.appCode, QDO.operationDescription, QDO.customName, QDO.displayOrder, QDO.apiName,
                QDO.apiDescription, QDO.apiUrl, QDO.apiMethod, QDO.enabled, QDO.permissionRef);
    }

    private QBean<OperationRequestInfoBO> qBeanRequestInfoBO() {
        return Projections.bean(OperationRequestInfoBO.class, QDO.operationCode, QDO.apiMethod, QDO.apiUrl, QDO.permissionRef);
    }
}
