package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.model.bo.BusinessParamBO;
import com.elitescloud.cloudt.system.service.model.entity.QSysBusinessObjectDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysBusinessOperationDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysBusinessParamDO;
import com.elitescloud.cloudt.system.service.model.entity.SysBusinessParamDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import java.util.Collection;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/6
 */
@Repository
public class BusinessParamRepoProc extends BaseRepoProc<SysBusinessParamDO> {
    private static final QSysBusinessParamDO QDO = QSysBusinessParamDO.sysBusinessParamDO;
    private static final QSysBusinessObjectDO QDO_BUSINESS_OBJECT = QSysBusinessObjectDO.sysBusinessObjectDO;
    private static final QSysBusinessOperationDO QDO_OPERATION = QSysBusinessOperationDO.sysBusinessOperationDO;
    private static final QSysBusinessOperationDO QDO_BUSINESS_OPERATION = QSysBusinessOperationDO.sysBusinessOperationDO;

    public BusinessParamRepoProc() {
        super(QDO);
    }

    /**
     * 删除操作已不存在的
     *
     * @param openApiCode
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteForGatherByOpenApiCode(@NotBlank String openApiCode) {
        var subExpression = JPAExpressions.select(QDO_BUSINESS_OPERATION.id)
                .from(QDO_BUSINESS_OPERATION)
                .where(QDO_BUSINESS_OPERATION.id.eq(QDO.businessOperationId))
                .notExists();
        super.delete(QDO.openApiCode.eq(openApiCode)
                .and(QDO.businessOperationId.eq(SysBusinessParamDO.DEFAULT_ID).or(subExpression)));
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByBusinessObject(long businessObjectId, Long operationId) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.businessObjectId, businessObjectId)
                .andEq(QDO.businessOperationId, operationId)
                .build();
        super.delete(predicate);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteForOperation(long businessObjectId) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.businessObjectId, businessObjectId)
                .andNe(QDO.businessOperationId, SysBusinessParamDO.DEFAULT_ID)
                .build();
        super.delete(predicate);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteForOperation(Collection<Long> businessObjectIds) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.businessObjectId, businessObjectIds)
                .andNe(QDO.businessOperationId, SysBusinessParamDO.DEFAULT_ID)
                .build();
        super.delete(predicate);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByBusinessObject(Collection<Long> businessObjectIds) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.businessObjectId, businessObjectIds)
                .build();
        super.delete(predicate);
    }

    public void updateDataPermissionEnabled(long id, boolean enabled) {
        super.updateValue(QDO.dataPermissionEnabled, enabled, id);
    }

    public void updateFieldPermissionEnabled(long id, boolean enabled) {
        super.updateValue(QDO.fieldPermissionEnabled, enabled, id);
    }

    public List<SysBusinessParamDO> listByOpenApiCode(@NotBlank String openApiCode) {
        return super.getList(QDO.openApiCode.eq(openApiCode));
    }

    public List<SysBusinessParamDO> listByBusinessObjectId(long businessObjectId) {
        return super.getList(QDO.businessObjectId.eq(businessObjectId));
    }

    public List<SysBusinessParamDO> listByBusinessObjectId(long businessObjectId, Collection<String> fieldNames) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.businessObjectId, businessObjectId)
                .andIn(QDO.fieldName, fieldNames)
                .build();
        return super.getList(predicate, QDO.sortNo.asc());
    }

    public List<BusinessParamBO> listSimpleBoByBusinessObjectCode(@NotBlank String businessObjectCode) {
        var predicate = PredicateBuilder.builder()
                .andEq(true, QDO.businessObjectId,
                        JPAExpressions.select(QDO_BUSINESS_OBJECT.id)
                                .from(QDO_BUSINESS_OBJECT)
                                .where(QDO_BUSINESS_OBJECT.code.eq(businessObjectCode))
                )
                .build();

        return super.jpaQueryFactory.select(qBeanSimpleBO())
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.sortNo.asc())
                .fetch();
    }

    public List<BusinessParamBO> listRelatedFieldByBusinessObjectCode(@NotBlank String businessObjectCode) {
        var predicate = PredicateBuilder.builder()
                .andEq(true, QDO.businessObjectId,
                        JPAExpressions.select(QDO_BUSINESS_OBJECT.id)
                                .from(QDO_BUSINESS_OBJECT)
                                .where(QDO_BUSINESS_OBJECT.code.eq(businessObjectCode))
                )
                .andNotNull(true, QDO.relatedBusinessObject)
                .build();

        return super.jpaQueryFactory.select(qBeanSimpleBO())
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.sortNo.asc())
                .fetch();
    }

    public List<BusinessParamBO> listSimpleBoByBusinessOperationCode(@NotBlank String operationCode, Collection<String> fieldNames) {
        var predicate = PredicateBuilder.builder()
                .andEq(true, QDO.businessObjectId, JPAExpressions.select(QDO_OPERATION.businessObjectId).from(QDO_OPERATION).where(QDO_OPERATION.operationCode.eq(operationCode)))
                .andIn(QDO.fieldName, fieldNames)
                .build();

        return super.jpaQueryFactory.select(qBeanSimpleBO())
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.sortNo.asc())
                .fetch();
    }

    private QBean<BusinessParamBO> qBeanSimpleBO() {
        return Projections.bean(BusinessParamBO.class, QDO.id, QDO.enabled, QDO.entityClassName, QDO.tableName, QDO.fieldName, QDO.fieldJavaType, QDO.fieldDescription
                , QDO.customDescription, QDO.relatedBusinessObject, QDO.relatedField, QDO.valuesJson, QDO.userType, QDO.dataPermissionEnabled,
                QDO.fieldPermissionEnabled, QDO.baseField);
    }
}
