package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.model.bo.PermissionBO;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.model.entity.*;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/13
 */
@Repository
public class RolePermissionRepoProc extends BaseRepoProc<SysRolePermissionDO> {
    private static final QSysRolePermissionDO QDO = QSysRolePermissionDO.sysRolePermissionDO;
    private static final QSysRoleDO QDO_ROLE = QSysRoleDO.sysRoleDO;
    private static final QSysTenantMenuTreeDO QDO_TENANT_MENU_TREE = QSysTenantMenuTreeDO.sysTenantMenuTreeDO;

    public RolePermissionRepoProc() {
        super(QDO);
    }

    /**
     * 根据角色删除
     *
     * @param roleId
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteByRole(Long roleId) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.roleId.eq(roleId))
                .execute();
    }

    /**
     * 根据角色删除
     *
     * @param roleId         角色ID
     * @param appCode        应用编码
     * @param permissionType 权限类型
     * @param custom         是否自定义
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteByRole(long roleId, @NotBlank String appCode, @NotEmpty Collection<String> permissionType, boolean custom) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.roleId.eq(roleId).and(QDO.appCode.eq(appCode)).and(QDO.permissionType.in(permissionType)).and(QDO.custom.eq(custom)))
                .execute();
    }

    /**
     * 根据角色删除
     *
     * @param roleId         角色ID
     * @param appCode        应用编码
     * @param permissionType 权限类型
     * @param custom         是否自定义
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteByRole(long roleId, @NotBlank String appCode, @NotBlank String permissionType,
                             @NotEmpty Collection<String> permissionCodes, Boolean custom) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.permissionCode, permissionCodes)
                .andEq(QDO.permissionType, permissionType)
                .andEq(QDO.appCode, appCode)
                .andEq(QDO.roleId, roleId)
                .andEq(QDO.custom, custom)
                .build();

        super.delete(predicate);
    }

    /**
     * 根据角色获取权限
     *
     * @param roleId
     * @return
     */
    public List<SysRolePermissionDO> listByRole(long roleId) {
        return super.getListByValue(QDO.roleId, roleId);
    }

    /**
     * 根据角色获取权限编码
     *
     * @param roleId
     * @param permissionType
     * @return
     */
    public List<String> getCodeByRole(Long roleId, Collection<String> permissionType) {
        return jpaQueryFactory.select(QDO.permissionCode)
                .from(QDO)
                .where(QDO.roleId.eq(roleId).and(QDO.permissionType.in(permissionType)))
                .fetch();
    }

    /**
     * 根据角色获取权限
     *
     * @param roleId
     * @param permissionType
     * @return
     */
    public List<PermissionBO> getPermissionByRole(Long roleId, String appCode, Collection<String> permissionType, Boolean custom) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.roleId, roleId)
                .andEq(QDO.appCode, appCode)
                .andIn(QDO.permissionType, permissionType)
                .andEq(QDO.custom, custom)
                .build();
        return jpaQueryFactory.select(Projections.bean(PermissionBO.class, QDO.permissionCode, QDO.permissionType))
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 根据角色查询权限
     *
     * @param roleCodes
     * @param permissionType
     * @param custom
     * @return
     */
    public List<PermissionBO> queryPermissionByRole(Collection<String> roleCodes, Collection<String> permissionType, Boolean custom) {
        var roleIdsQuery = JPAExpressions.select(QDO_ROLE.id)
                .from(QDO_ROLE)
                .where(QDO_ROLE.code.in(roleCodes).and(QDO_ROLE.enabled.eq(true)));
        var predicate = PredicateBuilder.builder()
                .andIn(true, QDO.roleId, roleIdsQuery)
                .andIn(QDO.permissionType, permissionType)
                .andEq(QDO.custom, custom)
                .build();

        return super.getList(this.qBeanPermissionBO(), predicate);
    }

    /**
     * 根据角色查询自定义的权限菜单
     *
     * @param roleCodes
     * @param permissionType
     * @return
     */
    public List<SysTenantMenuTreeDO> queryMenuForCustom(Collection<String> roleCodes, Collection<String> permissionType) {
        // 根据角色编码查询角色ID
        var roleIdsQuery = JPAExpressions.select(QDO_ROLE.id)
                .from(QDO_ROLE)
                .where(QDO_ROLE.code.in(roleCodes).and(QDO_ROLE.enabled.eq(true)));

        return jpaQueryFactory.select(QDO_TENANT_MENU_TREE)
                .from(QDO_TENANT_MENU_TREE)
                .leftJoin(QDO).on(QDO.permissionCode.eq(QDO_TENANT_MENU_TREE.menuCode))
                .where(QDO.roleId.in(roleIdsQuery).and(QDO.permissionType.in(permissionType)).and(QDO.custom.eq(true)))
                .fetch();
    }

    public List<String> queryPermissionCodesByRoles(@NotEmpty Collection<Long> roleIds, boolean includeMenu, boolean includeAction) {
        Set<String> permissionTypes = new HashSet<>(4);
        if (includeMenu) {
            permissionTypes.add(MenuTreeNodeType.MENU.getValue());
        }
        if (includeAction) {
            permissionTypes.add(MenuTreeNodeType.ACTION.getValue());
        }
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.roleId, roleIds)
                .andIn(QDO.permissionType, permissionTypes)
                .build();

        return super.getValueList(QDO.permissionCode, predicate);
    }

    /**
     * 根据角色查询权限编码
     *
     * @param roleCodes
     * @param permissionType
     * @param custom
     * @return
     */
    public Set<String> queryPermissionCodeByRoles(Collection<String> roleCodes, Collection<String> permissionType, boolean custom) {
        return queryPermissionCodeByRoles(roleCodes, permissionType, null, custom);
    }

    /**
     * 根据角色查询权限编码
     *
     * @param roleCodes
     * @param permissionType
     * @param custom
     * @return
     */
    public Set<String> queryPermissionCodeByRoles(Collection<String> roleCodes, Collection<String> permissionType, Collection<String> permissionCodes, boolean custom) {
        var predicate = PredicateBuilder.builder()
                .and(CollUtil.isNotEmpty(roleCodes), () -> QDO.roleId.in(JPAExpressions.select(QDO_ROLE.id)
                        .from(QDO_ROLE)
                        .where(QDO_ROLE.code.in(roleCodes).and(QDO_ROLE.enabled.eq(true)))))
                .andIn(CollUtil.isNotEmpty(permissionCodes), QDO.permissionCode, permissionCodes)
                .andIn(CollUtil.isNotEmpty(permissionType), QDO.permissionType, permissionType)
                .andEq(true, QDO.custom, custom)
                .build();

        return new HashSet<>(jpaQueryFactory.select(QDO.permissionCode)
                .from(QDO)
                .where(predicate)
                .fetch())
                ;
    }

    /**
     * 查询API编码对应的角色编码
     *
     * @param custom
     * @return
     */
    public Map<String, Set<String>> queryRoleCodeForApi(boolean custom) {
        return jpaQueryFactory.select(QDO.permissionCode, QDO_ROLE.code)
                .from(QDO).leftJoin(QDO_ROLE).on(QDO_ROLE.id.eq(QDO.roleId))
                .where(QDO.permissionType.eq(MenuTreeNodeType.API.getValue()).and(QDO.custom.eq(custom)).and(QDO_ROLE.enabled.eq(true)))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.permissionCode),
                        Collectors.collectingAndThen(Collectors.toList(), t -> t.stream().map(tt -> tt.get(QDO_ROLE.code)).collect(Collectors.toSet()))))
                ;
    }

    private QBean<PermissionBO> qBeanPermissionBO() {
        return Projections.bean(PermissionBO.class, QDO.appCode, QDO.permissionCode, QDO.permissionType);
    }
}
