package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformNumberRuleDO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformNumberRuleDO;
import com.elitescloud.cloudt.system.model.bo.SysSeqRuleBO;
import com.elitescloud.cloudt.system.model.vo.query.sequence.SeqRulePageQueryVO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/11/14
 */
@Repository
public class SeqRuleRepoProc extends BaseRepoProc<SysPlatformNumberRuleDO> {
    private static final QSysPlatformNumberRuleDO QDO = QSysPlatformNumberRuleDO.sysPlatformNumberRuleDO;

    public SeqRuleRepoProc() {
        super(QDO);
    }

    /**
     * 更新启用状态
     *
     * @param id
     * @param enabled
     */
    public void updateEnabled(@NotNull Long id, boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    public void deleteByRuleCode(@NotBlank String appCode, @NotBlank String ruleCode) {
        super.jpaQueryFactory.delete(QDO)
                .where(QDO.ruleCode.eq(ruleCode).and(QDO.appCode.eq(appCode)));
    }

    /**
     * 获取启用状态
     *
     * @param id
     * @return
     */
    public Boolean getEnabled(@NotNull Long id) {
        return super.getValue(QDO.enabled, id);
    }

    /**
     * 根据规则编码获取ID
     *
     * @param ruleCode
     * @return
     */
    public Long getIdByRuleCode(@NotBlank String appCode, @NotBlank String ruleCode) {
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.ruleCode.eq(ruleCode).and(QDO.appCode.eq(appCode)))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据规则编码获取ID
     *
     * @param ruleCode
     * @return
     */
    public Long getIdByRuleCode(@NotBlank String ruleCode) {
        return super.getIdByValue(QDO.ruleCode, ruleCode);
    }

    /**
     * 判断编号是否存在
     *
     * @param appCode
     * @param ruleCode
     * @return
     */
    public boolean existsRuleCode(@NotBlank String appCode, @NotBlank String ruleCode) {
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.ruleCode.eq(ruleCode).and(QDO.appCode.eq(appCode)))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 判断编号是否存在
     *
     * @param ruleCode
     * @return
     */
    public boolean existsRuleCode(@NotBlank String ruleCode) {
        return super.exists(QDO.ruleCode, ruleCode);
    }

    /**
     * 获取所有的编码
     *
     * @param appCode
     * @return
     */
    public List<String> allCodes(@NotBlank String appCode) {
        return jpaQueryFactory.select(QDO.ruleCode)
                .from(QDO)
                .where(QDO.appCode.eq(appCode))
                .fetch();
    }

    /**
     * 分页查询
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysPlatformNumberRuleDO> pageMng(SeqRulePageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(StringUtils.hasText(queryVO.getAppCode()), QDO.appCode, queryVO.getAppCode())
                .andLike(StringUtils.hasText(queryVO.getRuleCode()), QDO.ruleCode, queryVO.getRuleCode())
                .andLike(StringUtils.hasText(queryVO.getRuleName()), QDO.ruleName, queryVO.getRuleName())
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.createTime.desc());
    }

    /**
     * 分页查询
     *
     * @param page
     * @param pageSize
     * @param appCode
     * @return
     */
    public PagingVO<SysSeqRuleBO> pageBo(int page, int pageSize, @NotBlank String appCode) {
        var jpaQuery = jpaQueryFactory.select(qBeanBo())
                .from(QDO)
                .where(QDO.appCode.eq(appCode));
        return super.queryByPage(jpaQuery, super.ofPage(page, pageSize));
    }

    public SysSeqRuleBO getBo(long id) {
        return super.jpaQueryFactory.select(qBeanBo())
                .from(QDO)
                .where(QDO.id.eq(id))
                .limit(1)
                .fetchOne();
    }

    public SysSeqRuleBO getBo(@NotBlank String appCode, @NotBlank String ruleCode) {
        return super.jpaQueryFactory.select(qBeanBo())
                .from(QDO)
                .where(QDO.ruleCode.eq(ruleCode).and(QDO.appCode.eq(appCode)))
                .limit(1)
                .fetchOne();
    }

    public SysPlatformNumberRuleDO get(@NotBlank String appCode, @NotBlank String ruleCode) {
        return super.jpaQueryFactory.selectFrom(QDO)
                .where(QDO.ruleCode.eq(ruleCode).and(QDO.appCode.eq(appCode)))
                .limit(1)
                .fetchOne();
    }

    private QBean<SysSeqRuleBO> qBeanBo() {
        return Projections.bean(SysSeqRuleBO.class, QDO.id, QDO.appCode, QDO.ruleCode, QDO.ruleName,
                QDO.sampleCode, QDO.enabled, QDO.remark, QDO.tenantCustom);
    }
}
