package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.TmplImportQueryParam;
import com.elitescloud.cloudt.system.service.model.entity.QSysTmplDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysTmplImportRecordDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTmplImportRecordDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotEmpty;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/5/31
 */
@Repository
public class SysTmplImportRecordRepoProc extends BaseRepoProc<SysTmplImportRecordDO> {

    private static final QSysTmplImportRecordDO QDO = QSysTmplImportRecordDO.sysTmplImportRecordDO;
    private static final QSysTmplDO QDO_TMPL = QSysTmplDO.sysTmplDO;

    public SysTmplImportRecordRepoProc() {
        super(QDO);
    }

    /**
     * 更新导入结果
     *
     * @param id       记录ID
     * @param numTotal 总记录数
     */
    public void updateNumTotal(long id, Long numTotal) {
        super.updateValue(QDO.numTotal, numTotal, id);
    }

    /**
     * 更新导入结果
     *
     * @param id         记录ID
     * @param suc        是否成功
     * @param failReason 失败原因
     */
    public void updateImportResult(Long id, Boolean suc, String failReason, String customTipMsg, Long numSuc, String fileCode, String costTimeInfo) {
        var jpaUpdate = super.jpaQueryFactory.update(QDO)
                .set(QDO.finish, true)
                .set(QDO.succ, suc)
                .set(QDO.timeFinish, LocalDateTime.now())
                .set(QDO.numSuc, numSuc)
                .set(QDO.failReason, failReason)
                .set(QDO.customTipMsg, customTipMsg)
                .set(QDO.costTimeInfo, costTimeInfo)
                ;
        if (StringUtils.hasText(fileCode)) {
            jpaUpdate.set(QDO.fileCode, fileCode);
        }

        jpaUpdate.where(QDO.id.eq(id).and(QDO.finish.eq(false)))
                .execute();
    }

    /**
     * 更新失败记录的文件编码
     *
     * @param id           记录ID
     * @param failFileCode 文件编码
     */
    public void updateFailFileCode(long id, String failFileCode) {
        super.updateValue(QDO.failFileCode, failFileCode, id);
    }

    /**
     * 更新结果
     *
     * @param ids 记录标识
     * @param msg 异常信息
     */
    public void updateResultForError(@NotEmpty Collection<Long> ids, String msg) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.succ, false)
                .set(QDO.finish, true)
                .set(QDO.timeFinish, LocalDateTime.now())
                .set(QDO.failReason, msg)
                .where(QDO.id.in(ids))
                .execute();
    }

    /**
     * 获取未结束的记录ID
     *
     * @return 记录ID
     */
    public List<Long> getUnfinished(String appCode) {
        var predicate = PredicateBuilder.builder()
                .and(StringUtils.hasText(appCode), () -> QDO.tmplId.in(JPAExpressions.select(QDO_TMPL.id).from(QDO_TMPL).where(QDO_TMPL.appCode.eq(appCode))))
                .andEq(QDO.finish, false)
                .build();
        return super.jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 获取记录的文件标识
     *
     * @param id 记录ID
     * @return 文件标识
     */
    public String getFileCode(long id) {
        return super.getValue(QDO.fileCode, id);
    }

    /**
     * 获取记录的文件标识
     *
     * @param id 记录ID
     * @return 文件标识
     */
    public Long getTmplId(long id) {
        return super.getValue(QDO.tmplId, id);
    }

    /**
     * 获取导入总数
     *
     * @param id 导入记录ID
     * @return 导入总数
     */
    public Long getNumTotal(long id) {
        return super.getValue(QDO.numTotal, id);
    }

    /**
     * 分页查询导出记录
     *
     * @param tmplId     模板ID
     * @param queryParam 查询参数
     * @return
     */
    public PagingVO<SysTmplImportRecordDO> pageMng(Long tmplId, TmplImportQueryParam queryParam) {
        Predicate predicate = PredicateBuilder.builder()
                .andEq(QDO.tmplId, tmplId)
                .andEq(QDO.userId, queryParam.getUserId())
                .build();

        var pageRequest = queryParam.getPageRequest();
        var sort = pageRequest.getSort();
        if (sort.isUnsorted()) {
            // 如果没有传递排序字段，则默认按时间排序
            sort = Sort.by(Sort.Direction.DESC, QDO.timeImport.getMetadata().getName());
            pageRequest = PageRequest.of(pageRequest.getPageNumber(), pageRequest.getPageSize(), sort);
        }

        return super.queryByPage(predicate, pageRequest);
    }

    /**
     * 查询最近的记录列表
     *
     * @param tmplId        模板ID
     * @param hoursRecently 最近几个小时
     * @param userId        用户ID
     * @return 记录列表
     */
    public List<SysTmplImportRecordDO> queryRecentlyList(Long tmplId, Integer hoursRecently, Long userId) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.tmplId, tmplId)
                .andAfter(hoursRecently != null, QDO.timeImport, LocalDateTime.now().minusHours(hoursRecently))
                .andEq(QDO.userId, userId)
                .build();
        return super.jpaQueryFactory.selectFrom(QDO)
                .where(predicate)
                .fetch();
    }
}
