package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.model.bo.TenantMenuBO;
import com.elitescloud.cloudt.system.model.vo.resp.menu.CustomMenuNodeDetailRespVO;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantMenuTreeDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantMenuTreeDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/14
 */
@Repository
public class TenantMenuTreeRepoProc extends BaseRepoProc<SysTenantMenuTreeDO> {
    private static final QSysTenantMenuTreeDO QDO = QSysTenantMenuTreeDO.sysTenantMenuTreeDO;

    public TenantMenuTreeRepoProc() {
        super(QDO);
    }

    public void updateEnabled(long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    public void deleteAll() {
        super.jpaQueryFactory.delete(QDO)
                .execute();
    }

    /**
     * 租户ID
     */
    public void deleteByTenantId() {
        super.delete((Predicate) null);
    }

    public void deleteBoundMenuCodes(@NotBlank String parentMenuCode) {
        super.delete(QDO.parentMenuCode.eq(parentMenuCode).and(QDO.nodeType.eq(MenuTreeNodeType.MENU.getValue())));
    }

    public void updateMenuName(long id, String menuName) {
        super.updateValue(QDO.menuName, menuName, id);
    }

    public void updateParent(long id, String parentCode, Integer sortNo) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.parentMenuCode, parentCode)
                .set(QDO.sortNo, sortNo)
                .where(QDO.id.eq(id))
                .execute();
    }

    public boolean existsCustomMenuCode(@NotBlank String menuCode, Long id) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.menuCode, menuCode)
                .andEq(QDO.custom, true)
                .andNe(QDO.id, id)
                .build();
        return super.exists(predicate);
    }

    public List<String> existsChildren(@NotEmpty Collection<Long> ids) {
        QSysTenantMenuTreeDO CHILD = new QSysTenantMenuTreeDO("child");
        return jpaQueryFactory.select(QDO.menuName)
                .from(QDO)
                .where(
                        QDO.id.in(ids).and(JPAExpressions.select(CHILD.id).from(CHILD).where(CHILD.parentMenuCode.eq(QDO.menuCode)).exists())
                ).fetch();
    }

    public List<Long> listChildrenIds(@NotEmpty Collection<Long> ids) {
        QSysTenantMenuTreeDO CHILD = new QSysTenantMenuTreeDO("child");
        return jpaQueryFactory.select(CHILD.id)
                .from(CHILD)
                .where(CHILD.parentMenuCode.in(
                        JPAExpressions.select(QDO.menuCode).from(QDO).where(QDO.id.in(ids))
                ))
                .fetch();
    }

    public List<CustomMenuNodeDetailRespVO> queryByParentMenuCode(@NotBlank String parentMenuCode) {
        QSysTenantMenuTreeDO CHILD = new QSysTenantMenuTreeDO("child");
        var countExp = JPAExpressions.select(CHILD.count().as("childrenNum")).from(CHILD).where(CHILD.parentMenuCode.eq(QDO.menuCode));

        return super.jpaQueryFactory.select(QDO.id, QDO.nodeType, QDO.parentMenuCode,
                        QDO.menuCode, QDO.menuName, QDO.enabled, QDO.sortNo, QDO.icon, countExp)
                .from(QDO)
                .where(QDO.parentMenuCode.eq(parentMenuCode))
                .fetch()
                .stream()
                .map(t -> {
                    CustomMenuNodeDetailRespVO respVO = new CustomMenuNodeDetailRespVO();
                    respVO.setId(t.get(QDO.id));
                    respVO.setNodeType(t.get(QDO.nodeType));
                    respVO.setParentMenuCode(t.get(QDO.parentMenuCode));
                    respVO.setMenuCode(t.get(QDO.menuCode));
                    respVO.setMenuName(t.get(QDO.menuName));
                    respVO.setEnabled(t.get(QDO.enabled));
                    respVO.setSortNo(ObjectUtil.defaultIfNull(t.get(QDO.sortNo), 0));
                    respVO.setIcon(t.get(QDO.icon));
                    respVO.setChildrenNum(t.get(countExp));

                    return respVO;
                }).sorted(Comparator.comparingInt(CustomMenuNodeDetailRespVO::getSortNo))
                .collect(Collectors.toList());
    }

    public List<IdCodeNameParam> listGroup() {
        var qBean = Projections.bean(IdCodeNameParam.class, QDO.id, QDO.menuCode.as("code"), QDO.menuName.as("name"));
        return super.getList(qBean, QDO.nodeType.eq(MenuTreeNodeType.MENU_GROUP.getValue()).and(QDO.parentMenuCode.isNull().or(QDO.parentMenuCode.isEmpty())),
                QDO.sortNo.asc());
    }

    public List<CodeNameParam> getBoundMenu(@NotBlank String parentMenuCode) {
        var qBean = Projections.bean(CodeNameParam.class, QDO.menuCode.as("code"), QDO.menuName.as("name"));
        return super.getList(qBean, QDO.parentMenuCode.eq(parentMenuCode).and(QDO.nodeType.eq(MenuTreeNodeType.MENU.getValue())));
    }

    public List<String> getBoundMenuCodes(@NotBlank String parentMenuCode) {
        return super.getValueList(QDO.menuCode, QDO.parentMenuCode.eq(parentMenuCode).and(QDO.nodeType.eq(MenuTreeNodeType.MENU.getValue())));
    }

    /**
     * 根据菜单名称模糊查询菜单编码
     *
     * @param menuNameLike
     * @return
     */
    public List<String> queryMenuCodeByMenuNameLike(@NotBlank String menuNameLike) {
        return jpaQueryFactory.select(QDO.menuCode)
                .from(QDO)
                .where((QDO.menuName.like("%" + menuNameLike + "%")))
                .fetch();
    }

    /**
     * 根据菜单编码获取菜单名称
     *
     * @param menuCodes
     * @return
     */
    public Map<String, String> getMenuNameByMenuCode(@NotEmpty Collection<String> menuCodes) {
        return jpaQueryFactory.select(QDO.menuName, QDO.menuCode)
                .from(QDO)
                .where(QDO.menuCode.in(menuCodes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.menuCode), t -> t.get(QDO.menuName), (t1, t2) -> t1));
    }

    public String getMenuNameByMenuCode(@NotBlank String menuCode) {
        return super.getValueByValue(QDO.menuName, QDO.menuCode, menuCode);
    }

    public String getNodeType(long id) {
        return super.getValue(QDO.nodeType, id);
    }

    public String getMenuCode(long id) {
        return super.getValue(QDO.menuCode, id);
    }

    /**
     * 根据租户查询菜单
     *
     * @return
     */
    public List<SysTenantMenuTreeDO> queryByTenantId() {
        return jpaQueryFactory.select(QDO)
                .from(QDO)
                .fetch();
    }

    /**
     * 根据菜单编码获取
     *
     * @param menuCode
     * @return
     */
    public SysTenantMenuTreeDO getByMenuCode(@NotBlank String menuCode) {
        return super.getOneByValue(QDO.menuCode, menuCode);
    }

    public List<TenantMenuBO> queryMenuBos() {
        var qBean = qBeanTenantMenu();
        var predicate = PredicateBuilder.builder()
                .build();
        return super.getList(qBean, predicate).stream()
                .filter(t -> t.getNodeType().equals(MenuTreeNodeType.MENU_GROUP.getValue()) || t.getNodeType().equals(MenuTreeNodeType.MENU.getValue()))
                .collect(Collectors.toList());
    }

    public List<String> allMenuCodes() {
        return super.getValueList(QDO.menuCode);
    }

    private QBean<TenantMenuBO> qBeanTenantMenu() {
        return Projections.bean(TenantMenuBO.class, QDO.id, QDO.menuCode, QDO.menuName, QDO.nodeType,
                QDO.sortNo, QDO.enabled, QDO.parentMenuCode,
                QDO.icon);
    }
}
