package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.common.param.IdCodeNameCheckParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.service.common.constant.BelongType;
import com.elitescloud.cloudt.system.service.model.entity.QSysEmployeeDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysRoleDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserRoleDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserRoleDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/16
 */
@Repository
public class UserRoleRepoProc extends BaseRepoProc<SysUserRoleDO> {
    private static final QSysUserRoleDO QDO = QSysUserRoleDO.sysUserRoleDO;
    private static final QSysRoleDO QDO_ROLE = QSysRoleDO.sysRoleDO;
    private static final QSysEmployeeDO QDO_EMP = QSysEmployeeDO.sysEmployeeDO;

    public UserRoleRepoProc() {
        super(QDO);
    }

    /**
     * 根据用户删除角色
     *
     * @param userId
     */
    public void deleteByUser(@NotNull Long userId) {
        super.deleteByValue(QDO.userId, userId);
    }

    /**
     * 删除角色下的账户
     *
     * @param roleId
     * @param employeeOnly
     */
    public void deleteByRole(long roleId, boolean employeeOnly) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.roleId, roleId)
                .and(employeeOnly, () -> JPAExpressions.select(QDO_EMP.id).from(QDO_EMP).where(QDO_EMP.userId.eq(QDO.userId)).exists())
                .build();
        super.delete(predicate);
    }

    /**
     * 删除角色下的账号
     *
     * @param roleId
     * @param userIds
     */
    public void deleteRoleUser(long roleId, Collection<Long> userIds) {
        super.delete(QDO.roleId.eq(roleId).and(QDO.userId.in(userIds)));
    }

    public void deleteRolesOfUser(long userId, BelongType.Belonger belonger) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.userId, userId)
                .and(belonger != null, () -> QDO_ROLE.typeId.eq(belonger.getBelongId()).and(QDO_ROLE.type.eq(belonger.getBelongType().getValue())))
                .build();
        super.delete(predicate);
    }

    /**
     * 获取用户分配的角色
     *
     * @param userId
     * @param belonger
     * @return
     */
    public List<SysUserRoleDO> getByUser(@NotNull Long userId, @NotNull BelongType.Belonger belonger) {
        Predicate predicate = PredicateBuilder.builder()
                .andEq(true, QDO.userId, userId)
                .andNotNull(true, QDO.userId)
                .and(belonger != null, () -> QDO_ROLE.type.eq(belonger.getBelongType().getValue()).and(QDO_ROLE.typeId.eq(belonger.getBelongId())))
                .andEq(true, QDO_ROLE.enabled, true)
                .build();

        return jpaQueryFactory.select(QDO)
                .from(QDO_ROLE)
                .leftJoin(QDO).on(QDO.roleId.eq(QDO_ROLE.id))
                .where(predicate)
                .fetch();
    }

    /**
     * 获取用户的角色
     *
     * @param userId
     * @param belonger
     * @param onlyOwn
     * @return
     */
    public List<IdCodeNameCheckParam> getByUser(@NotNull Long userId, @NotNull BelongType.Belonger belonger, boolean onlyOwn) {
        Predicate predicate = PredicateBuilder.builder()
                .andNotNull(onlyOwn, QDO.userId)
                .and(belonger != null, () -> QDO_ROLE.type.eq(belonger.getBelongType().getValue()).and(QDO_ROLE.typeId.eq(belonger.getBelongId())))
                .andEq(true, QDO_ROLE.enabled, true)
                .build();

        return jpaQueryFactory.select(QDO_ROLE.id, QDO_ROLE.code, QDO_ROLE.name, QDO.userId)
                .from(QDO_ROLE)
                .leftJoin(QDO).on(QDO.roleId.eq(QDO_ROLE.id).and(QDO.userId.eq(userId)))
                .where(predicate)
                .fetch()
                .stream()
                .map(t -> {
                    IdCodeNameCheckParam param = new IdCodeNameCheckParam();
                    param.setId(t.get(QDO_ROLE.id));
                    param.setCode(t.get(QDO_ROLE.code));
                    param.setName(t.get(QDO_ROLE.name));
                    param.setChecked(t.get(QDO.userId) != null);

                    return param;
                }).collect(Collectors.toList());
    }

    /**
     * 获取用户拥有的全部的角色编码
     *
     * @param userId
     * @return
     */
    public List<IdCodeNameParam> getRoleOfUser(long userId, Long tenantId, Long tenantOrgId) {
        var roleIdQuery = JPAExpressions.select(QDO.roleId)
                .from(QDO)
                .where(QDO.userId.eq(userId));
        var predicateType = PredicateBuilder.builder()
                .and(true, () -> QDO_ROLE.typeId.eq("").and(QDO_ROLE.type.eq(BelongType.CUSTOM.getValue())))
                .and(tenantId != null, () -> QDO_ROLE.typeId.eq(tenantId.toString()).and(QDO_ROLE.type.eq(BelongType.TENANT.getValue())))
                .and(tenantOrgId != null, () -> QDO_ROLE.typeId.eq(tenantOrgId.toString()).and(QDO_ROLE.type.eq(BelongType.TENANT_ORG.getValue())))
                .buildOr();

        QBean<IdCodeNameParam> qBean = Projections.bean(IdCodeNameParam.class, QDO_ROLE.id, QDO_ROLE.code, QDO_ROLE.name);
        var roleList = jpaQueryFactory.select(qBean)
                .from(QDO_ROLE)
                .where(QDO_ROLE.id.in(roleIdQuery).and(QDO_ROLE.enabled.eq(true)).and(predicateType))
                .fetch();
        if (roleList.isEmpty()) {
            return roleList;
        }

        // 去重并返回
        List<IdCodeNameParam> resultList = new ArrayList<>(roleList.size());
        Set<Long> existsCodes = new HashSet<>(4);
        for (IdCodeNameParam idCodeNameParam : roleList) {
            if (existsCodes.contains(idCodeNameParam.getId())) {
                continue;
            }
            existsCodes.add(idCodeNameParam.getId());
            resultList.add(idCodeNameParam);
        }
        return resultList;
    }

    /**
     * 获取用户拥有的角色
     *
     * @param userIds
     * @return
     */
    public Map<Long, List<IdCodeNameParam>> getRoleOfUser(@NotEmpty Collection<Long> userIds, Long tenantId, Long tenantOrgId) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.userId, userIds)
                .andEq(QDO_ROLE.enabled, true)
                .and(true, () -> QDO_ROLE.typeId.eq("").and(QDO_ROLE.type.eq(BelongType.CUSTOM.getValue())))
                .and(tenantId != null, () -> QDO_ROLE.typeId.eq(tenantId.toString()).and(QDO_ROLE.type.eq(BelongType.TENANT.getValue())))
                .and(tenantOrgId != null, () -> QDO_ROLE.typeId.eq(tenantOrgId.toString()).and(QDO_ROLE.type.eq(BelongType.TENANT_ORG.getValue())))
                .buildOr();

        return jpaQueryFactory.select(QDO.userId, QDO_ROLE.id, QDO_ROLE.code, QDO_ROLE.name)
                .from(QDO)
                .leftJoin(QDO_ROLE).on(QDO.roleId.eq(QDO_ROLE.id))
                .where(predicate)
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.userId),
                        Collectors.collectingAndThen(Collectors.toList(), t -> t.stream().map(tt ->
                                new IdCodeNameParam(tt.get(QDO_ROLE.id), tt.get(QDO_ROLE.code), tt.get(QDO_ROLE.name))).collect(Collectors.toList()))));
    }

    public boolean existsRole(long roleId) {
        return super.exists(QDO.roleId, roleId);
    }

    /**
     * 根据角色编码查询用户ID
     *
     * @param roleCode 角色编码
     * @return 用户ID
     */
    public Set<Long> getUserIdByRole(@NotBlank String roleCode) {
        var roleIdQuery = JPAExpressions.select(QDO_ROLE.id).from(QDO_ROLE).where(QDO_ROLE.code.eq(roleCode));
        return new HashSet<>(jpaQueryFactory.select(QDO.userId)
                .from(QDO)
                .where(QDO.roleId.eq(roleIdQuery))
                .fetch());
    }

    /**
     * 根据角色ID查询用户ID
     *
     * @param roleId 角色ID
     * @return 用户ID
     */
    public Set<Long> getUserIdsByRole(long roleId) {
        return new HashSet<>(jpaQueryFactory.select(QDO.userId)
                .from(QDO)
                .where(QDO.roleId.eq(roleId))
                .fetch());
    }

    public Set<Long> getUserIdsByRole(long roleId, boolean employeeOnly) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.roleId, roleId)
                .and(employeeOnly, () -> JPAExpressions.select(QDO_EMP.id).from(QDO_EMP).where(QDO_EMP.userId.eq(QDO.userId)).exists())
                .build();
        return new HashSet<>(jpaQueryFactory.select(QDO.userId)
                .from(QDO)
                .where(predicate)
                .fetch());
    }

    /**
     * 根据角色ID查询用户ID
     *
     * @param roleIds 角色ID
     * @return 用户ID
     */
    public Set<Long> getUserIdsByRole(Collection<Long> roleIds) {
        return new HashSet<>(jpaQueryFactory.select(QDO.userId)
                .from(QDO)
                .where(QDO.roleId.in(roleIds))
                .fetch());
    }

    /**
     * 根据角色ID查询用户ID
     *
     * @param roleIds 角色ID
     * @return 用户ID
     */
    public Map<Long, Set<Long>> getUserIdByRole(Collection<Long> roleIds) {
        return jpaQueryFactory.select(QDO.userId, QDO.roleId)
                .from(QDO)
                .where(QDO.roleId.in(roleIds))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.roleId),
                        Collectors.mapping(t -> t.get(QDO.userId), Collectors.toSet())));
    }

    public Map<Long, Set<String>> getRoleCodesOfUser(@NotEmpty Collection<Long> userIds, BelongType.Belonger belonger) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.userId, userIds)
                .and(belonger != null, () -> QDO_ROLE.typeId.eq(belonger.getBelongId()).and(QDO_ROLE.type.eq(belonger.getBelongType().getValue())))
                .build();
        return jpaQueryFactory.select(QDO.userId, QDO_ROLE.code)
                .from(QDO)
                .leftJoin(QDO_ROLE).on(QDO.roleId.eq(QDO_ROLE.id))
                .where(predicate)
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.userId), Collectors.mapping(t -> t.get(QDO_ROLE.code), Collectors.toSet())));
    }

    public List<IdCodeNameParam> getRolesOfUser(long userId, BelongType.Belonger belonger) {
        QBean<IdCodeNameParam> qBean = Projections.bean(IdCodeNameParam.class, QDO_ROLE.id, QDO_ROLE.code, QDO_ROLE.name);
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.userId, userId)
                .and(belonger != null, () -> QDO_ROLE.typeId.eq(belonger.getBelongId()).and(QDO_ROLE.type.eq(belonger.getBelongType().getValue())))
                .build();
        return jpaQueryFactory.select(qBean)
                .from(QDO)
                .leftJoin(QDO_ROLE).on(QDO.roleId.eq(QDO_ROLE.id))
                .where(predicate)
                .fetch();
    }
}
