package com.elitescloud.cloudt.system.service.repo.old;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.AccountType;
import com.elitescloud.cloudt.context.spi.ServiceProviderLoader;
import com.elitescloud.cloudt.system.component.param.SysUserComponentPagingParam;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.param.SysUserBatchSwitchParam;
import com.elitescloud.cloudt.system.param.SysUserQueryParam;
import com.elitescloud.cloudt.system.provider.dto.SysUserRpcDTO;
import com.elitescloud.cloudt.system.provider.param.SysUserRpcDtoParam;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserRoleDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.service.vo.component.SysUserComponentPagingVO;
import com.elitescloud.cloudt.system.spi.SysUserQuerySpi;
import com.elitescloud.cloudt.system.vo.SysUserDetailsVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Repository;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;


/**
 * <pre>
 * [功能说明]
 * </pre>
 *
 * @author Michael Li
 * 2020/7/8
 */
@Repository
@Log4j2
public class SysUserRepoProc extends BaseRepoProc<SysUserDO> {

    private static final QSysUserDO SYS_USER = QSysUserDO.sysUserDO;
    private final QSysUserRoleDO sysUserRoleDO = QSysUserRoleDO.sysUserRoleDO;

    @Autowired
    private SystemProperties systemProperties;
    private final AtomicBoolean spiLoaded = new AtomicBoolean(false);
    private SysUserQuerySpi userQuerySpi;

    public SysUserRepoProc() {
        super(SYS_USER);
    }

    public PagingVO<SysUserDO> selectSysUser(SysUserQueryParam param) {
        var spi = getUserQuerySpi();

        JPAQuery<SysUserDO> query = null;
        if (spi != null) {
            query = spi.selectSysUserForJpaQuery();
        }
        if (query == null) {
            query = jpaQueryFactory.select(SYS_USER).from(SYS_USER);
        }

        // 查询条件
        var predicate = this.where(param);
        if (spi != null) {
            predicate = andPredicate(predicate, spi.selectSysUserForPredicate());
        }

        query.where(predicate);

        //数据权限开关
        //   GeneralUserDetails user = SecurityContextUtil.currentUser();
//        if (user == null) {
//            throw new BusinessException(ApiCode.FORBIDDEN, "数据权限异常，USER 为空");
//        }
        //   DataAuthJpaUtil.dataAuthJpaFilter(query, QSysUserDO.sysUserDO.getMetadata());

        param.fillOrders(query, SYS_USER);
        param.setPaging(query);
        return PagingVO.<SysUserDO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();

    }

    /**
     * 查询用户分页接口-放大镜组件
     *
     * @param param 分页参数
     * @return 分页结果
     */
    public PagingVO<SysUserComponentPagingVO> findSysUserComPagingByParam(SysUserComponentPagingParam param) {
        var spi = getUserQuerySpi();

        JPAQuery<SysUserComponentPagingVO> query = null;
        if (spi != null) {
            query = spi.findSysUserComPagingByParamForJpaQuery();
        }
        if (query == null) {
            query = jpaQueryFactory
                    .select(Projections.bean(
                            SysUserComponentPagingVO.class,
                            SYS_USER.id,
                            SYS_USER.username,
                            SYS_USER.lastName,
                            SYS_USER.firstName,
                            SYS_USER.mobile,
                            SYS_USER.email
                    ))
                    .from(SYS_USER);
        }

        // 组装 where
        List<Predicate> predicates = new ArrayList<>();
        if (StringUtils.isNotBlank(param.getUsername())) {
            var likeStr = "%" + param.getUsername() + "%";
            predicates.add(SYS_USER.username.like(likeStr));
        }
        if (StringUtils.isNotBlank(param.getFirstName())) {
            var likeStr = "%" + param.getFirstName() + "%";
            predicates.add(SYS_USER.firstName.like(likeStr));
        }
        if (StringUtils.isNotBlank(param.getLastName())) {
            var likeStr = "%" + param.getLastName() + "%";
            predicates.add(SYS_USER.lastName.like(likeStr));
        }
        if (StringUtils.isNotBlank(param.getMobile())) {
            var likeStr = "%" + param.getMobile() + "%";
            predicates.add(SYS_USER.mobile.like(likeStr));
        }
        if (StringUtils.isNotBlank(param.getEmail())) {
            var likeStr = "%" + param.getEmail() + "%";
            predicates.add(SYS_USER.email.like(likeStr));
        }
        predicates.add(SYS_USER.enabled.eq(true));
        predicates.add(SYS_USER.deleteFlag.isNull().or(SYS_USER.deleteFlag.eq(0)));

        if (spi != null) {
            Predicate predicate = spi.findSysUserComPagingByParamForPredicate();
            if (predicate != null) {
                predicates.add(predicate);
            }
        }

        query.where(ExpressionUtils.allOf(predicates));

        // 设置 分页
        param.fillOrders(query, SYS_USER);
        param.setPaging(query);

        return PagingVO.<SysUserComponentPagingVO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }

    public Predicate where(SysUserQueryParam param) {

        Predicate predicate = SYS_USER.isNotNull().or(SYS_USER.isNull());

        predicate = StringUtils.isBlank(param.getKeyword()) ? predicate :
                ExpressionUtils.and(predicate, SYS_USER.username.like("%" + param.getKeyword() + "%")
                        .or(SYS_USER.firstName.like("%" + param.getKeyword() + "%"))
                        .or(SYS_USER.lastName.like("%" + param.getKeyword() + "%"))
                        .or(SYS_USER.mobile.like("%" + param.getKeyword() + "%"))
                        .or(SYS_USER.email.like("%" + param.getKeyword() + "%")));

        predicate = StringUtils.isBlank(param.getUsername()) ? predicate : ExpressionUtils.and(predicate, SYS_USER.username.like("%" + param.getUsername() + "%"));
        predicate = StringUtils.isBlank(param.getName()) ? predicate : ExpressionUtils.and(predicate, SYS_USER.firstName.like("%" + param.getName() + "%"));
        predicate = StringUtils.isBlank(param.getMobile()) ? predicate : ExpressionUtils.and(predicate, SYS_USER.mobile.like("%" + param.getMobile() + "%"));
        predicate = StringUtils.isBlank(param.getEmail()) ? predicate : ExpressionUtils.and(predicate, SYS_USER.email.like("%" + param.getEmail() + "%"));
        predicate = ObjectUtils.isEmpty(param.getEnabled()) ? predicate : ExpressionUtils.and(predicate, SYS_USER.enabled.eq(param.getEnabled()));

        predicate = ObjectUtils.isEmpty(param.getSourceType()) ? predicate : ExpressionUtils.and(predicate, SYS_USER.sourceType.eq(param.getSourceType()));

        return predicate;
    }

    /**
     * 批量更新用户状态
     *
     * @param param 参数
     */
    public void updateStatusByIds(SysUserBatchSwitchParam param) {
        jpaQueryFactory
                .update(SYS_USER)
                .set(SYS_USER.enabled, param.isEnable())
                .where(SYS_USER.id.in(param.getUserIds()))
                .execute();
    }

    /**
     * 根据ID获取用户详情
     *
     * @param id 用户ID
     * @return 详情VO
     */
    public SysUserDetailsVO findDetailsById(Long id) {
        return jpaQueryFactory
                .select(Projections.bean(
                        SysUserDetailsVO.class,
                        SYS_USER.id,
                        SYS_USER.lastName,
                        SYS_USER.firstName,
                        SYS_USER.mobile,
                        SYS_USER.email,
                        SYS_USER.enabled,
                        SYS_USER.avatarUrl,
                        SYS_USER.avatarCode
                ))
                .from(SYS_USER)
                .where(SYS_USER.id.eq(id))
                .fetchOne();
    }

    /**
     * 根据登录号获取用户ID
     *
     * @param username 登录号
     * @return 用户ID
     */
    public Long getIdByUsername(String username) {
        return jpaQueryFactory.select(SYS_USER.id)
                .from(SYS_USER)
                .where(SYS_USER.username.eq(username))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据账号获取用户列表
     *
     * @param account
     * @return
     */
    public List<SysUserDO> queryByAccount(@NonNull String account) {
        if (systemProperties.getAccountTypes().isEmpty()) {
            log.error("未配置账号类型");
            return Collections.emptyList();
        }

        List<AccountType> accountTypes = systemProperties.getAccountTypes();
        var predicate = PredicateBuilder.builder()
                .andEq(accountTypes.contains(AccountType.USERNAME), SYS_USER.username, account)
                .andEq(accountTypes.contains(AccountType.MOBILE), SYS_USER.mobile, account)
                .andEq(accountTypes.contains(AccountType.EMAIL), SYS_USER.email, account)
                .buildOr();
        return jpaQueryFactory.select(SYS_USER)
                .from(SYS_USER)
                .where(predicate)
                .fetch()
                ;
    }

    /**
     * 根据邮箱获取用户列表
     *
     * @param email
     * @return
     */
    public List<SysUserDO> queryByEmail(@NonNull String email) {
        return jpaQueryFactory.select(SYS_USER)
                .from(SYS_USER)
                .where(SYS_USER.email.eq(email))
                .fetch()
                ;
    }

    /**
     * 查询用户 dto
     *
     * @param param 查询参数
     * @return dto
     */
    public List<SysUserRpcDTO> findRpcDtoByParam(SysUserRpcDtoParam param) {
        var query = jpaQueryFactory
                .select(Projections.bean(
                        SysUserRpcDTO.class,
                        SYS_USER.id,
                        SYS_USER.username,
                        SYS_USER.lastName,
                        SYS_USER.firstName,
                        SYS_USER.mobile,
                        SYS_USER.email,
                        SYS_USER.sourceType,
                        SYS_USER.enabled,
                        SYS_USER.needReset,
                        SYS_USER.avatarUrl,
                        SYS_USER.avatarCode
                ))
                .from(SYS_USER);

        List<Predicate> predicates = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(param.getUserIds())) {
            var userIds = filterList(param.getUserIds());
            if (CollectionUtils.isNotEmpty(userIds)) {
                predicates.add(SYS_USER.id.in(userIds));
            }
        }
        if (CollectionUtils.isNotEmpty(param.getUsernames())) {
            var usernames = filterList(param.getUsernames());
            if (CollectionUtils.isNotEmpty(usernames)) {
                predicates.add(SYS_USER.username.in(usernames));
            }
        }
        if (CollectionUtils.isNotEmpty(param.getEmails())) {
            var emails = filterList(param.getEmails());
            if (CollectionUtils.isNotEmpty(emails)) {
                predicates.add(SYS_USER.email.in(emails));
            }
        }
        if (CollectionUtils.isNotEmpty(param.getMobiles())) {
            var mobiles = filterList(param.getMobiles());
            if (CollectionUtils.isNotEmpty(mobiles)) {
                predicates.add(SYS_USER.mobile.in(mobiles));
            }
        }
        if (CollectionUtils.isNotEmpty(param.getSourceTypes())) {
            var sourceTypes = filterList(param.getSourceTypes());
            if (CollectionUtils.isNotEmpty(sourceTypes)) {
                predicates.add(SYS_USER.sourceType.in(sourceTypes));
            }
        }
        if (param.getEnabled() != null) {
            predicates.add(SYS_USER.enabled.eq(param.getEnabled()));
        }

        if (CollectionUtils.isEmpty(predicates)) {
            return Collections.emptyList();
        } else {
            query.where(ExpressionUtils.allOf(predicates));
        }

        return query.fetch();
    }

    /**
     * 获取指定角色下的用户id
     *
     * @param roleIds 应用角色角色id 列表
     * @return 用户id 列表
     */
    public List<Long> findUserIdsByRoleIds(List<Long> roleIds) {
        return jpaQueryFactory.
                select(sysUserRoleDO.userId)
                .from(sysUserRoleDO)
                .where(sysUserRoleDO.roleId.in(roleIds))
                .fetch();
    }

    private <T> List<T> filterList(List<T> list) {
        return list.stream().distinct().filter(Objects::nonNull).collect(Collectors.toList());
    }

    private SysUserQuerySpi getUserQuerySpi() {
        if (spiLoaded.compareAndSet(false, true)) {
            userQuerySpi = ServiceProviderLoader.loadProviderInstanceOne(SysUserQuerySpi.class).orElse(null);
            if (userQuerySpi != null) {
                log.info("加载用户查询SPI：{}", userQuerySpi.getServiceName());
            }
        }
        return userQuerySpi;
    }
}
