package com.elitescloud.cloudt.ucenter.repo;

import cn.hutool.core.util.ObjUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.ucenter.api.dto.ArticleInfoDTO;
import com.elitescloud.cloudt.ucenter.api.vo.param.ArticleInfoPagingParam;
import com.elitescloud.cloudt.ucenter.api.vo.resp.ArticleInfoRespVO;
import com.elitescloud.cloudt.ucenter.entity.ArticleInfoDO;
import com.elitescloud.cloudt.ucenter.entity.QArticleCategoryDO;
import com.elitescloud.cloudt.ucenter.entity.QArticleInfoDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.StringExpression;
import com.querydsl.jpa.impl.JPAQuery;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

@Repository
public class ArticleInfoRepoProc extends BaseRepoProc<ArticleInfoDO> {

    private final static QArticleCategoryDO Q_ARTICLE_CATEGORY_DO = QArticleCategoryDO.articleCategoryDO;

    private final static QArticleInfoDO Q_DO = QArticleInfoDO.articleInfoDO;

    protected ArticleInfoRepoProc() {
        super(Q_DO);
    }


    /**
     * 分页查询
     *
     * @param queryVO 条件
     * @return 分页结果
     */
    public PagingVO<ArticleInfoDTO> findArticleInfoPagingResults(ArticleInfoPagingParam queryVO) {
        JPAQuery<ArticleInfoDTO> jpaQuery = select(ArticleInfoDTO.class)
                .from(Q_DO)
                .leftJoin(Q_ARTICLE_CATEGORY_DO).on(Q_DO.categoryId.eq(Q_ARTICLE_CATEGORY_DO.id))
                .where(where(queryVO))
                .orderBy(Q_DO.showFlag.asc())
                .orderBy(Q_DO.sortNo.desc())
                .orderBy(Q_DO.createTime.desc());;
        queryVO.setPaging(jpaQuery);
        return PagingVO.<ArticleInfoDTO>builder()
                .total(jpaQuery.fetchCount())
                .records(jpaQuery.fetch())
                .build();
    }

    private Predicate where(ArticleInfoPagingParam queryVO) {
        return PredicateBuilder.builder()
                .andLike(StrUtil.isNotEmpty(queryVO.getArticleAuthor()), Q_DO.articleAuthor, queryVO.getArticleAuthor())
                .andLike(StrUtil.isNotEmpty(queryVO.getShowFlag()), Q_DO.showFlag, queryVO.getShowFlag())
                .andLike(StrUtil.isNotEmpty(queryVO.getCodeName()), new StringExpression[]{Q_DO.articleCode, Q_DO.articleName}, queryVO.getCodeName())
                .andEq(ObjUtil.isNotNull(queryVO.getCategoryId()), Q_DO.categoryId, queryVO.getCategoryId())
                .build();
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                Q_DO.id,
                Q_DO.articleAuthor,
                Q_DO.articleName,
                Q_DO.articleCode,
                Q_ARTICLE_CATEGORY_DO.categoryName,
                Q_DO.sortNo,
                Q_DO.showFlag,
                Q_DO.categoryId,
                Q_DO.articleDetail,
                Q_ARTICLE_CATEGORY_DO.pid
        )).from(Q_DO);
    }

    public ArticleInfoDTO findById(Long id) {
        return select(ArticleInfoDTO.class)
                .from(Q_DO)
                .leftJoin(Q_ARTICLE_CATEGORY_DO).on(Q_DO.categoryId.eq(Q_ARTICLE_CATEGORY_DO.id))
                .where(Q_DO.id.eq(id)).fetchOne();
    }

    public List<ArticleInfoRespVO> findByCategoryIdInAndShowFlag(Set<Long> keySet, String showFlag) {
        return jpaQueryFactory.select(Projections.bean(ArticleInfoRespVO.class,
                Q_DO.id,
                Q_DO.articleAuthor,
                Q_DO.articleName,
                Q_DO.articleCode,
                Q_DO.sortNo,
                Q_DO.showFlag,
                Q_DO.categoryId,
                Q_DO.articleDetail
        ))
                .from(Q_DO)
                .where(Q_DO.categoryId.in(keySet).and(Q_DO.showFlag.eq(showFlag)))
                .orderBy(Q_DO.sortNo.desc())
                .fetch();
    }
}
