package com.elitescloud.cloudt.system.config;

import com.elitescloud.cloudt.system.common.InfinityWebProperties;
import com.elitescloud.cloudt.system.common.OAuth2Client;
import com.elitescloud.cloudt.system.service.impl.InfinityRestClient;
import com.elitescloud.cloudt.system.util.RedisUtil;
import com.elitescloud.cloudt.system.util.RestWrapper;
import io.netty.channel.ChannelOption;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.client.reactive.ReactorClientHttpConnector;
import org.springframework.web.reactive.function.client.ExchangeStrategies;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.netty.http.client.HttpClient;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import io.netty.handler.ssl.SslContext;

import javax.net.ssl.SSLException;

/**
 * @Auther: Mark
 * @Date: 2024/12/24 10:59
 * @Description:
 */
@Slf4j
@Configuration
@EnableConfigurationProperties(InfinityWebProperties.class)
@ConditionalOnProperty(prefix = InfinityWebProperties.INFINITY_WEB_CONFIG_PREFIX, name = "enabled", havingValue =
        "true")
@RequiredArgsConstructor
public class InfinityWebAutoConfiguration {

    @Bean
    @ConditionalOnClass
    public OAuth2Client oAuth2Client(RestWrapper restWrapper, RedisUtil redisUtil) {
        return new OAuth2Client(restWrapper, redisUtil);
    }

    @Bean
    @ConditionalOnMissingBean
    public WebClient webClient(InfinityWebProperties properties) {
        HttpClient httpClient = HttpClient.create()
                // 连接超时
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, InfinityWebProperties.CONNECT_TIMEOUT)
                // 响应超时
                .responseTimeout(Duration.ofMillis(InfinityWebProperties.RESPONSE_TIMEOUT))
                .doOnConnected(conn -> conn
                        // 读写超时
                        .addHandlerLast(new ReadTimeoutHandler(InfinityWebProperties.READ_TIMEOUT,
                                TimeUnit.MILLISECONDS))
                        .addHandlerLast(new WriteTimeoutHandler(InfinityWebProperties.WRITE_TIMEOUT,
                                TimeUnit.MILLISECONDS))
                );
        // 尝试忽略证书验证（仅用于测试环境）【生产环境中是不安全的】
        if(properties.isIgnoreSslSecure()) {
            try {
                SslContext sslContext = SslContextBuilder.forClient()
                        .trustManager(InsecureTrustManagerFactory.INSTANCE)
                        .build();
                httpClient.secure(ssl -> ssl.sslContext(sslContext));
            } catch (SSLException e) {
                log.error("忽略证书验证失败：", e);
            }
        }
        return WebClient.builder()
                .clientConnector(new ReactorClientHttpConnector(httpClient))
                .exchangeStrategies(ExchangeStrategies.builder()
                        .codecs(clientCodecConfigurer -> clientCodecConfigurer
                                .defaultCodecs()
                                // 设置缓冲区大小，单位，字节(B) => MB 转 B
                                .maxInMemorySize(100 * 1024 * 1024))
                        .build())
                .build();
    }

    @Bean
    public InfinityRestClient infinityRestClient(WebClient webClient) {
        return new InfinityRestClient(webClient);
    }

}
